// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model.jam;

import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceProvider;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReferenceSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * FQN class reference in String literal.
 */
public class StringLiteralPsiClassConverter extends JamConverter<PsiClass> {

  private static final JavaClassReferenceProvider JAVA_CLASS_REFERENCE_PROVIDER = new JavaClassReferenceProvider();

  static {
    JAVA_CLASS_REFERENCE_PROVIDER.setOption(JavaClassReferenceProvider.ALLOW_DOLLAR_NAMES, Boolean.TRUE);
  }

  @Nullable
  @Override
  public PsiClass fromString(@Nullable String s, JamStringAttributeElement<PsiClass> context) {
    if (StringUtil.isEmptyOrSpaces(s)) return null;

    final PsiAnnotationMemberValue psiElement = context.getPsiElement();
    if (psiElement == null) return null;

    return JavaPsiFacade.getInstance(psiElement.getProject()).findClass(s, psiElement.getResolveScope());
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(JamStringAttributeElement<PsiClass> context) {
    final PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    final String stringValue = context.getStringValue();
    if (stringValue == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    final JavaClassReferenceSet set =
      new JavaClassReferenceSet(stringValue, psiLiteral,
                                ElementManipulators.getOffsetInElement(psiLiteral),
                                false, JAVA_CLASS_REFERENCE_PROVIDER) {
        @Override
        public boolean isAllowDollarInNames() {
          return true;
        }
      };
    return set.getAllReferences();
  }
}
