// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model.properties.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.PsiModifierListOwner;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.boot.SpringBootClassesConstants;
import com.intellij.spring.boot.model.jam.ResourceItemsConverter;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.List;

public abstract class ConfigurationProperties implements JamElement {

  private static final JamStringAttributeMeta.Single<String> VALUE_META =
    JamAttributeMeta.singleString("value");
  private static final JamStringAttributeMeta.Single<String> PREFIX_META =
    JamAttributeMeta.singleString("prefix");

  private static final JamBooleanAttributeMeta IGNORE_INVALID_FIELDS_META =
    JamAttributeMeta.singleBoolean("ignoreInvalidFields", false);

  private static final JamBooleanAttributeMeta IGNORE_NESTED_PROPERTIES_META =
    JamAttributeMeta.singleBoolean("ignoreNestedProperties", false);

  private static final JamBooleanAttributeMeta IGNORE_UNKNOWN_FIELDS_META =
    JamAttributeMeta.singleBoolean("ignoreUnknownFields", true);

  private static final JamBooleanAttributeMeta EXCEPTION_IF_INVALID_META =
    JamAttributeMeta.singleBoolean("exceptionIfInvalid", true);

  private static final JamStringAttributeMeta.Collection<Collection<PsiFileSystemItem>> LOCATIONS_META =
    JamAttributeMeta.collectionString("locations", new ResourceItemsConverter());

  private static final JamBooleanAttributeMeta MERGE_META =
    JamAttributeMeta.singleBoolean("merge", true);

  private static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(SpringBootClassesConstants.CONFIGURATION_PROPERTIES)
      .addAttribute(VALUE_META)
      .addAttribute(PREFIX_META)
      .addAttribute(IGNORE_INVALID_FIELDS_META)
      .addAttribute(IGNORE_NESTED_PROPERTIES_META)
      .addAttribute(IGNORE_UNKNOWN_FIELDS_META)
      .addAttribute(EXCEPTION_IF_INVALID_META)
      .addAttribute(LOCATIONS_META)
      .addAttribute(MERGE_META);

  public static final JamClassMeta<ConfigurationProperties> CLASS_META =
    new JamClassMeta<>(ConfigurationProperties.class)
      .addAnnotation(ANNOTATION_META);

  @NotNull
  @JamPsiConnector
  protected abstract PsiModifierListOwner getPsiElement();

  public String getValueOrPrefix() {
    final String value = ANNOTATION_META.getAttribute(getPsiElement(), VALUE_META).getValue();
    if (StringUtil.isNotEmpty(value)) {
      return value;
    }
    return ANNOTATION_META.getAttribute(getPsiElement(), PREFIX_META).getValue();
  }

  public boolean isIgnoreInvalidFields() {
    return ANNOTATION_META.getAttribute(getPsiElement(), IGNORE_INVALID_FIELDS_META).getValue();
  }

  public boolean isIgnoreNestedProperties() {
    return ANNOTATION_META.getAttribute(getPsiElement(), IGNORE_NESTED_PROPERTIES_META).getValue();
  }

  public boolean isIgnoreUnknownFields() {
    return ANNOTATION_META.getAttribute(getPsiElement(), IGNORE_UNKNOWN_FIELDS_META).getValue();
  }

  public boolean isExceptionIfInvalid() {
    return ANNOTATION_META.getAttribute(getPsiElement(), EXCEPTION_IF_INVALID_META).getValue();
  }

  public Collection<PsiFileSystemItem> getLocations() {
    final List<JamStringAttributeElement<Collection<PsiFileSystemItem>>> attributeElements =
      ANNOTATION_META.getAttribute(getPsiElement(), LOCATIONS_META);

    List<PsiFileSystemItem> result = new SmartList<>();
    for (JamStringAttributeElement<Collection<PsiFileSystemItem>> element : attributeElements) {
      //noinspection ConstantConditions
      ContainerUtil.addAllNotNull(result, element.getValue());
    }
    return result;
  }

  public boolean isMerge() {
    return ANNOTATION_META.getAttribute(getPsiElement(), MERGE_META).getValue();
  }


  public static abstract class Method extends ConfigurationProperties {

    public static final JamMethodMeta<ConfigurationProperties.Method> METHOD_META =
      new JamMethodMeta<>(ConfigurationProperties.Method.class)
        .addAnnotation(ANNOTATION_META);

    public boolean isDefinitionFor(@NotNull PsiClass psiClass) {
      final PsiMethod psiMethod = (PsiMethod)getPsiElement();
      final PsiClass returnPsiClass = PsiTypesUtil.getPsiClass(psiMethod.getReturnType());
      return psiClass.isEquivalentTo(returnPsiClass) &&
             JamService.getJamService(psiMethod.getProject()).getJamElement(ContextJavaBean.BEAN_JAM_KEY, psiMethod) != null;
    }

  }
}
