// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle.beans;

import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.CommonSpringModel;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Set;

/**
 * @author konstantin.aleev
 */
public interface LiveBean {
  @NotNull
  String getId();

  @Nullable
  String getScope();

  @Nullable
  String getType();

  @Nullable
  LiveResource getResource();

  @NotNull
  Set<LiveBean> getDependencies();

  @NotNull
  Set<LiveBean> getInjectedInto();

  boolean isInnerBean();

  /**
   * @return live bean name
   */
  @NotNull
  String getName();

  /**
   * @return bean class qualified name including inner class name and anonymous class index separated by {@code '$'}.
   */
  @Nullable
  String getClassName();

  /**
   * Returns live bean class.
   *
   * @param project     the project used to search live bean class.
   * @param searchScope the scope in which live bean class is searched.
   * @return live bean class or {@code null} if class could not be found within the specified scope.
   */
  @Nullable
  PsiClass getBeanClass(@NotNull Project project, @NotNull GlobalSearchScope searchScope);

  /**
   * Returns pointer to live bean definition searched within {@code springModel}.
   * <p>
   * If pointer could not be found within {@code springModel},
   * then instance of {@link LiveResourcePointer} referring to {@code resourceElement.getContainingFile()} is returned.
   * <p>
   * If live bean resource is empty or {@code "null"} then pointer referring to {@code beanClass} is returned.
   * <p>
   * Pointer may refer to {@code null}.
   *
   * @param beanClass       live bean class.
   * @param resourceElement PSI element corresponding to this live bean's resource.
   * @param springModel     spring model in which bean pointer is searched.
   * @return pointer to live bean definition.
   */
  @NotNull
  PsiElementPointer getBeanPointer(@Nullable PsiClass beanClass,
                                   @Nullable PsiElement resourceElement,
                                   @Nullable CommonSpringModel springModel);

  /**
   * @return icon in accordance with live bean's resource.
   */
  @NotNull
  Icon getIcon();

  /**
   * Indicates that pointer to live bean definition is not found within Spring model,
   * and a pointer to live bean resource is used instead.
   */
  interface LiveResourcePointer extends PsiElementPointer {
  }
}
