// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle.mappings;

import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.search.GlobalSearchScope;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Method;
import java.util.List;

/**
 * @author konstantin.aleev
 */
public interface LiveHandlerMethod {
  /**
   * @return raw string method representation provided by {@link Method#toString()}
   * <br>Example: {@code public java.lang.Object org.springframework.boot.actuate.endpoint.mvc.EndpointMvcAdapter.invoke()}
   */
  @NotNull
  String getRawMethod();

  @NotNull
  String getClassName();

  @NotNull
  String getMethodName();

  @NotNull
  List<String> getParameters();

  /**
   * @return handler method display name
   */
  @NotNull
  String getDisplayName();

  /**
   * Returns handler method containing class.
   *
   * @param project     the project used to search containing class.
   * @param searchScope the scope in which containing class is searched.
   * @return handler method containing class or {@code null} if class could not be found within the specified scope.
   */
  @Nullable
  PsiClass getContainingClass(Project project, GlobalSearchScope searchScope);

  /**
   * Returns handler method.
   *
   * @param project     the project used to search handler method.
   * @param searchScope the scope in which handler method is searched.
   * @return handler method or {@code null} if method could not be found within the specified scope.
   */
  @Nullable
  PsiMethod getMethod(Project project, GlobalSearchScope searchScope);

  /**
   * Checks that PSI method matches handler method by comparing class name, method name, and parameters list.
   * @param psiMethod method to match
   * @return {@code true} if handler method matches the given PSI method, otherwise {@code false}.
   */
  boolean matches(@NotNull PsiMethod psiMethod);
}
