// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.run.lifecycle.mappings;

import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author konstantin.aleev
 */
public interface LiveRequestMapping {
  @NotNull
  String getMapping();

  @NotNull
  String getPath();

  @NotNull
  List<String> getRequestMethods();

  /**
   * Returns a key-value list of request mapping headers.
   * Key may start with {@code '!'} which means that requests with that header would not be mapped.
   * Value may start with {@code '!'} which means that requests with that header and that value would not be mapped.
   *
   * @return a key-value list of request mapping headers.
   */
  @NotNull
  List<Pair<String, String>> getHeaders();

  /**
   * Returns producible media types.
   * Value may start with {@code '!'} which means that requests with that {@code Accept} would not be mapped.
   *
   * @return a producible media types.
   */
  @NotNull
  List<String> getProduces();

  /**
   * Returns a consumable media types.
   * Value may start with {@code '!'} which means that requests with that {@code Content-Type} would not be mapped.
   *
   * @return a consumable media types.
   */
  @NotNull
  List<String> getConsumes();

  /**
   * Returns a key-value list of request mapping parameters.
   * Key may start with {@code '!'} which means that requests with that parameter would not be mapped.
   * Value may start with {@code '!'} which means that requests with that parameter and that value would not be mapped.
   *
   * @return a key-value list of request mapping parameters.
   */
  @NotNull
  List<Pair<String, String>> getParams();

  @Nullable
  String getBean();

  @Nullable
  LiveHandlerMethod getMethod();

  /**
   * @return {@code true} if method handler is specified and request methods contains <b>GET</b> or empty, otherwise {@code false}.
   */
  boolean canNavigate();

  @NotNull
  static String appendMappingPath(@NotNull String applicationUrl, @Nullable String mappingPath) {
    boolean endsWithSlash = StringUtil.endsWith(applicationUrl, "/");
    if (mappingPath == null) {
      mappingPath = endsWithSlash ? "" : "/";
    }
    else {
      if (endsWithSlash) {
        mappingPath = StringUtil.trimStart(mappingPath, "/");
      }
      else {
        if (!StringUtil.startsWith(mappingPath, "/")) {
          mappingPath = "/" + mappingPath;
        }
      }
    }
    return applicationUrl + mappingPath;
  }
}
