/*
 * Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
 */
package com.intellij.spring.boot.run.update;

import com.intellij.execution.Executor;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.spring.boot.run.SpringBootApplicationRunConfigurationBase;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * SpringBootApplicationUpdatePolicy may be set up for Spring Boot run configuration to perform an update of running application.
 * Update may be performed on frame deactivation or by invoking update action.
 * @author Konstantin Aleev
 */
public abstract class SpringBootApplicationUpdatePolicy {
  public static final ExtensionPointName<SpringBootApplicationUpdatePolicy> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.applicationUpdatePolicy");

  @NotNull private final String myId;
  @NotNull private final String myDescription;

  protected SpringBootApplicationUpdatePolicy(@NotNull String id, @NotNull String description) {
    myId = id;
    myDescription = description;
  }

  @NotNull
  public String getId() {
    return myId;
  }

  @NotNull
  public String getDescription() {
    return myDescription;
  }

  public boolean isAvailableOnFrameDeactivation() {
    return true;
  }

  public boolean isAvailableForExecutor(@NotNull Executor executor) {
    return true;
  }

  public boolean isAvailableForConfiguration(@NotNull SpringBootApplicationRunConfigurationBase configuration) {
    return true;
  }

  public abstract void runUpdate(@NotNull Project project, @NotNull Set<Module> modules, boolean onFrameDeactivation);

  @NotNull
  public static List<SpringBootApplicationUpdatePolicy> getAvailablePolicies(boolean onFrameDeactivation) {
    if (onFrameDeactivation) {
      return Arrays.stream(EP_NAME.getExtensions()).filter(policy -> policy.isAvailableOnFrameDeactivation()).collect(Collectors.toList());
    }
    return Arrays.asList(EP_NAME.getExtensions());
  }

  @Nullable
  public static SpringBootApplicationUpdatePolicy findPolicy(@Nullable String id) {
    if (id == null) return null;

    for (SpringBootApplicationUpdatePolicy updatePolicy : EP_NAME.getExtensions()) {
      if (id.equals(updatePolicy.getId())) return updatePolicy;
    }

    return null;
  }
}
