import {serverLogger} from "./logger-impl";
import CompletionsRequest = ts.server.protocol.CompletionsRequest;
import CompletionEntry = ts.CompletionEntry;


/**
 * Default tsserver implementation doesn't return response in most cases ("open", "close", etc.)
 * we want to override the behaviour and send empty-response holder
 */
const doneRequest = {
    responseRequired: true,
    response: "done"
}
export function getSession(ts_impl, logger:ts.server.Logger):ts.server.Session {

    var TypeScriptSession:typeof ts.server.Session = ts_impl.server.Session;
    var TypeScriptProjectService:typeof ts.server.ProjectService = ts_impl.server.ProjectService;
    var TypeScriptCommandNames:typeof ts.server.CommandNames = ts_impl.server.CommandNames;
    TypeScriptCommandNames.IDEChangeFiles = "ideChangeFiles";
    TypeScriptCommandNames.IDECompile = "ideCompile";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDECompletions = "ideCompletions";

    // var 
    class IDESession extends TypeScriptSession {
        private _host:ts.server.ServerHost;

        private _mySeq:number;

        constructor(host:ts.server.ServerHost,
                    byteLength:(buf:string, encoding?:string) => number,
                    hrtime:(start?:number[]) => number[],
                    logger:ts.server.Logger) {
            super(host, byteLength, hrtime, logger);
            this._host = host;
            let handler = this.projectService.eventHandler;
            //reuse handler
            this.projectService = new IDEProjectService(host, logger, handler);
        }

        public send(msg:ts.server.protocol.Message) {
            const json = JSON.stringify(msg);
            this._host.write(json + "\n");
        }

        executeCommand(request:ts.server.protocol.Request):{response?:any; responseRequired?:boolean} {
            if (TypeScriptCommandNames.Open == request.command) {
                //use own implementation
                const openArgs = <ts.server.protocol.OpenRequestArgs>request.arguments;
                this.openClientFileExt(openArgs);
                return doneRequest;

            } else if (TypeScriptCommandNames.IDEChangeFiles == request.command) {
                const updateFilesArgs = <ts.server.protocol.IDEUpdateFilesContentArgs>request.arguments;
                return this.updateFilesExt(updateFilesArgs);
            } else if (TypeScriptCommandNames.IDECompile == request.command) {
                const fileArgs = <ts.server.protocol.FileRequestArgs>request.arguments;
                return this.compileFileExt(fileArgs);
            } else if (TypeScriptCommandNames.Close == request.command) {
                super.executeCommand(request);
                return doneRequest;
            } else if (TypeScriptCommandNames.IDEGetErrors == request.command) {
                let args = request.arguments;

                return this.getDiagnosticsExt(args.files);
            } else if (TypeScriptCommandNames.IDECompletions == request.command) {
                let result = super.executeCommand({
                    command: TypeScriptCommandNames.Completions,
                    arguments: request.arguments,
                    seq: request.seq,
                    type: request.type
                });
                const args = <ts.server.protocol.CompletionsRequestArgs>request.arguments;
                let response:ts.server.protocol.CompletionEntry[] = result.response;

                return {
                    response: this.getIDECompletions(args, response),
                    responseRequired: true
                }
            }

            return super.executeCommand(request);
        }

        updateFilesExt(args:ts.server.protocol.IDEUpdateFilesContentArgs) {
            let updated = false;
            let files = args.files;
            for (let fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    let content = files[fileName];
                    if (content) {
                        this.changeFileExt(fileName, content);
                        updated = true;
                    }
                }
            }

            if (args.filesToReloadContentFromDisk) {
                for (let fileName of args.filesToReloadContentFromDisk) {
                    if (!fileName) {
                        continue;
                    }

                    let file = ts_impl.normalizePath(fileName);
                    this.projectService.closeClientFile(file);


                    serverLogger("Reload file from disk " + file);
                    updated = true;
                }
            }

            if (updated) {
                this.updateProjectStructureExt();
                // this.projectService.updateProjectStructure();
            }

            return doneRequest;
        }

        private updateProjectStructureExt() {
            let mySeq = this.getChangeSeq();
            let matchSeq = (n) => n === mySeq;
            setTimeout(() => {
                if (matchSeq(this.getChangeSeq())) {
                    this.projectService.updateProjectStructure();
                }
            }, 1500);
        }

        private getChangeSeq() {
            let anyThis:any = this;
            let superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }

            return this._mySeq;
        }

        openClientFileExt(openArgs:ts.server.protocol.OpenRequestArgs):{response?:any; responseRequired?:boolean} {
            const fileName = openArgs.file;
            const fileContent = openArgs.fileContent;
            const configFile = openArgs.tsConfig;
            const file = ts_impl.normalizePath(fileName);

            return (<IDEProjectService>this.projectService).openClientFileExt(file, fileContent, configFile);
        }

        changeFileExt(fileName:string, content:string, tsconfig?:string) {
            const file = ts_impl.normalizePath(fileName);
            const project = this.projectService.getProjectForFile(file);
            if (project) {
                const compilerService = project.compilerService;
                let scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    serverLogger("Reload content from text " + file);
                } else {
                    serverLogger("ScriptInfo is null " + file);
                }
            } else {
                serverLogger("Cannot find project for " + file);
                this.openClientFileExt({
                    file: fileName,
                    fileContent: content,
                    tsConfig: tsconfig
                });
            }
        }

        compileFileExt(req:ts.server.protocol.FileRequestArgs) {
            let requestedFile = req.file;
            const project = this.projectService.getProjectForFile(requestedFile);
            if (project != null) {
                project.program.emit(); //emit all files?
                return doneRequest;
            }
        }

        logError(err:Error, cmd:string) {
            const typedErr = <any>err;
            serverLogger("Error processing message: " + err.message + " " + typedErr.stack);

            super.logError(err, cmd);
        }

        getIDECompletions(req:ts.server.protocol.CompletionsRequestArgs,
                          entries:ts.server.protocol.CompletionEntry[]) {
            if (!entries) {
                return entries;
            }
            const file = ts_impl.normalizePath(req.file);
            const project = this.projectService.getProjectForFile(file);
            const compilerService = project.compilerService;
            const position = compilerService.host.lineOffsetToPosition(file, req.line, req.offset);
            let count = 0;
            return entries.reduce((accum:ts.server.protocol.CompletionEntry[], entry:CompletionEntry) => {
                if (count++ > 20) {
                    accum.push(entry);
                } else {
                    const details:any = compilerService.languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        }

        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         *
         */
        private getDiagnosticsExt(fileNames:string[]) {
            const checkList = fileNames.reduce((accum:ts.server.PendingErrorCheck[], fileName:string) => {
                fileName = ts_impl.normalizePath(fileName);
                const project = this.projectService.getProjectForFile(fileName);
                if (project) {
                    accum.push({fileName, project});
                }
                return accum;
            }, []);

            let result = [];

            if (checkList.length > 0) {
                for (let checkSpec of checkList) {
                    let file = checkSpec.fileName;
                    let project = checkSpec.project;

                    if (project.getSourceFileFromName(file, true)) {
                        let diagnostics = [];
                        const syntacticDiagnostics = project.compilerService.languageService.getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics) {
                            const bakedDiagnostics = syntacticDiagnostics.map((diag) => formatDiagnostics(file, checkSpec.project, diag));
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }

                        const semanticDiagnostics = project.compilerService.languageService.getSemanticDiagnostics(file);
                        if (semanticDiagnostics) {
                            const bakedSemanticDiagnostics = semanticDiagnostics.map((diag) => formatDiagnostics(file, checkSpec.project, diag));
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);

                        }

                        result.push({
                            file,
                            diagnostics
                        })
                    }
                }
            }

            return {response: result, responseRequired: true};
        }
    }

    class IDEProjectService extends TypeScriptProjectService {
        constructor(host:ts.server.ServerHost,
                    psLogger:ts.server.Logger,
                    eventHandler:ts.server.ProjectServiceEventHandler) {
            super(host, psLogger, eventHandler);
        }

        openClientFileExt(fileName:string, fileContent:string, configFileName:string) {
            if (configFileName) {
                serverLogger("Open for specified tsconfig");
                this.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            } else {
                serverLogger("Try to find tsconfig");
                this.openOrUpdateConfiguredProjectForFile(fileName);
            }
            const info = this.openFile(fileName, /*openedByClient*/ true, fileContent);
            this.addOpenFile(info);
            return info;
        }


        configFileToProjectOptions(configFilename:string):{succeeded:boolean; projectOptions?:ts.server.ProjectOptions; error?:ts.server.ProjectOpenResult} {
            function getBaseFileName(path) {
                if (path === undefined) {
                    return undefined;
                }
                var i = path.lastIndexOf(ts_impl.directorySeparator);
                return i < 0 ? path : path.substring(i + 1);
            }

            let configFileToProjectOptions:{
                succeeded:boolean;
                projectOptions?:ts.server.ProjectOptions;
                error?:ts.server.ProjectOpenResult
            } = super.configFileToProjectOptions(configFilename);

            if (configFileToProjectOptions && configFileToProjectOptions.projectOptions) {
                let projectOptions = configFileToProjectOptions.projectOptions;
                let files = projectOptions.files;

                if (files) {
                    let compilerOptions = projectOptions.compilerOptions;
                    let extensions:string[] = ts_impl.getSupportedExtensions(compilerOptions);
                    let newFiles = [];

                    l: for (let file of files) {
                        let fileName = getBaseFileName(file);
                        for (let extension of extensions) {
                            if (fileName.lastIndexOf(extension) > 0) {
                                newFiles.push(file);
                                continue l;
                            }
                        }
                        for (let extension of extensions) {
                            if (this.host.fileExists(file + extension)) {
                                newFiles.push(file + extension);
                                continue l;
                            }
                        }

                        newFiles.push(file);
                    }

                    let newOptions:{
                        succeeded:boolean;
                        projectOptions?:ts.server.ProjectOptions;
                        error?:ts.server.ProjectOpenResult
                    } = {
                        succeeded : configFileToProjectOptions.succeeded,
                        projectOptions: {
                            compilerOptions: compilerOptions,
                            files :newFiles
                        }
                    }
                    if (configFileToProjectOptions.error) {
                        newOptions.error = configFileToProjectOptions.error;
                    }

                    return newOptions;
                }
            }

            return configFileToProjectOptions;
        }
    }

    /**
     * copy formatDiag method (but we use 'TS' prefix)
     */
    function formatDiagnostics(fileName:string, project:ts.server.Project, diag:ts.Diagnostic):{
        start:ts.server.ILineInfo,
        end:ts.server.ILineInfo,
        text:string
    } {
        return {
            start: project.compilerService.host.positionToLineOffset(fileName, diag.start),
            end: project.compilerService.host.positionToLineOffset(fileName, diag.start + diag.length),
            text: "TS" + diag.code + ":" + ts_impl.flattenDiagnosticMessageText(diag.messageText, "\n")
        };
    }


    return new IDESession(ts_impl.sys, Buffer.byteLength, process.hrtime, logger);
}





