/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.VolatileNotNullLazyValue;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PsiModificationTracker;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.custom.CustomBeanParserModificationTracker;
import com.intellij.spring.contexts.model.graph.LazyModelDependenciesGraph;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.custom.CustomLocalComponentsDiscoverer;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.graph.Graph;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class AbstractSimpleLocalModel<T extends PsiElement> extends AbstractSimpleSpringModel implements LocalModel<T> {

  private static final Key<CachedValue<Map<String, LazyModelDependenciesGraph>>> MODELS_GRAPH_KEY = Key.create("MODELS_GRAPH_KEY");

  private final NotNullLazyValue<CommonSpringModel> myCustomDiscoveredBeansModel =
    new VolatileNotNullLazyValue<CommonSpringModel>() {
      @NotNull
      protected CommonSpringModel compute() {
        return new BeansSpringModel(getModule(), new NotNullLazyValue<Collection<? extends SpringBeanPointer>>() {
          @NotNull
          @Override
          protected Collection<? extends SpringBeanPointer> compute() {
            return computeCustomBeans();
          }
        });
      }
    };

  protected static void addNotNullModel(@NotNull Set<Pair<LocalModel, LocalModelDependency>> models,
                                        @Nullable LocalModel model,
                                        @NotNull LocalModelDependency dependency) {
    if (model != null) {
      models.add(Pair.create(model, dependency));
    }
  }

  private Collection<? extends SpringBeanPointer> computeCustomBeans() {
    final Set<CommonSpringBean> customSpringComponents = ContainerUtil.newLinkedHashSet();
    for (CustomLocalComponentsDiscoverer discoverer : Extensions.getExtensions(CustomLocalComponentsDiscoverer.EP_NAME)) {
      customSpringComponents.addAll(discoverer.getCustomComponents(this));
    }
    return BeanService.getInstance().mapSpringBeans(customSpringComponents);
  }

  protected CommonSpringModel getCustomDiscoveredBeansModel() {
    return myCustomDiscoveredBeansModel.getValue();
  }

  @NotNull
  public Set<LocalModel> getRelatedLocalModels() {
    Module module = getModule();
    if (module == null || module.isDisposed()) return Collections.emptySet();

    final Graph<LocalModel> graph = getOrCreateLocalModelDependenciesGraph(module, getActiveProfiles());

    Set<LocalModel> visited = ContainerUtil.newLinkedHashSet();

    visitRelated(this, graph, visited);

    visited.remove(this);

    return visited;
  }

  private static void visitRelated(@NotNull LocalModel model, @NotNull Graph<LocalModel> graph, @NotNull Set<LocalModel> visited) {
    visited.add(model);
    final Iterator<LocalModel> out = graph.getOut(model);
    while (out.hasNext()) {
      final LocalModel outModel = out.next();
      if (!visited.contains(outModel)) {
        visitRelated(outModel, graph, visited);
      }
    }
  }

  private static Object[] getDependencies(@NotNull Project project) {
    Set<Object> set = ContainerUtil.newLinkedHashSet();

    set.add(SpringModificationTrackersManager.getInstance(project).getProfilesModificationTracker());
    set.add(PsiModificationTracker.OUT_OF_CODE_BLOCK_MODIFICATION_COUNT);
    set.add(CustomBeanParserModificationTracker.getInstance());
    set.add(ProjectRootManager.getInstance(project));

    return set.toArray();
  }

  @NotNull
  // todo move to api
  public static LazyModelDependenciesGraph getOrCreateLocalModelDependenciesGraph(@NotNull final Module module,
                                                                                  @NotNull final Set<String> activeProfiles) {
    String key = SpringProfileUtils.profilesAsString(activeProfiles);
    final Map<String, LazyModelDependenciesGraph> graphsMap = CachedValuesManager.getManager(module.getProject())
      .getCachedValue(module, MODELS_GRAPH_KEY,
                      createGraphProvider(module, activeProfiles),
                      false);

    return graphsMap.get(key);
  }

  @NotNull
  private static CachedValueProvider<Map<String, LazyModelDependenciesGraph>> createGraphProvider(final Module module,
                                                                                                  final Set<String> activeProfiles) {
    return () -> {
      Map<String, LazyModelDependenciesGraph> map =
        new ConcurrentFactoryMap<String, LazyModelDependenciesGraph>() {
          @Override
          protected LazyModelDependenciesGraph create(String key) {
            return new LazyModelDependenciesGraph(module, activeProfiles);
          }
        };

      return Result.create(map, getDependencies(module.getProject()));
    };
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "[" + getConfig() + "]";
  }

  @NotNull
  protected Object[] getOutsideModelDependencies(@NotNull LocalModel model) {
    final Project project = model.getConfig().getProject();

    final List<Object> dependencies = new ArrayList<Object>(5);
    dependencies.add(model.getConfig());
    final SpringModificationTrackersManager springModificationTrackersManager =
      SpringModificationTrackersManager.getInstance(project);
    dependencies.add(springModificationTrackersManager.getProfilesModificationTracker());
    dependencies.add(springModificationTrackersManager.getMultipleContextsModificationTracker());

    dependencies.add(ProjectRootManager.getInstance(project));
    dependencies.add(PsiModificationTracker.JAVA_STRUCTURE_MODIFICATION_COUNT);

    return ArrayUtil.toObjectArray(dependencies);
  }


  @NotNull
  protected PsiElement[] getConfigs(@NotNull Set<LocalModel> localModels) {
    return ContainerUtil.map2Array(localModels, PsiElement.class, LocalModel::getConfig);
  }
}
