/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.psi.ElementManipulators;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.PsiReference;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

/**
 * @author Dmitry Avdeev
 */
public class PackageListConverter extends Converter<Collection<PsiPackage>> implements CustomReferenceConverter {

  @NotNull
  public Collection<PsiPackage> fromString(@Nullable @NonNls final String s, final ConvertContext context) {
    if (s == null) {
      return Collections.emptyList();
    }
    if (context.getXmlElement() == null) {
      return Collections.emptyList();
    }

    if (SpringConverterUtil.containsPatternReferences(s)) {
      final PsiReference[] references = createReferences((GenericDomValue)context.getInvocationElement(), context.getXmlElement(), context);
      return SpringConverterUtil.getPsiPackages(references);
    }

    return SpringConverterUtil.getPackages(s, getDelimiters(), context.getPsiManager().getProject());
  }

  public String toString(@Nullable final Collection<PsiPackage> psiPackages, final ConvertContext context) {
    return null;
  }

  @NotNull
  public PsiReference[] createReferences(final GenericDomValue genericDomValue, final PsiElement element, final ConvertContext context) {
    final String text = genericDomValue.getStringValue();
    if (text == null) {
      return PsiReference.EMPTY_ARRAY;
    }
    GlobalSearchScope scope = context.getSearchScope();
    return SpringReferenceUtils.getPsiPackagesReferences(element, text, ElementManipulators.getOffsetInElement(element), getDelimiters(),
                                                         scope != null ? scope : GlobalSearchScope.allScope(element.getProject()));
  }

  protected String getDelimiters() {
    return ",; \n\t";
  }
}
