/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.javaConfig;

import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiMethod;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.SpringManager;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.contexts.model.SpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringBeanUtils;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class SpringOldJavaConfigurationUtil {

  public static List<JavaConfigConfiguration> getJavaConfigurations(@NotNull Module module) {
    final JamService jamService = JamService.getJamService(module.getProject());

    return jamService.getJamClassElements(JavaConfigConfiguration.META,
                                          SpringAnnotationsConstants.JAVA_CONFIG_CONFIGURATION,
                                          GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module));
  }

  @NotNull
  public static List<SpringBeanPointer> findExternalBeans(final PsiMethod psiMethod) {
    final Module module = ModuleUtilCore.findModuleForPsiElement(psiMethod);
    final PsiClass psiClass = psiMethod.getContainingClass();
    if (module == null || psiClass == null) {
      return Collections.emptyList();
    }
    if (getExternalBean(psiMethod) == null) {
      return Collections.emptyList();
    }
    List<SpringBeanPointer> extBeans = new SmartList<SpringBeanPointer>();
    for (SpringModel model : SpringManager.getInstance(psiMethod.getProject()).getAllModels(module)) {
      boolean hasJavaConfigBean = SpringModelSearchers.doesBeanExist(model, SpringModelSearchParameters.byClass(psiClass));
      if (hasJavaConfigBean) {
        final String externalBeanName = psiMethod.getName();
        for (SpringBeanPointer springBean : model.getAllDomBeans()) {
          final String beanName = springBean.getName();
          if (externalBeanName.equals(beanName) || Arrays.asList(springBean.getAliases()).contains(externalBeanName)) {
            extBeans.add(springBean);
          }
        }
      }
    }
    return extBeans;
  }

  @NotNull
  public static List<SpringJavaExternalBean> findExternalBeanReferences(final CommonSpringBean springBean) {
    final XmlTag element = springBean.getXmlTag();
    if (element == null) {
      return Collections.emptyList();
    }
    final Module module = ModuleUtilCore.findModuleForPsiElement(element);
    if (module == null) {
      return Collections.emptyList();
    }

    final Set<String> beanNames = SpringBeanUtils.getInstance().findBeanNames(springBean);
    if (beanNames.isEmpty()) {
      return Collections.emptyList();
    }

    List<SpringJavaExternalBean> extBeans = new ArrayList<SpringJavaExternalBean>();
    for (SpringJavaConfiguration javaConfiguration : getJavaConfigurations(module)) {
      if (javaConfiguration instanceof JavaConfigConfiguration) {
        for (SpringJavaExternalBean externalBean : ((JavaConfigConfiguration)javaConfiguration).getExternalBeans()) {
          final PsiMethod psiMethod = externalBean.getPsiElement();
          if (beanNames.contains(psiMethod.getName())) {
            extBeans.add(externalBean);
          }
        }
      }
    }
    return extBeans;
  }

  @Nullable
  public static SpringJavaExternalBean getExternalBean(final PsiMethod psiMethod) {
    final Module module = ModuleUtilCore.findModuleForPsiElement(psiMethod);
    if (module != null) {
      for (SpringJavaConfiguration javaConfiguration : getJavaConfigurations(module)) {
        if (javaConfiguration instanceof JavaConfigConfiguration) {
          if (psiMethod.getContainingFile().equals(javaConfiguration.getPsiClass().getContainingFile())) {
            for (SpringJavaExternalBean externalBean : ((JavaConfigConfiguration)javaConfiguration).getExternalBeans()) {
              if (psiMethod.equals(externalBean.getPsiElement())) {
                return externalBean;
              }
            }
          }
        }
      }
    }
    return null;
  }
}
