/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.stereotype;

import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiAnchor;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

public abstract class SpringMetaStereotypeComponent extends SpringStereotypeElement {

  private static final NullableFunction<PsiClass, String> PSI_CLASS_FQN = PsiClass::getQualifiedName;

  @Nullable private final String myAnno;
  private final PsiAnchor myPsiClassAnchor;

  public SpringMetaStereotypeComponent(@NotNull PsiClass psiClassAnchor) {
    this(null, psiClassAnchor);
  }

  public SpringMetaStereotypeComponent(@Nullable String anno, @NotNull PsiClass psiClassAnchor) {
    super(anno);

    myAnno = anno;
    myPsiClassAnchor = PsiAnchor.create(psiClassAnchor);
  }

  protected static Collection<String> getAnnotations(@Nullable Module module,
                                                     @NotNull String annotation) {
    if (module == null || module.isDisposed()) {
      return Collections.singleton(annotation);
    }

    final Collection<PsiClass> classes = JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(annotation);
    return ContainerUtil.mapNotNull(classes, PSI_CLASS_FQN);
  }

  @NotNull
  public PsiClass getPsiElement() {
    PsiElement element = myPsiClassAnchor.retrieve();
    if (element == null) {
      String msg;
      if (myPsiClassAnchor instanceof PsiAnchor.StubIndexReference) {
        msg = ((PsiAnchor.StubIndexReference)myPsiClassAnchor).diagnoseNull();
      }
      else {
        msg = "Anchor hasn't survived: " + myPsiClassAnchor;
      }
      throw new RuntimeException(msg);
    }

    return (PsiClass)element;
  }

  @Override
  public boolean isValid() {
    PsiElement psiElement = myPsiClassAnchor.retrieve();
    return psiElement != null && psiElement.isValid();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof SpringMetaStereotypeComponent)) return false;

    SpringMetaStereotypeComponent that = (SpringMetaStereotypeComponent)o;

    if (myAnno != null ? !myAnno.equals(that.myAnno) : that.myAnno != null) return false;
    if (myPsiClassAnchor != null ? !myPsiClassAnchor.equals(that.myPsiClassAnchor) : that.myPsiClassAnchor != null) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = myAnno != null ? myAnno.hashCode() : 0;
    result = 31 * result + (myPsiClassAnchor != null ? myPsiClassAnchor.hashCode() : 0);
    return result;
  }
}