/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.psi.PsiClass;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.Processors;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * Provides convenience methods to find or check existence of beans with specific type/name.
 * <p/>
 * All methods must be invoked under read-action.
 *
 * @since 14
 */
public class SpringModelSearchers {

  @NotNull
  public static List<SpringBeanPointer> findBeans(@NotNull CommonSpringModel model,
                                                  @NotNull SpringModelSearchParameters.BeanClass parameters) {
    if (!parameters.canSearch()) {
      return Collections.emptyList();
    }

    final List<SpringBeanPointer> beans = new SmartList<SpringBeanPointer>();
    Processor<SpringBeanPointer> processor = Processors.cancelableCollectProcessor(beans);
    model.processByClass(parameters, processor);

    return beans;
  }

  @Nullable
  public static SpringBeanPointer findBean(@NotNull CommonSpringModel model, @NotNull String beanName) {
    CommonProcessors.FindFirstProcessor<SpringBeanPointer> findFirstProcessor =
      new CommonProcessors.FindFirstProcessor<SpringBeanPointer>();
    model.processByName(SpringModelSearchParameters.byName(beanName), findFirstProcessor);
    return findFirstProcessor.getFoundValue();
  }

  public static boolean doesBeanExist(@NotNull CommonSpringModel model,
                                      @NotNull SpringModelSearchParameters.BeanClass parameters) {
    return containsBean(model, parameters);
  }

  /**
   * Checks whether a bean with the given class (actual, inheritor or effective type) exists.
   * <p/>
   * NOTE: Expensive operation. Use {@link #doesBeanExist(CommonSpringModel, SpringModelSearchParameters.BeanClass)} for "plain" class
   * search.
   *
   * @param model
   * @param beanClass
   * @return
   */
  public static boolean doesBeanExist(@NotNull CommonSpringModel model, @NotNull PsiClass beanClass) {
    return containsBean(model, SpringModelSearchParameters.byClass(beanClass).withInheritors().effectiveBeanTypes());
  }

  private static boolean containsBean(@NotNull CommonSpringModel model,
                                      @NotNull final SpringModelSearchParameters.BeanClass parameters) {
    if (!parameters.canSearch()) {
      return false;
    }

    CommonProcessors.FindFirstProcessor<SpringBeanPointer> findFirstProcessor =
      new CommonProcessors.FindFirstProcessor<SpringBeanPointer>();

    model.processByClass(parameters, findFirstProcessor);

    return findFirstProcessor.isFound();
  }
}