/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.Processor;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

public class SpringProfileUtils {

  @NotNull
  public static Set<String> getAllProfiles(@Nullable Beans beans) {
    final Set<String> names = ContainerUtil.newLinkedHashSet();

    processProfiles(beans, nestedBeans -> {
      names.addAll(nestedBeans.getProfile().getNames());
      return true;
    });

    return names;
  }

  public static boolean processProfiles(@Nullable Beans beans, @NotNull Processor<Beans> processor) {
    if (beans != null) {
      if (!processor.process(beans)) return false;
      for (Beans childrenBeans : beans.getBeansProfiles()) {
        if (!processProfiles(childrenBeans, processor)) return false;
      }
    }
    return true;
  }

  public static boolean isInActiveProfile(@NotNull DomElement value) {
    final Beans beans = DomUtil.getParentOfType(value, Beans.class, true);
    if (beans != null) {
      final CommonSpringModel model = SpringModelUtils.getInstance().getSpringModel(value.getXmlElement());
      return isActiveProfile(beans, model.getActiveProfiles());
    }

    return true;
  }

  public static boolean isInActiveProfile(@NotNull DomElement domElement, @NotNull final Set<String> profiles) {
    final Beans beans = DomUtil.getParentOfType(domElement, Beans.class, true);
    if (beans != null) {
      return isActiveProfile(beans, profiles);
    }

    return true;
  }

  public static boolean isActiveProfile(@Nullable Beans beans, @Nullable Set<String> activeProfiles) {
    if (beans == null) return false;
    if (activeProfiles == null || activeProfiles.isEmpty()) return true;

    if (isActive(beans.getProfile().getNames(), activeProfiles)) {
      //IDEA-67463
      final Beans parentBeans = DomUtil.getParentOfType(beans, Beans.class, true);

      return parentBeans == null || isActiveProfile(parentBeans, activeProfiles);
    }

    if (StringUtil.isEmptyOrSpaces(beans.getProfile().getStringValue())) {
      final Beans parentBeans = DomUtil.getParentOfType(beans, Beans.class, true);

      return parentBeans == null || isActiveProfile(parentBeans, activeProfiles); //"default" or "inherited" profile
    }
    return false;
  }

  public static boolean isActive(@NotNull Set<String> profiles, @NotNull Set<String> activeProfiles) {
    if (profiles.size() == 1 && profiles.iterator().next().equals(SpringProfile.DEFAULT_PROFILE_NAME)) return true;
    for (String profile : profiles) {
      if (profile.startsWith("!")) {
        String notProfile = profile.substring(1);
        if (!activeProfiles.contains(notProfile)) {
          return true;
        }
      }
      else if (activeProfiles.contains(profile)) {
        return true;
      }
    }
    return false;
  }

  @NotNull
  public static String profilesAsString(@Nullable Set<String> activeProfiles) {
    if (activeProfiles == null ||
        activeProfiles.isEmpty()) {
      return "";
    }

    final Set<String> profiles = new TreeSet<String>();

    for (String activeProfile : activeProfiles) {
      if (!SpringProfile.DEFAULT_PROFILE_NAME.equals(activeProfile)) {
        profiles.add(activeProfile);
      }
    }

    if (profiles.size() > 1) {
      return StringUtil.join(profiles, ", ");
    }
    return ContainerUtil.getFirstItem(profiles, "");
  }

  @NotNull
  public static <T extends CommonSpringBean> List<T> filterBeansInActiveProfiles(@NotNull Collection<T> allBeans,
                                                                                 @Nullable Set<String> activeProfiles) {
    if (activeProfiles == null || activeProfiles.isEmpty()) return new SmartList<T>(allBeans);
    List<T> profiledStereotypes = new SmartList<T>();
    for (T stereotypeElement : allBeans) {
      if (stereotypeElement.isValid()) {
        final SpringProfile profile = stereotypeElement.getProfile();
        if (isProfileAccepted(profile, activeProfiles)) {
          profiledStereotypes.add(stereotypeElement);
        }
      }
    }
    return profiledStereotypes;
  }

  public static boolean isProfileAccepted(@NotNull SpringProfile profile, @Nullable Set<String> activeProfiles) {
    if (profile == SpringProfile.DEFAULT || activeProfiles == null || activeProfiles.isEmpty()) return true;
    return isActive(profile.getNames(), activeProfiles);
  }
}
