/// <reference path="v8debugger.d.ts" />

"use strict";

// http://webreflection.blogspot.de/2013/04/playing-with-v8-native-and-javascript.html

module $_JbV8DebuggerSupport {
  import ExecutionState = v8debug.ExecutionState;
  import ValueMirror = v8debug.ValueMirror;

  import Mirror = v8debug.Mirror;
  import ScriptMirror = v8debug.ScriptMirror;
  import ScopeMirror = v8debug.ScopeMirror;
  import ObjectMirror = v8debug.ObjectMirror;
  import FrameMirror = v8debug.FrameMirror;
  import StringMirror = v8debug.StringMirror;
  import FunctionMirror = v8debug.FunctionMirror;
  import ArrayMirror = v8debug.ArrayMirror;
  import DateMirror = v8debug.DateMirror;
  import RegExpMirror = v8debug.RegExpMirror;

  import PropertyKind = v8debug.PropertyKind;

  import FrameDetails = v8debug.FrameDetails;
  import ScopeHost = v8debug.ScopeHost;
  import ScopeDetails = v8debug.ScopeDetails;
  import SourceLocation = v8debug.SourceLocation;
  import DebugCommandProcessor = v8debug.DebugCommandProcessor;

  DateMirror.prototype.isDate = function () {
    return true;
  };
  DateMirror.prototype.isRegExp = function () {
    return false;
  };
  DateMirror.prototype.isArray = function () {
    return false;
  };
  RegExpMirror.prototype.isDate = function () {
    return false;
  };
  RegExpMirror.prototype.isRegExp = function () {
    return true;
  };
  RegExpMirror.prototype.isArray = function () {
    return false;
  };
  ObjectMirror.prototype.isUndefined = function () {
    return false;
  };
  ObjectMirror.prototype.isNull = function () {
    return false;
  };

  // node adds \n\n})
  const nodejsSuffix = "(?:[\\r\\n]+[});]+)?"
  // https://youtrack.jetbrains.com/issue/WEB-20997#comment=27-1372063
  const otherComments = "(?:[\\r\\n]+//[^\\r\\n]+)?"
  const sourceMapRegExp = new RegExp("[\\r\\n]//[@|#][ \\t]sourceMappingURL=[ \\t]*(?:file://)?([^\\r\\n]*)\\s*(?:\\*/\\s*)?" + nodejsSuffix + otherComments + "$")

  var ScopeType = {
    Global: 0,
    Local: 1,
    With: 2,
    Closure: 3,
    Catch: 4,
    Block: 5,
    Script: 6
  };

  function truncateString(stringMirror: StringMirror, maxLength: number) {
    if (maxLength != -1 && stringMirror.length() > maxLength) {
      return stringMirror.value().substring(0, maxLength);
    }
    else {
      return stringMirror.value();
    }
  }

  export interface GetScopeObject {
    scopeIndex:number;
    frameIndex:number;

    functionId:number;

    includeProperties:boolean;
  }

  export interface Evaluate {
    expression:string;
    frame:number;

    additionalContext:Array<any>;

    includeProperties:boolean;
    enableBreak:boolean;
  }

  export interface GetObjects {
    ids:Array<number>;
    maxStringLength:number;
    includeProperties:boolean;
  }

  export interface GetProperties {
    objectId:number
    includeIndexed: boolean
  }

  export interface GetIndexedProperties {
    arrayId:number;
    bucketThreshold:number;

    from:number;
    to:number;

    componentType:ValueType;
  }

  export interface GetFrames {
    from:number;
    to:number;
  }

  export interface GetFrameReceiver {
    index:number;
    includeProperties:boolean;
  }

  export interface GetFunction {
    id:number;
  }

  export interface Locatable {
    line:number;
    column:number;
  }

  export class FunctionDescriptor implements Locatable {
    scriptId: number;
    scopes: Array<ScopeDescriptor>;

    line: number;
    column: number;

    constructor(scriptId: number, scopes: Array<ScopeDescriptor>, location: SourceLocation) {
      this.scriptId = scriptId;
      this.scopes = scopes;
      serializeLocationFields(location, this);
    }
  }

  export class PropertyDescriptor {
    name: string;
    type: ValueType;
    value: string|number;

    objectId: number;

    length: number;

    getter: number;
    setter: number;

    // string or number
    constructor(name: any) {
      this.name = name;
    }
  }

  export class ObjectDescriptor {
    id: number;
    type: number;

    className: string;
    description: string;

    // length of array or string
    length: number;

    constructor(id: number, type: number, description: string, length: number) {
      this.id = id;
      this.type = type;
      if (description != null) {
        this.description = description;
      }
      if (length != -1) {
        this.length = length;
      }
    }

    properties: Array<PropertyDescriptor>;
  }

  export interface ScopeDescriptor {
    type:number;
    index:number;
  }

  class FrameDescriptor implements Locatable {
    index: number;

    functionName: string;
    functionId: number;

    scriptId: number;
    scopes: Array<ScopeDescriptor>;

    line: number;
    column: number;

    hasReceiver: boolean;

    constructor(index: number, functionName: string, functionId: number, scriptId: number, scopes: Array<ScopeDescriptor>, location: SourceLocation, hasReceiver: boolean) {
      this.index = index;

      if (functionName != null) {
        this.functionName = functionName;
      }
      this.functionId = functionId;

      this.scriptId = scriptId;
      this.scopes = scopes;
      this.hasReceiver = hasReceiver;

      serializeLocationFields(location, this);
    }
  }

  var MAX_STRING_LENGTH = 1000;
  var STRING_MIRROR_THRESHOLD = 4096;

  export const enum ValueType {
    //noinspection JSUnusedGlobalSymbols
    OBJECT,
    NUMBER,
    STRING,
    FUNCTION,
    BOOLEAN,

    ARRAY,
    NODE,

    UNDEFINED,
    NULL,
  }

  // don't add position, line/column is enough
  function serializeLocationFields(location: SourceLocation, result: Locatable): void {
    if (!location) {
      return;
    }

    var line = location.line;
    if (line !== undefined) {
      result.line = line;
    }
    var column = location.column;
    if (column !== undefined) {
      result.column = column;
    }
  }

  export function stringifyObjects(list: Array<ObjectDescriptor>): string {
    return stringifyList(list, "objects")
  }

  export function stringifyList(list: Array<any>, name: string): string {
    return list == null || list.length == 0 ? "" : ', "' + name + '": ' + JSON.stringify(list)
  }

  function isLineBreak(s: string, index: number): boolean {
    var char = s.charCodeAt(index);
    return char === 10 || char == 13;
  }

  function isWhitespace(s: string, index: number): boolean {
    var char = s.charCodeAt(index);
    return char === 32 || char === 10 || char == 13 || char == 9;
  }

  export class CacheWrapper {
    public getMirror: (id: number) => ValueMirror;
    public clearCache: () => void;
    public v8mc: () => Array<ValueMirror>;

    public findMirror: (object: any) => ValueMirror;
    public cacheMirror: (mirror: ValueMirror) => void;

    constructor() {
      var cache: Array<ValueMirror> = [];
      var v8mc = v8debug.mirror_cache_;

      this.getMirror = function (id: number) {
        var mirror = cache[id]
        if (mirror == null && v8mc != null) {
          mirror = v8mc[id]
        }

        if (mirror == null) {
          throw new Error('Object ' + id + ' not found, v8debug.mirror_cache_ is ' + typeof v8debug.mirror_cache_ + " " + typeof cacheManager.v8mc());
        }
        return mirror;
      };

      this.v8mc = function () {
        return v8mc
      };

      this.clearCache = function () {
        cache.length = 0
      };

      this.findMirror = function (object: any) {
        return CacheWrapper.doFindMirror(object, cache)
      };

      this.cacheMirror = function (mirror: ValueMirror) {
        if (v8mc != null) {
          v8mc[mirror.handle()] = mirror
        }
        cache[mirror.handle()] = mirror;
      }
    }

    static doFindMirror(object: any, cache: Array<ValueMirror>): ValueMirror {
      for (var key in cache) {
        if (cache.hasOwnProperty(key)) {
          var mirror: ValueMirror = cache[key];
          if (mirror.value() === object) {
            return mirror;
          }
        }
      }
      return null;
    }
  }

  export var cacheManager = new CacheWrapper();

  export class DebuggerSupport {
    private getObjectMirror(id: number): ValueMirror {
      return cacheManager.getMirror(id);
    }

    getFrames(request: GetFrames, executionState: ExecutionState): string {
      var totalCount = executionState.frameCount()
      var from = 0
      var to = totalCount
      if (request.from != null) {
        from = request.from
        if (from < 0) {
          throw new Error('Invalid from index')
        }
      }
      if (request.to != null) {
        to = Math.min(totalCount, request.to)
      }

      if (to <= from) {
        if (request.to == null) {
          return '{"totalCount": ' + totalCount + ',"frames":[]}'
        }

        throw new Error('Invalid frame range: from ' + from + ", to " + request.to + ", total " + totalCount)
      }

      var objects: Array<ObjectDescriptor> = [];
      var frames = new Array<FrameDescriptor>(to - from);
      for (var frameIndex = from, arrayIndex = 0; frameIndex < to; frameIndex++, arrayIndex++) {
        frames[arrayIndex] = this.describeFrame(executionState.frame(frameIndex), objects)
      }
      return '{"totalCount": ' + totalCount + ',"frames":' + JSON.stringify(frames) + stringifyObjects(objects) + '}'
    }

    getFrameReceiver(request: GetFrameReceiver, executionState: ExecutionState) {
      var frameMirror: FrameMirror = executionState.frame(request.index);
      var receiver = frameMirror.details_.receiver();
      if (receiver == null) {
        return '{}';
      }

      var objects: Array<ObjectDescriptor> = [];
      var objectId = this.getOrCreateMirror(receiver, objects, request.includeProperties, MAX_STRING_LENGTH).handle();
      return '{"objectId":' + objectId + stringifyObjects(objects) + '}';
    }

    getScopeObject(request: GetScopeObject, executionState: ExecutionState): string {
      var frameMirror: FrameMirror = request.frameIndex === undefined ? null : executionState.frame(request.frameIndex);
      var functionMirror: FunctionMirror = frameMirror === null ? <FunctionMirror>this.getObjectMirror(request.functionId) : null;
      var scopeHolder: ScopeHost = frameMirror == null ? functionMirror : frameMirror;

      var scopeIndex = request.scopeIndex;
      if (scopeIndex < 0 || scopeIndex >= scopeHolder.scopeCount()) {
        throw new Error('Invalid scope index');
      }

      var scopeDetails = scopeHolder.scope(scopeIndex).details_
      var isTransient = scopeDetails.type() == ScopeType.Local || scopeDetails.type() == ScopeType.Closure;
      var objects: Array<ObjectDescriptor> = [];
      var id = this.getOrCreateMirror(scopeDetails.object(), objects, request.includeProperties, MAX_STRING_LENGTH, isTransient).handle();
      return '{"id":' + id + stringifyObjects(objects) + '}';
    }

    evaluate(request: Evaluate, executionState: ExecutionState): string {
      var additionalContext: any;
      if (request.additionalContext != null) {
        additionalContext = {}
        for (var i = 0, n = request.additionalContext.length; i < n; i++) {
          var mapping = request.additionalContext[i]
          if ("handle" in mapping) {
            additionalContext[mapping.name] = this.getObjectMirror(mapping.handle).value()
          }
          else {
            additionalContext[mapping.name] = DebugCommandProcessor.resolveValue_(mapping)
          }
        }
      }

      var valueMirror: ValueMirror
      try {
        if (request.frame == -1) {
          valueMirror = executionState.evaluateGlobal(request.expression, !request.enableBreak, additionalContext)
        }
        else {
          valueMirror = executionState.frame(request.frame).evaluate(request.expression, !request.enableBreak, additionalContext)
        }
      }
      catch (e) {
        return this.wrapWasThrown(e, request.includeProperties)
      }

      var propertyDescriptor = new PropertyDescriptor("");
      switch (valueMirror.type()) {
        case 'undefined':
          propertyDescriptor.type = ValueType.UNDEFINED
          break;
        case 'null':
          propertyDescriptor.type = ValueType.NULL
          break;
        case 'boolean':
          propertyDescriptor.type = ValueType.BOOLEAN;
          propertyDescriptor.value = valueMirror.value()
          break;
        case 'number':
          propertyDescriptor.type = ValueType.NUMBER;
          propertyDescriptor.value = numberToJson(valueMirror.value())
          break;
        case 'string':
          var s: string = <string>valueMirror.value()
          if (s.length <= STRING_MIRROR_THRESHOLD) {
            propertyDescriptor.type = ValueType.STRING
            propertyDescriptor.value = s;
          }
          break;
      }

      if (propertyDescriptor.type == null) {
        var objects: Array<ObjectDescriptor> = []
        propertyDescriptor.objectId = valueMirror.handle()
        this.describeObjectMirror(valueMirror, request.includeProperties, objects, MAX_STRING_LENGTH)
        // ensure that mirror cached, WEB-12794
        cacheManager.cacheMirror(valueMirror)
        return '{"result":' + JSON.stringify(propertyDescriptor) + stringifyObjects(objects) + '}'
      }
      else {
        // remove ugly V8 value mirror from cache
        var v8MirrorCache = cacheManager.v8mc()
        if (v8MirrorCache != null) {
          delete v8MirrorCache[valueMirror.handle()]
        }
        return '{"result":' + JSON.stringify(propertyDescriptor) + '}';
      }
    }

    wrapWasThrown(e: Error | string, includeProperties: boolean): string {
      var propertyDescriptor = new PropertyDescriptor("wasThrown")

      if (typeof e === 'string') {
        propertyDescriptor.type = ValueType.STRING
        propertyDescriptor.value = e
        return '{"result":' + JSON.stringify(propertyDescriptor) + '}'
      }

      var objects: Array<ObjectDescriptor> = [];
      propertyDescriptor.type = ValueType.OBJECT;

      // wrapped error object should have detailed description opposite to common serialization (the same as WIP does)
      var mirror = new ObjectMirror(e, 'object', false);
      cacheManager.cacheMirror(mirror);
      this.describeObject2(mirror, mirror.className(), e.toString(), includeProperties, MAX_STRING_LENGTH, objects);
      propertyDescriptor.objectId = mirror.handle();
      return '{"result":' + JSON.stringify(propertyDescriptor) + stringifyObjects(objects) + '}';
    }

    describeObjectMirror(mirror: ValueMirror, includeProperties: boolean, objects: Array<ObjectDescriptor>, maxStringLength: number): void {
      switch (mirror.type()) {
        case 'function':
          this.describeFunction(<FunctionMirror>mirror, includeProperties, maxStringLength, objects);
          break;

        case 'string':
          this.describeString(<StringMirror>mirror, maxStringLength, objects);
          break;

        default:
          if (mirror.isArray()) {
            this.describeArray(<ArrayMirror>mirror, includeProperties, maxStringLength, objects);
          }
          else {
            this.describeObject(<ObjectMirror>mirror, includeProperties, maxStringLength, objects);
          }
          break;
      }
    }

    getObjects(request: GetObjects): string {
      var ids = request.ids;
      var objects: Array<ObjectDescriptor> = [];
      var maxStringLength = request.maxStringLength === undefined ? MAX_STRING_LENGTH : request.maxStringLength;
      for (var i = 0, n = ids.length; i < n; i++) {
        this.describeObjectMirror(this.getObjectMirror(ids[i]), request.includeProperties, objects, maxStringLength);
      }
      return '{"objects":' + JSON.stringify(objects) + '}';
    }

    getProperties(request: GetProperties): string {
      var mirror = <ObjectMirror>this.getObjectMirror(request.objectId);
      var objects: Array<ObjectDescriptor> = [];
      return '{"properties":' + JSON.stringify(this.describeProperties(mirror, objects)) + stringifyObjects(objects) + '}';
    }

    getIndexedProperties(request: GetIndexedProperties): string {
      var mirror = <ObjectMirror>this.getObjectMirror(request.arrayId)
      var size = 0;
      var value: any = mirror.value_
      var isSparseArray = mirror.isArray()

      var to = request.to
      if (to === -1) {
        to = value.length
      }

      if (isSparseArray) {
        for (var i = request.from; i < to; i++) {
          if (i in value) {
            size++;
          }
        }
      }
      else {
        // nodejs Buffer
        size = to - request.from;
      }

      if (size <= request.bucketThreshold) {
        if (request.componentType !== undefined) {
          var result = '{"values":[';
          var isFirst = true;
          var valueConvertor = request.componentType == ValueType.NUMBER ? numberToJson : JSON.stringify;
          for (var i = request.from; i < to; i++) {
            if (isFirst) {
              isFirst = false;
            }
            else {
              result += ",";
            }
            result += valueConvertor(value[i]);
          }
          return result + ']}';
        }

        var objects: Array<ObjectDescriptor> = [];
        var properties = new Array<PropertyDescriptor>(size);
        if (isSparseArray) {
          for (var i = request.from, j = 0; i < to; i++) {
            if (i in value) {
              properties[j++] = this.describeProperty(value[i], i, objects, MAX_STRING_LENGTH);
            }
          }
        }
        else {
          for (var i = request.from, j = 0; i < to; i++) {
            properties[j++] = this.describeProperty(value[i], i, objects, MAX_STRING_LENGTH);
          }
        }
        return '{"properties":' + JSON.stringify(properties) + stringifyObjects(objects) + '}';
      }

      var bucketSize = Math.pow(request.bucketThreshold, Math.ceil(Math.log(size) / Math.log(request.bucketThreshold)) - 1)
      var start = -1;
      var end = 0;
      var count = 0;
      var result = "";
      var isFirst = true;
      // we don't implement optimized version for not-sparse array - client must build ranges
      for (var i = request.from; i < to; i++) {
        if (!isSparseArray || i in value) {
          if (start == -1) {
            start = i;
          }
          end = i;
          if (++count == bucketSize) {
            if (isFirst) {
              isFirst = false;
            }
            else {
              result += ",";
            }
            result += start + "," + end;
            count = 0;
            start = -1;
          }
        }
      }

      if (count > 0) {
        if (!isFirst) {
          result += ",";
        }
        result += start + "," + end;
      }

      return '{"ranges":[' + result + ']}';
    }

    getFunction(request: GetFunction): string {
      var mirror = <FunctionMirror>this.getObjectMirror(request.id)
      var script = mirror.script()
      return '{"fun":' + JSON.stringify(new FunctionDescriptor(script == null ? -1 : script.id(), DebuggerSupport.getScopes(null, mirror, mirror), mirror.sourceLocation())) + '}'
    }

    /**
     * return null if script should be ignored
     */
    describeScript(script: v8debug.Script): any {
      var name = script.source_url || script.name
      if (name == null /* source void 0; - skip it */) {
        return null
      }

      var source = script.source
      if (source == null || source.length === 0 || (source[0] == 'j' && source.lastIndexOf("javascript:void(0);", 0) === 0)) {
        return null
      }

      var descriptor: any = {
        id: script.id,
        path: name,
        type: script.type,
        endLine: (script.line_offset + script.line_ends.length) - 1
      }

      var startLine = script.line_offset
      if (startLine !== 0) {
        descriptor.startLine = startLine
      }

      var startColumn = script.column_offset
      if (startColumn !== 0) {
        descriptor.startColumn = startColumn
      }

      var result = sourceMapRegExp.exec(source)
      if (result != null && result.length > 1) {
        descriptor.sourceMapUrl = result[1]
      }
      return descriptor
    }

    /**
     * Don't serialize locals. It must be requested later via scope if need.
     * Don't serialize sourceLineText. What for?
     * Don't serialize arguments (frame function invocation arguments). It must be requested later via scope if need.
     * Don't serialize script - get via func
     */
    describeFrame(frame: FrameMirror, objects: Array<ObjectDescriptor>, maxStringLength: number = MAX_STRING_LENGTH): FrameDescriptor {
      var frameDetails = frame.details_;
      var scopes = DebuggerSupport.getScopes(frame, null, frame);
      var functionMirror = <FunctionMirror>this.getOrCreateMirror(frameDetails.func(), objects, false, maxStringLength);
      var script = functionMirror.script();
      var scriptId: number = script == null ? -1 : script.id();
      return new FrameDescriptor(frame.index(), DebuggerSupport.getFunctionName(functionMirror), functionMirror.handle(), scriptId, scopes, frame.sourceLocation(),
          frameDetails.receiver() !== undefined);
    }

    private static getScopes(frameMirror: FrameMirror, functionMirror: FunctionMirror, scopeHost: ScopeHost): Array<ScopeDescriptor> {
      // WEB-18853 Debugger: debugging doesn't work with Node.js 5
        if (frameMirror != null && "allScopes" in frameMirror) {
        // new v8
        var scopeMirrors = frameMirror.allScopes(true)
        var scopeCount = scopeMirrors.length
        var scopes = new Array<ScopeDescriptor>(scopeCount)
        for (var i = 0; i < scopeCount; i++) {
          scopes[i] = {type: scopeMirrors[i].scopeType(), index: i};
        }
        return scopes
      }
      else {
        var scopeCount = scopeHost.scopeCount()
        var scopes = new Array<ScopeDescriptor>(scopeCount)
        if (functionMirror != null && "scope" in functionMirror) {
          for (var i = 0; i < scopeCount; i++) {
            scopes[i] = {type: functionMirror.scope(i).scopeType(), index: i};
          }
        }
        else {
          for (var i = 0; i < scopeCount; i++) {
            scopes[i] = {type: new ScopeDetails(frameMirror, functionMirror, i).type(), index: i};
          }
        }
        return scopes
      }
    }

    private getOrCreateMirror(object: any, objects: Array<ObjectDescriptor>, includeProperties: boolean, maxStringLength: number, isTransient: boolean = false, isProto: boolean = false): ValueMirror {
      if (!isTransient) {
        var existingMirror = cacheManager.findMirror(object);
        if (existingMirror != null) {
          return existingMirror;
        }
      }

      var type = typeof object;
      if (type === 'string') {
        return this.mirrorAndDescribeString(object, maxStringLength, objects)
      }
      else if (Array.isArray(object)) {
        var arrayMirror = new ArrayMirror(object)
        cacheManager.cacheMirror(arrayMirror)
        this.describeArray(arrayMirror, includeProperties, maxStringLength, objects)
        return arrayMirror;
      }
      else if (type === "function") {
        var functionMirror = new FunctionMirror(object)
        cacheManager.cacheMirror(functionMirror)
        this.describeFunction(functionMirror, includeProperties, maxStringLength, objects)
        return functionMirror;
      }
      else {
        //noinspection JSUnresolvedVariable
        var constructorName = DebuggerSupport.getConstructorName(object)
        if (constructorName === "String") {
          return this.mirrorAndDescribeString(object, maxStringLength, objects)
        }
        else {
          var objectMirror: ObjectMirror
          if (constructorName === "Date") {
            objectMirror = new DateMirror(object);
          }
          else if (constructorName === "RegExp") {
            objectMirror = new RegExpMirror(object)
          }
          else {
            objectMirror = new ObjectMirror(object, 'object', isTransient);
          }
        }

        cacheManager.cacheMirror(objectMirror);
        this.describeObject(objectMirror, includeProperties, maxStringLength, objects, isProto)
        return objectMirror;
      }
    }

    private static getConstructorName(object: any): string {
      //noinspection JSUnresolvedVariable
      return typeof object.constructor === "function" ? object.constructor.name : null;
    }

    private describeObject(mirror: ObjectMirror, includeProperties: boolean, maxStringLength: number, objects: Array<ObjectDescriptor>, isProto: boolean = false): ObjectDescriptor {
      var description: string
      var className: string
      if (mirror.isDate() || mirror.isRegExp()) {
        description = mirror.value().toString()
        className = mirror.isDate() ? "Date" : "RegExp"
      }
      else {
        className = DebuggerSupport.getClassName(mirror)
        if (!isProto && (className === "Buffer" || isTypedArray(className))) {
          let descriptor = this.describe(mirror, ValueType.ARRAY, className, mirror.value().length, mirror, includeProperties, maxStringLength, objects)
          descriptor.className = className
          return descriptor
        }
      }
      return this.describeObject2(mirror, className, description, includeProperties, maxStringLength, objects)
    }

    private static getClassName(mirror: ObjectMirror): string {
      var className = mirror.className();
      // V8 returns "Object" if constructor has empty name also (stupid fallback?), so, we must use next logic regardless of it
      if (!(className == null || className.length == 0 || className === "Object")) {
        return className;
      }

      var v = mirror.value();
      if (typeof v.constructor === "function") {
        // V8 reports classname for Buffer (nodejs) as Object, but we don't need ever resolve inferred name - constructor.name could be correct
        //noinspection JSUnresolvedVariable
        className = v.constructor.name;
        if (className == null || className.length == 0 || className === "Object") {
          var constructorFunction = mirror.constructorFunction()
          if (constructorFunction != null && constructorFunction.isFunction()) {
            className = DebuggerSupport.getFunctionName(constructorFunction);
            if (className === "Object") {
              className = null;
            }
          }
        }
      }
      return className;
    }

    private describeObject2(mirror: ObjectMirror, className: string, description: string, includeProperties: boolean, maxStringLength: number, objects: Array<ObjectDescriptor>): ObjectDescriptor {
      var objectDescriptor = this.describe(mirror, ValueType.OBJECT, description, -1, mirror, includeProperties, maxStringLength, objects)
      if (className != null) {
        objectDescriptor.className = className
      }
      return objectDescriptor
    }

    private describeArray(mirror: ArrayMirror, includeProperties: boolean, maxStringLength: number, objects: Array<ObjectDescriptor>): ObjectDescriptor {
      return this.describe(mirror, ValueType.ARRAY, mirror.className(), mirror.length(), mirror, includeProperties, maxStringLength, objects);
    }

    private describeFunction(mirror: FunctionMirror, includeProperties: boolean, maxStringLength: number, objects: Array<ObjectDescriptor>): void {
      var source = mirror.source();
      var endIndex = 0;
      while (endIndex < source.length && !isLineBreak(source, endIndex)) {
        endIndex++;
      }
      while (endIndex > 0 && isWhitespace(source, endIndex - 1)) {
        endIndex--;
      }
      source = source.substring(0, endIndex);

      this.describe(mirror, ValueType.FUNCTION, source, -1, mirror, includeProperties, maxStringLength, objects);
    }

    private static getFunctionName(functionMirror: FunctionMirror) {
      var functionName = functionMirror.name();
      if (functionName == null || functionName.length == 0) {
        functionName = functionMirror.inferredName();
      }
      return functionName == null || functionName.length == 0 ? null : functionName;
    }

    private describeString(mirror: StringMirror, maxStringLength: number, objects: Array<ObjectDescriptor>): ObjectDescriptor {
      var length = mirror.length();
      return this.describe(mirror, ValueType.STRING, truncateString(mirror, maxStringLength), length > MAX_STRING_LENGTH ? length : -1, null, false, maxStringLength, objects);
    }

    private mirrorAndDescribeString(value: any, maxStringLength: number, objects: Array<ObjectDescriptor>): ValueMirror {
      var mirror = new StringMirror(value);
      cacheManager.cacheMirror(mirror);
      this.describeString(mirror, maxStringLength, objects);
      return mirror;
    }

    private describe(mirror: ValueMirror, type: ValueType, description: string, length: number, objectMirror: ObjectMirror, includeProperties: boolean, maxStringLength: number, objects: Array<ObjectDescriptor>): ObjectDescriptor {
      var descriptor = new ObjectDescriptor(mirror.handle(), type, description, length)
      if (objectMirror != null && includeProperties) {
        descriptor.properties = this.describeProperties(objectMirror, objects, maxStringLength)
      }
      objects.push(descriptor)
      return descriptor
    }

    private describeProperties(hostMirror: ObjectMirror, objects: Array<ObjectDescriptor>, maxStringLength: number = MAX_STRING_LENGTH): Array<PropertyDescriptor> {
      let protoDescriptor: PropertyDescriptor
      // transitive objects doesn't have proto
      if (hostMirror.handle() < 0) {
        protoDescriptor = null
      }
      else {
        let proto = Object.getPrototypeOf(hostMirror.value_)
        if (proto == null) {
          protoDescriptor = null
        }
        else {
          protoDescriptor = new PropertyDescriptor("__proto__")
          protoDescriptor.objectId = this.getOrCreateMirror(proto, objects, false, maxStringLength, false, true).handle()
        }
      }

      const className = DebuggerSupport.getClassName(hostMirror)

      const isNewV8Vm = PropertyKind == null;
      let propertyNames = isTypedArray(className) ? typedArrayProperties : (isNewV8Vm ? hostMirror.propertyNames() : hostMirror.propertyNames(PropertyKind.Named, NaN))
      const hostIsArray = hostMirror.isArray()
      if (!isNewV8Vm && !hostIsArray) {
        propertyNames = propertyNames.concat(hostMirror.propertyNames(PropertyKind.Indexed, NaN))
      }

      let properties = new Array<PropertyDescriptor>(propertyNames.length  + (protoDescriptor == null ? 0 : 1))
      let itemIndex = 0

      for (let name of propertyNames) {
        if (typeof name === "symbol") {
          // ignore symbol for now
          continue
        }

        if (isNewV8Vm && hostIsArray && isIndexed(name)) {
          continue
        }

        properties[itemIndex++] = this.describePropertyByMirror(hostMirror, name, objects, maxStringLength)
      }

      if (protoDescriptor != null) {
        properties[itemIndex++] = protoDescriptor
      }

      properties.length = itemIndex
      return properties
    }

    private describePropertyByMirror(hostMirror: ObjectMirror, name: string, objects: Array<ObjectDescriptor>, maxStringLength: number) {
      var propertyMirror = hostMirror.property(name)
      if (propertyMirror.getter_ != null || propertyMirror.setter_ != null) {
        var propertyDescriptor = new PropertyDescriptor(name)
        if (propertyMirror.getter_ != null) {
          var mirror = this.getOrCreateMirror(propertyMirror.getter_, objects, false, maxStringLength)
          propertyDescriptor.getter = mirror.handle()
          if (!mirror.isFunction()) {
            throw new Error("Incorrect getter type: " + mirror.type)
          }
        }
        if (propertyMirror.setter_ != null) {
          var mirror = this.getOrCreateMirror(propertyMirror.setter_, objects, false, maxStringLength);
          propertyDescriptor.setter = mirror.handle()
          if (!mirror.isFunction()) {
            throw new Error("Incorrect setter type: " + mirror.type)
          }
        }
        return propertyDescriptor
      }
      else {
        let v = propertyMirror.value_
        if (v === undefined) {
          try {
            let rawObject = hostMirror.value_
            if (rawObject != null) {
              v = rawObject[name]
            }
          }
          catch (e) {
            console.error(e)
          }
        }
        return this.describeProperty(v, name, objects, maxStringLength)
      }
    }

    private describeProperty(value: any, name: any, objects: Array<ObjectDescriptor>, maxStringLength: number): PropertyDescriptor {
      var descriptor = new PropertyDescriptor(name)
      if (value === undefined) {
        descriptor.type = ValueType.UNDEFINED
      }
      else if (value === null) {
        descriptor.type = ValueType.NULL
      }
      else {
        var type = typeof value;
        if (type === 'boolean') {
          descriptor.type = ValueType.BOOLEAN
          descriptor.value = value
        }
        else if (type === 'number') {
          descriptor.type = ValueType.NUMBER
          descriptor.value = numberToJson(value)
        }
        else if (type === 'string' && (<string>value).length <= STRING_MIRROR_THRESHOLD) {
          descriptor.type = ValueType.STRING
          descriptor.value = value
        }
        else {
          descriptor.objectId = this.getOrCreateMirror(value, objects, false, maxStringLength, false, name === "prototype").handle()
        }
      }
      return descriptor
    }
  }

  function numberToJson(value: number): string|number {
    if (isNaN(value)) {
      return 'NaN'
    }
    else if (!isFinite(value)) {
      return value > 0 ? 'Infinity' : '-Infinity';
    }
    else {
      return value;
    }
  }
}

function isIndexed(name: string) {
  for (var i = 0; i < name.length; i++) {
    if (name[i] < '0' || '9' < name[i]) {
      return false
    }
  }
  return true
}

const typedArrayProperties = ["buffer", "byteLength", "byteOffset", "length"]

function isTypedArray(className: string) {
  return className === "Int8Array"
    || className === "Uint8Array"
    || className === "Uint8ClampedArray"
    || className === "Int16Array"
    || className === "Uint16Array"
    || className === "Int32Array"
    || className === "Uint32Array"
    || className === "Float32Array"
    || className === "Float64Array"
}