"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var logger_impl_1 = require("./logger-impl");
/**
 * Default tsserver implementation doesn't return response in most cases ("open", "close", etc.)
 * we want to override the behaviour and send empty-response holder
 */
var doneRequest = {
    responseRequired: true,
    response: "done"
};
function getSession(ts_impl, logger) {
    var TypeScriptSession = ts_impl.server.Session;
    var TypeScriptProjectService = ts_impl.server.ProjectService;
    var TypeScriptCommandNames = ts_impl.server.CommandNames;
    TypeScriptCommandNames.IDEChangeFiles = "ideChangeFiles";
    TypeScriptCommandNames.IDECompile = "ideCompile";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDECompletions = "ideCompletions";
    // var 
    var IDESession = (function (_super) {
        __extends(IDESession, _super);
        function IDESession(host, byteLength, hrtime, logger) {
            _super.call(this, host, byteLength, hrtime, logger);
            this._host = host;
            var handler = this.projectService.eventHandler;
            //reuse handler
            this.projectService = new IDEProjectService(host, logger, handler);
        }
        IDESession.prototype.send = function (msg) {
            var json = JSON.stringify(msg);
            this._host.write(json + "\n");
        };
        IDESession.prototype.executeCommand = function (request) {
            if (TypeScriptCommandNames.Open == request.command) {
                //use own implementation
                var openArgs = request.arguments;
                this.openClientFileExt(openArgs);
                return doneRequest;
            }
            else if (TypeScriptCommandNames.IDEChangeFiles == request.command) {
                var updateFilesArgs = request.arguments;
                return this.updateFilesExt(updateFilesArgs);
            }
            else if (TypeScriptCommandNames.IDECompile == request.command) {
                var fileArgs = request.arguments;
                return this.compileFileExt(fileArgs);
            }
            else if (TypeScriptCommandNames.Close == request.command) {
                _super.prototype.executeCommand.call(this, request);
                return doneRequest;
            }
            else if (TypeScriptCommandNames.IDEGetErrors == request.command) {
                var args = request.arguments;
                return this.getDiagnosticsExt(args.files);
            }
            else if (TypeScriptCommandNames.IDECompletions == request.command) {
                var result = _super.prototype.executeCommand.call(this, {
                    command: TypeScriptCommandNames.Completions,
                    arguments: request.arguments,
                    seq: request.seq,
                    type: request.type
                });
                var args = request.arguments;
                var response = result.response;
                return {
                    response: this.getIDECompletions(args, response),
                    responseRequired: true
                };
            }
            return _super.prototype.executeCommand.call(this, request);
        };
        IDESession.prototype.updateFilesExt = function (args) {
            var updated = false;
            var files = args.files;
            for (var fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    var content = files[fileName];
                    if (content) {
                        this.changeFileExt(fileName, content);
                        updated = true;
                    }
                }
            }
            if (args.filesToReloadContentFromDisk) {
                for (var _i = 0, _a = args.filesToReloadContentFromDisk; _i < _a.length; _i++) {
                    var fileName = _a[_i];
                    if (!fileName) {
                        continue;
                    }
                    var file = ts_impl.normalizePath(fileName);
                    this.projectService.closeClientFile(file);
                    logger_impl_1.serverLogger("Reload file from disk " + file);
                    updated = true;
                }
            }
            if (updated) {
                this.updateProjectStructureExt();
            }
            return doneRequest;
        };
        IDESession.prototype.updateProjectStructureExt = function () {
            var _this = this;
            var mySeq = this.getChangeSeq();
            var matchSeq = function (n) { return n === mySeq; };
            setTimeout(function () {
                if (matchSeq(_this.getChangeSeq())) {
                    _this.projectService.updateProjectStructure();
                }
            }, 1500);
        };
        IDESession.prototype.getChangeSeq = function () {
            var anyThis = this;
            var superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }
            return this._mySeq;
        };
        IDESession.prototype.openClientFileExt = function (openArgs) {
            var fileName = openArgs.file;
            var fileContent = openArgs.fileContent;
            var configFile = openArgs.tsConfig;
            var file = ts_impl.normalizePath(fileName);
            return this.projectService.openClientFileExt(file, fileContent, configFile);
        };
        IDESession.prototype.changeFileExt = function (fileName, content, tsconfig) {
            var file = ts_impl.normalizePath(fileName);
            var project = this.projectService.getProjectForFile(file);
            if (project) {
                var compilerService = project.compilerService;
                var scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    logger_impl_1.serverLogger("Reload content from text " + file);
                }
                else {
                    logger_impl_1.serverLogger("ScriptInfo is null " + file);
                }
            }
            else {
                logger_impl_1.serverLogger("Cannot find project for " + file);
                this.openClientFileExt({
                    file: fileName,
                    fileContent: content,
                    tsConfig: tsconfig
                });
            }
        };
        IDESession.prototype.compileFileExt = function (req) {
            var requestedFile = req.file;
            var project = this.projectService.getProjectForFile(requestedFile);
            if (project != null) {
                project.program.emit(); //emit all files?
                return doneRequest;
            }
        };
        IDESession.prototype.logError = function (err, cmd) {
            var typedErr = err;
            logger_impl_1.serverLogger("Error processing message: " + err.message + " " + typedErr.stack);
            _super.prototype.logError.call(this, err, cmd);
        };
        IDESession.prototype.getIDECompletions = function (req, entries) {
            if (!entries) {
                return entries;
            }
            var file = ts_impl.normalizePath(req.file);
            var project = this.projectService.getProjectForFile(file);
            var compilerService = project.compilerService;
            var position = compilerService.host.lineOffsetToPosition(file, req.line, req.offset);
            var count = 0;
            return entries.reduce(function (accum, entry) {
                if (count++ > 20) {
                    accum.push(entry);
                }
                else {
                    var details = compilerService.languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        };
        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         *
         */
        IDESession.prototype.getDiagnosticsExt = function (fileNames) {
            var _this = this;
            var checkList = fileNames.reduce(function (accum, fileName) {
                fileName = ts_impl.normalizePath(fileName);
                var project = _this.projectService.getProjectForFile(fileName);
                if (project) {
                    accum.push({ fileName: fileName, project: project });
                }
                return accum;
            }, []);
            var result = [];
            if (checkList.length > 0) {
                var _loop_1 = function(checkSpec) {
                    var file = checkSpec.fileName;
                    var project = checkSpec.project;
                    if (project.getSourceFileFromName(file, true)) {
                        var diagnostics = [];
                        var syntacticDiagnostics = project.compilerService.languageService.getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics) {
                            var bakedDiagnostics = syntacticDiagnostics.map(function (diag) { return formatDiagnostics(file, checkSpec.project, diag); });
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }
                        var semanticDiagnostics = project.compilerService.languageService.getSemanticDiagnostics(file);
                        if (semanticDiagnostics) {
                            var bakedSemanticDiagnostics = semanticDiagnostics.map(function (diag) { return formatDiagnostics(file, checkSpec.project, diag); });
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);
                        }
                        result.push({
                            file: file,
                            diagnostics: diagnostics
                        });
                    }
                };
                for (var _i = 0, checkList_1 = checkList; _i < checkList_1.length; _i++) {
                    var checkSpec = checkList_1[_i];
                    _loop_1(checkSpec);
                }
            }
            return { response: result, responseRequired: true };
        };
        return IDESession;
    }(TypeScriptSession));
    var IDEProjectService = (function (_super) {
        __extends(IDEProjectService, _super);
        function IDEProjectService(host, psLogger, eventHandler) {
            _super.call(this, host, psLogger, eventHandler);
        }
        IDEProjectService.prototype.openClientFileExt = function (fileName, fileContent, configFileName) {
            if (configFileName) {
                logger_impl_1.serverLogger("Open for specified tsconfig");
                this.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            }
            else {
                logger_impl_1.serverLogger("Try to find tsconfig");
                this.openOrUpdateConfiguredProjectForFile(fileName);
            }
            var info = this.openFile(fileName, /*openedByClient*/ true, fileContent);
            this.addOpenFile(info);
            return info;
        };
        IDEProjectService.prototype.configFileToProjectOptions = function (configFilename) {
            function getBaseFileName(path) {
                if (path === undefined) {
                    return undefined;
                }
                var i = path.lastIndexOf(ts_impl.directorySeparator);
                return i < 0 ? path : path.substring(i + 1);
            }
            var configFileToProjectOptions = _super.prototype.configFileToProjectOptions.call(this, configFilename);
            if (configFileToProjectOptions && configFileToProjectOptions.projectOptions) {
                var projectOptions = configFileToProjectOptions.projectOptions;
                var files = projectOptions.files;
                if (files) {
                    var compilerOptions = projectOptions.compilerOptions;
                    var extensions = ts_impl.getSupportedExtensions(compilerOptions);
                    var newFiles = [];
                    l: for (var _i = 0, files_1 = files; _i < files_1.length; _i++) {
                        var file = files_1[_i];
                        var fileName = getBaseFileName(file);
                        for (var _a = 0, extensions_1 = extensions; _a < extensions_1.length; _a++) {
                            var extension = extensions_1[_a];
                            if (fileName.lastIndexOf(extension) > 0) {
                                newFiles.push(file);
                                continue l;
                            }
                        }
                        for (var _b = 0, extensions_2 = extensions; _b < extensions_2.length; _b++) {
                            var extension = extensions_2[_b];
                            if (this.host.fileExists(file + extension)) {
                                newFiles.push(file + extension);
                                continue l;
                            }
                        }
                        newFiles.push(file);
                    }
                    var newOptions = {
                        succeeded: configFileToProjectOptions.succeeded,
                        projectOptions: {
                            compilerOptions: compilerOptions,
                            files: newFiles
                        }
                    };
                    if (configFileToProjectOptions.error) {
                        newOptions.error = configFileToProjectOptions.error;
                    }
                    return newOptions;
                }
            }
            return configFileToProjectOptions;
        };
        return IDEProjectService;
    }(TypeScriptProjectService));
    /**
     * copy formatDiag method (but we use 'TS' prefix)
     */
    function formatDiagnostics(fileName, project, diag) {
        return {
            start: project.compilerService.host.positionToLineOffset(fileName, diag.start),
            end: project.compilerService.host.positionToLineOffset(fileName, diag.start + diag.length),
            text: "TS" + diag.code + ":" + ts_impl.flattenDiagnosticMessageText(diag.messageText, "\n")
        };
    }
    return new IDESession(ts_impl.sys, Buffer.byteLength, process.hrtime, logger);
}
exports.getSession = getSession;
