/**
 * Entry point for the TypeScript plugin
 */

import {getService} from "./service-loader";
import {getSession} from "./ts-session";
import {createLoggerFromEnv} from "./logger-impl";
import {getPathProcessor, PathProcessor} from "./out-path-process";

class TypeScriptLanguagePlugin implements LanguagePlugin {

    private _session: ts.server.Session;
    readyMessage: {version: string, supportedErrorCodes?:(string|number)[]};

    constructor(state: TypeScriptPluginState) {

        let serviceInfo = getService(state.serverFolderPath);
        let serviceContext: {
            ts: typeof ts;
        } = serviceInfo.context;


        let serverFilePath = serviceInfo.serverFilePath;
        let ts_impl: typeof ts = serviceContext.ts;
        let loggerImpl = createLoggerFromEnv(ts_impl);
        overrideSysDefaults(ts_impl, serverFilePath);

        let commonDefaultCommandLine: ts.ParsedCommandLine = state.hasManualParams && state.commandLineArguments ?
            ts_impl.parseCommandLine(state.commandLineArguments) :
            null;

        let commonDefaultOptions: ts.CompilerOptions = null;
        if (commonDefaultCommandLine != null) {
            commonDefaultOptions = commonDefaultCommandLine.options;
        }

        if (commonDefaultOptions === null && state.hasManualParams) {
            commonDefaultOptions = {};
        }


        let pathProcessor: PathProcessor = null;
        if (state.hasManualParams && state.outPath) {
            pathProcessor = getPathProcessor(ts_impl, state);
        }

        let mainFile: string = null;
        if (state.hasManualParams && state.mainFilePath) {
            mainFile = state.mainFilePath;
        }

        this._session = getSession(ts_impl, loggerImpl, commonDefaultOptions, pathProcessor, mainFile);



        this.readyMessage = {version: ts_impl.version};

        if ((<any>ts_impl).getSupportedCodeFixes) {
            let codes:string[] = (<any>ts_impl).getSupportedCodeFixes();
            if (codes && codes.length >0)  {
                this.readyMessage.supportedErrorCodes = codes;
            }
        }
    }

    onMessage(p: string) {
        this._session.onMessage(p);
    }


}

class TypeScriptLanguagePluginFactory implements LanguagePluginFactory {
    create(state: PluginState): {languagePlugin: LanguagePlugin, readyMessage?:any } {
        let typeScriptLanguagePlugin = new TypeScriptLanguagePlugin(<TypeScriptPluginState>state);
        return {
            languagePlugin: typeScriptLanguagePlugin,
            readyMessage: typeScriptLanguagePlugin.readyMessage
        };
    }
}


function overrideSysDefaults(ts_impl: typeof ts, serverFolderPath: string) {
    const pending: string[] = [];
    let canWrite = true;

    function writeMessage(s: string) {
        if (!canWrite) {
            pending.push(s);
        }
        else {
            canWrite = false;
            process.stdout.write(new Buffer(s, "utf8"), setCanWriteFlagAndWriteMessageIfNecessary);
        }
    }

    function setCanWriteFlagAndWriteMessageIfNecessary() {
        canWrite = true;
        if (pending.length) {
            writeMessage(pending.shift());
        }
    }

    // Override sys.write because fs.writeSync is not reliable on Node 4
    ts_impl.sys.write = (s: string) => writeMessage(s);

    //ts 2.0 compatibility
    (<any>ts_impl.sys).setTimeout = setTimeout;
    (<any>ts_impl.sys).clearTimeout = clearTimeout;

    //ts2.0.5 & 2.1
    (<any>ts_impl.sys).setImmediate = setImmediate;
    (<any>ts_impl.sys).clearImmediate = clearImmediate;
    if (typeof global !== "undefined" && global.gc) {
        (<any>ts_impl.sys).gc = () => global.gc();
    }

    ts_impl.sys.getExecutingFilePath = () => {
        return serverFolderPath;
    }

    const pollingWatchedFileSet = createPollingWatchedFileSet(ts_impl);

    ts_impl.sys.watchFile = (fileName, callback) => {
        const watchedFile = pollingWatchedFileSet.addFile(fileName, callback);
        return {
            close: () => pollingWatchedFileSet.removeFile(watchedFile)
        };
    };
}

//copy ts-server implementation
function createPollingWatchedFileSet(ts_impl:any, interval = 2500, chunkSize = 30) {
    const fs: {
        openSync(path: string, options: string): number;
        close(fd: number): void;
        writeSync(fd: number, buffer: Buffer, offset: number, length: number, position?: number): number;
        writeSync(fd: number, data: any, position?: number, enconding?: string): number;
        statSync(path: string):any;
        stat(path: string, callback?: (err: NodeJS.ErrnoException, stats:any) => any): void;
    } = require("fs");

    const watchedFiles:any[] = [];
    let nextFileToCheck = 0;
    let watchTimer: any;

    function getModifiedTime(fileName: string): Date {
        return fs.statSync(fileName).mtime;
    }

    function poll(checkedIndex: number) {
        const watchedFile = watchedFiles[checkedIndex];
        if (!watchedFile) {
            return;
        }

        fs.stat(watchedFile.fileName, (err: any, stats: any) => {
            if (err) {
                watchedFile.callback(watchedFile.fileName);
            }
            else if (watchedFile.mtime.getTime() !== stats.mtime.getTime()) {
                watchedFile.mtime = getModifiedTime(watchedFile.fileName);
                watchedFile.callback(watchedFile.fileName, watchedFile.mtime.getTime() === 0);
            }
        });
    }

    // this implementation uses polling and
    // stat due to inconsistencies of fs.watch
    // and efficiency of stat on modern filesystems
    function startWatchTimer() {
        watchTimer = setInterval(() => {
            let count = 0;
            let nextToCheck = nextFileToCheck;
            let firstCheck = -1;
            while ((count < chunkSize) && (nextToCheck !== firstCheck)) {
                poll(nextToCheck);
                if (firstCheck < 0) {
                    firstCheck = nextToCheck;
                }
                nextToCheck++;
                if (nextToCheck === watchedFiles.length) {
                    nextToCheck = 0;
                }
                count++;
            }
            nextFileToCheck = nextToCheck;
        }, interval);
    }

    function addFile(fileName: string, callback: any): any {
        const file: any = {
            fileName,
            callback,
            mtime: getModifiedTime(fileName)
        };

        watchedFiles.push(file);
        if (watchedFiles.length === 1) {
            startWatchTimer();
        }
        return file;
    }

    function removeFile(file: any) {
        unorderedRemoveItem(watchedFiles, file);
    }

    function unorderedRemoveItem<T>(array: T[], item: T): void {
        unorderedRemoveFirstItemWhere(array, element => element === item);
    }

    function unorderedRemoveFirstItemWhere<T>(array: T[], predicate: (element: T) => boolean): void {
        for (let i = 0; i < array.length; i++) {
            if (predicate(array[i])) {
                unorderedRemoveItemAt(array, i);
                break;
            }
        }
    }

    function unorderedRemoveItemAt<T>(array: T[], index: number): void {
        // Fill in the "hole" left at `index`.
        array[index] = array[array.length - 1];
        array.pop();
    }

    return {
        getModifiedTime: getModifiedTime,
        poll: poll,
        startWatchTimer: startWatchTimer,
        addFile: addFile,
        removeFile: removeFile
    };
}


let typescriptLanguagePluginFactory: TypeScriptLanguagePluginFactory = new TypeScriptLanguagePluginFactory();

export {typescriptLanguagePluginFactory}



