/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.stereotype;

import com.intellij.jam.JamPomTarget;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.reflect.*;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.pom.PomTarget;
import com.intellij.pom.references.PomService;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiModifierListOwner;
import com.intellij.psi.PsiTarget;
import com.intellij.psi.targets.AliasingPsiTarget;
import com.intellij.psi.targets.AliasingPsiTargetUtil;
import com.intellij.spring.model.jam.JamPsiClassSpringBean;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.jam.javaConfig.SpringJavaBean;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.SmartHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public abstract class SpringStereotypeElement extends JamPsiClassSpringBean {
  private static final Map<String, JamAnnotationMeta> annotationMetaMap = ContainerUtil.newHashMap();
  private JamAnnotationMeta myMeta;

  private static final JamStringAttributeMeta.Single<String> NAME_VALUE_META = JamAttributeMeta.singleString("value");

  public static final JamMethodMeta<ContextJavaBean> BEANS_METHOD_META =
    new JamMethodMeta<>(null, ContextJavaBean.class, ContextJavaBean.BEAN_JAM_KEY);

  static {
    BEANS_METHOD_META.addPomTargetProducer((contextJavaBean, pomTargetConsumer) -> {
      for (PomTarget pomTarget : contextJavaBean.getPomTargets()) {
        pomTargetConsumer.consume(pomTarget);
      }
    });
  }

  public List<? extends SpringJavaBean> getBeans() {
    final List<SpringJavaBean> beans = new ArrayList<>();
    final JamService jamService = JamService.getJamService(getPsiManager().getProject());

    JamCommonUtil.processSuperClassList(getPsiElement(), new SmartHashSet<>(), psiClass -> {
      beans.addAll(jamService.getAnnotatedMembersList(psiClass, ContextJavaBean.BEAN_JAM_KEY, false, true, false, false));

      return true;
    });

    return beans;
  }

  protected SpringStereotypeElement(@Nullable String anno) {
    if (anno != null) myMeta = getMeta(anno);
  }

  @NotNull
  private static synchronized JamAnnotationMeta getMeta(@NotNull String anno) {
    JamAnnotationMeta meta = annotationMetaMap.get(anno);
    if (meta == null) {
      meta = new JamAnnotationMeta(anno);
      annotationMetaMap.put(anno, meta);
    }
    return meta;
  }

  public static <Psi extends PsiModifierListOwner, Jam extends SpringStereotypeElement> void addPomTargetProducer(JamMemberMeta<Psi, Jam> classMeta) {
    classMeta.addPomTargetProducer((stereotypeElement, consumer) -> consumer.consume(stereotypeElement.getPsiTarget()));
  }

  public PsiTarget getPsiTarget() {
    final JamStringAttributeElement<String> namedAttributeValue = getNamedStringAttributeElement();
    if (namedAttributeValue == null || StringUtil.isEmptyOrSpaces(namedAttributeValue.getStringValue())) {
      return getAliasingPsiTarget();
    }

    return new JamPomTarget(this, namedAttributeValue);
  }

  private PsiTarget getAliasingPsiTarget() {
    return new AliasingPsiTarget(getPsiElement()) {
      @Override
      public String getNameAlias(@Nullable String delegatePsiTargetName) {
        return StringUtil.decapitalize(delegatePsiTargetName);
      }

      @NotNull
      @Override
      public AliasingPsiTarget setAliasName(@NotNull String newAliasName) {
        AliasingPsiTargetUtil.renameTargets(this, StringUtil.capitalize(newAliasName));

        return super.setAliasName(newAliasName);
      }
    };
  }

  @Nullable
  private JamStringAttributeElement<String> getNamedStringAttributeElement() {
    return myMeta == null ? null : myMeta.getAttribute(getPsiElement(), NAME_VALUE_META);
  }

  public String getBeanName() {
    assert isValid();
    final String definedName = getAnnotationDefinedBeanName();

    if (!StringUtil.isEmptyOrSpaces(definedName)) return definedName;

    return super.getBeanName();
  }

  @Nullable
  private String getAnnotationDefinedBeanName() {
    final JamStringAttributeElement<String> namedStringAttributeElement = getNamedStringAttributeElement();

    return namedStringAttributeElement == null ? null : namedStringAttributeElement.getStringValue();
  }

  @Override
  @NotNull
  public PsiElement getIdentifyingPsiElement() {
    return PomService.convertToPsi(getPsiManager().getProject(), getPsiTarget());
  }

  @Override
  public String toString() {
    final String beanName = getBeanName();
    return beanName == null ? "" : beanName;
  }
}
