/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.values;

import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.psi.PsiEllipsisType;
import com.intellij.psi.PsiLanguageInjectionHost;
import com.intellij.psi.PsiType;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.psi.xml.XmlElement;
import com.intellij.spring.model.xml.beans.TypeHolder;
import com.intellij.spring.model.xml.beans.TypeHolderUtil;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.WrappingConverter;
import com.intellij.util.xml.converters.values.GenericDomValueConvertersRegistry;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class PropertyValueConverter extends WrappingConverter {

  public Converter getConverter(@NotNull final GenericDomValue domElement) {
    final List<Converter> converters = getConverters(domElement);
    return converters.isEmpty() ? null : converters.get(0);
  }

  @Override
  @NotNull
  public List<Converter> getConverters(@NotNull final GenericDomValue element) {
    XmlElement xmlElement = element.getXmlElement();
    if (xmlElement instanceof XmlAttribute) {
      PsiLanguageInjectionHost host = (PsiLanguageInjectionHost)((XmlAttribute)xmlElement).getValueElement();
      if (host == null || InjectedLanguageManager.getInstance(host.getProject()).getInjectedPsiFiles(host) != null) {
        return Collections.emptyList();
      }
    }
    final GenericDomValueConvertersRegistry registry = SpringValueConvertersRegistry.getInstance();

    final List<PsiType> types = getValueTypes(element);
    if (types.isEmpty()) {
      final Converter converter = registry.getConverter(element, null);
      return ContainerUtil.createMaybeSingletonList(converter);
    }

    final List<Converter> converters = new SmartList<>();
    for (PsiType type : types) {
      final Converter converter = registry.getConverter(element, type instanceof PsiEllipsisType ? ((PsiEllipsisType)type).getComponentType() : type);
      if (converter != null) {
        converters.add(converter);
      }
      else {
        return Collections.emptyList();
      }
    }
    return converters;
  }

  @NotNull
  public List<PsiType> getValueTypes(final GenericDomValue element) {
    if (element instanceof TypeHolder) {
      final List<PsiType> psiTypes = TypeHolderUtil.getRequiredTypes(((TypeHolder)element));
      if (!psiTypes.isEmpty()) {
        return psiTypes;
      }
    }

    final DomElement parent = element.getParent();
    return parent instanceof TypeHolder ? TypeHolderUtil.getRequiredTypes(((TypeHolder)parent)) : Collections.emptyList();
  }
}
