/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.references;

import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.util.NullableFunction;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;

public class SpringBeanReferenceProvider extends PsiReferenceProvider {
  @NotNull private final NullableFunction<PsiElement, PsiClass> myRequiredTypeFun;

  public SpringBeanReferenceProvider() {
    myRequiredTypeFun = member -> null;
  }

  public SpringBeanReferenceProvider(@NotNull NullableFunction<PsiElement, PsiClass> requiredTypeFun) {
    myRequiredTypeFun = requiredTypeFun;
  }

  @NotNull
  @Override
  public PsiReference[] getReferencesByElement(@NotNull final PsiElement element, @NotNull final ProcessingContext context) {
    String beanName = (String)((PsiLiteralExpression)element).getValue();
    boolean isFactoryBeanRef = false;
    if (StringUtil.isEmptyOrSpaces(beanName)) return PsiReference.EMPTY_ARRAY;
    if (beanName.startsWith("&")) {
      beanName = beanName.substring(1);
      isFactoryBeanRef = true;
    }
    TextRange range = TextRange.from(element.getText().indexOf(beanName), beanName.length());
    return new PsiReference[]{new SpringBeanReference(element, range, myRequiredTypeFun.fun(element), isFactoryBeanRef) {
      @Override
      public boolean isSoft() {
        return true;
      }
    }
    };
  }
}
