/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.sql.psi;

import com.intellij.database.model.DasNamespace;
import com.intellij.database.psi.DbDataSource;
import com.intellij.database.psi.DbElement;
import com.intellij.database.script.ScriptModel;
import com.intellij.database.util.TextWithRanges;
import com.intellij.lang.Language;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NotNullLazyKey;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiCodeFragment;
import com.intellij.psi.PsiFile;
import com.intellij.sql.dialects.SqlLanguageDialect;
import com.intellij.sql.script.SqlReader;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author Gregory.Shrago
 */
public abstract class SqlPsiFacade {

  private static final NotNullLazyKey<SqlPsiFacade, Project> INSTANCE_KEY = ServiceManager.createLazyKey(SqlPsiFacade.class);

  @NotNull
  public static SqlPsiFacade getInstance(Project project) {
    return INSTANCE_KEY.getValue(project);
  }

  @NotNull
  public abstract ScriptModel<?> createScriptModel(@NotNull PsiFile file);

  @NotNull
  public abstract SqlReader createSqlReader();

  @NotNull
  public abstract CharSequence format(@NotNull Project project,
                                      @NotNull SqlLanguageDialect dialect,
                                      @NotNull TextWithRanges text);

  public abstract void format(@NotNull Project project,
                              @NotNull SqlLanguageDialect dialect,
                              @NotNull Document document);

  @NotNull
  public abstract PsiCodeFragment createTableReferenceFragment(@NotNull Language dialect,
                                                               @Nullable DbDataSource dataSourceElement,
                                                               @Nullable DbElement schemaElement,
                                                               @NotNull String text);

  @NotNull
  public abstract PsiCodeFragment createTypeElementFragment(@NotNull Language dialect,
                                                            @Nullable DbDataSource dataSourceElement,
                                                            @Nullable List<DasNamespace> searchPath,
                                                            @NotNull String text);

  @NotNull
  public abstract PsiCodeFragment createEvaluableExpressionFragment(@NotNull Language dialect,
                                                                    @Nullable DbDataSource dataSourceElement,
                                                                    @Nullable List<DasNamespace> searchPath,
                                                                    @NotNull String text);

  @NotNull
  public abstract PsiCodeFragment createExpressionFragment(@NotNull Language dialect,
                                                           @Nullable DbDataSource dataSourceElement,
                                                           @Nullable List<DasNamespace> searchPath,
                                                           @NotNull String text);

  @NotNull
  public abstract PsiCodeFragment createExpressionFragment(@NotNull Language dialect,
                                                           @Nullable DbDataSource dataSourceElement,
                                                           @Nullable List<DasNamespace> searchPath,
                                                           @NotNull String text,
                                                           @NotNull String context);

  @NotNull
  public abstract SqlLanguageDialect getDefaultDialect();

  public abstract void setDialectMapping(@NotNull VirtualFile file, @NotNull SqlLanguageDialect dialect);
  @NotNull
  public abstract SqlLanguageDialect getDialectMapping(@NotNull VirtualFile file);
}