/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.facet;

import com.intellij.facet.*;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Key;
import com.intellij.spring.facet.beans.CustomSetting;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringFacet extends Facet<SpringFacetConfiguration> {

  public final static FacetTypeId<SpringFacet> FACET_TYPE_ID = new FacetTypeId<>("spring");

  protected SpringFacet(FacetType facetType, Module module, String name, SpringFacetConfiguration configuration, Facet underlyingFacet) {
    super(facetType, module, name, configuration, underlyingFacet);
  }

  @Nullable
  public static SpringFacet getInstance(@NotNull Module module) {
    if (module.isDisposed()) return null;

    return FacetManager.getInstance(module).getFacetByType(FACET_TYPE_ID);
  }

  public static FacetType<SpringFacet, SpringFacetConfiguration> getSpringFacetType() {
    return FacetTypeRegistry.getInstance().findFacetType(FACET_TYPE_ID);
  }

  /**
   * Returns all user-configured filesets.
   *
   * @return Filesets.
   * @see SpringFileSetService#getAllSets(SpringFacet)
   */
  public abstract Set<SpringFileSet> getFileSets();

  /**
   * Adds new empty fileset.
   *
   * @param id   Unique non-empty ID.
   * @param name Unique non-empty name.
   * @return Empty fileset.
   * @see SpringFileSetService#getUniqueId(Set)
   * @see SpringFileSetService#getUniqueName(String, Set)
   */
  public abstract SpringFileSet addFileSet(@NonNls @NotNull String id, @NotNull String name);

  public abstract SpringFileSet addFileSet(@NotNull SpringFileSet fileSet);

  public abstract void removeFileSets();

  /**
   * @deprecated Use {@link #findSetting(Key)} instead. To remove in 2017.3.
   */
  @Deprecated
  @Nullable
  public abstract CustomSetting findSetting(@NotNull String settingName);

  /**
   * @param settingsKey Key of settings.
   * @param <S>         Settings class. Usually {@link CustomSetting.BOOLEAN} or {@link CustomSetting.STRING}.
   * @return Setting or {@code null} if not set.
   * @since 2017.2
   */
  @Nullable
  public abstract <S extends CustomSetting> S findSetting(@NotNull Key<S> settingsKey);
}
