/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.module.Module;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.utils.SpringCommonUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SpringImplicitBean extends CustomSpringComponent implements SpringImplicitBeanMarker {

  @Nullable
  public static SpringImplicitBean create(@Nullable Module module, @Nullable String providerName, @NotNull String className, @NotNull String beanName) {
    if (module == null) return null;
    final PsiClass libraryClass = SpringCommonUtils.findLibraryClass(module, className);
    if (libraryClass == null) return null;
    return new SpringImplicitBean(providerName == null? "" : providerName, libraryClass, beanName);
  }

  @Nullable
  public static SpringImplicitBean create(@Nullable String providerName, @Nullable PsiClass psiClass, @NotNull String beanName) {
    if (psiClass == null) return null;
    return new SpringImplicitBean(providerName == null? "" : providerName, psiClass, beanName);
  }


  @NotNull
  private final String myBeanName;
  @NotNull
  private final String myProviderName;

  public SpringImplicitBean(@NotNull String providerName,
                            @NotNull PsiClass psiClass,
                            @NotNull String beanName) {
    super(psiClass);
    myProviderName = providerName;
    myBeanName = beanName;
  }

  @NotNull
  @Override
  public String getBeanName() {
    return myBeanName;
  }

  @NotNull
  @Override
  public String getProviderName() {
    return myProviderName;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    if (!super.equals(o)) return false;

    SpringImplicitBean bean = (SpringImplicitBean)o;

    if (!myBeanName.equals(bean.myBeanName)) return false;
    if (!myProviderName.equals(bean.myProviderName)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = super.hashCode();
    result = 31 * result + myBeanName.hashCode();
    result = 31 * result + myProviderName.hashCode();
    return result;
  }
}
