/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiSubstitutor;
import com.intellij.psi.PsiType;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.BitUtil;
import com.intellij.util.Processor;
import com.intellij.util.containers.ConcurrentFactoryMap;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.FactoryMap;
import org.jetbrains.annotations.NotNull;

import java.util.Map;

/**
 * Defines search parameters for querying Spring model for matching bean(s).
 *
 * @see com.intellij.spring.model.utils.SpringModelSearchers
 * @see com.intellij.spring.CommonSpringModel#processByClass(BeanClass, Processor)
 * @see com.intellij.spring.CommonSpringModel#processByName(BeanName, Processor)
 */
public abstract class SpringModelSearchParameters {

  protected SpringModelSearchParameters() {
  }

  /**
   * Returns whether given search parameters are valid to start a search.
   *
   * @return {@code false} if search cannot be performed.
   */
  public abstract boolean canSearch();

  public static BeanClass byClass(@NotNull PsiClass psiClass) {
    return new BeanClass(psiClass);
  }

  /**
   * @since 2016.2
   */
  public static BeanClass byType(@NotNull PsiType psiType) {
    return new BeanClass(psiType);
  }

  public static class BeanClass extends SpringModelSearchParameters {
    private final PsiType myType;

    private byte myOptions;

    private static final byte CAN_SEARCH = 1;
    private static final byte WITH_INHERITORS = 2;
    private static final byte EFFECTIVE_BEAN_TYPES = 4;

    private BeanClass(@NotNull PsiClass psiClass) {
      myType = JavaPsiFacade.getElementFactory(psiClass.getProject()).createType(psiClass, PsiSubstitutor.EMPTY);
      myOptions = BitUtil.set(myOptions, CAN_SEARCH, SpringCommonUtils.isSpringBeanCandidateClass(psiClass));
    }

    private BeanClass(@NotNull PsiType psiType) {
      myType = psiType;
      final PsiClass searchClass = PsiTypesUtil.getPsiClass(psiType);
      myOptions = BitUtil.set(myOptions, CAN_SEARCH, searchClass == null || SpringCommonUtils.isSpringBeanCandidateClass(searchClass));
    }

    @Override
    public boolean canSearch() {
      return BitUtil.isSet(myOptions, CAN_SEARCH);
    }

    public BeanClass withInheritors() {
      myOptions = BitUtil.set(myOptions, WITH_INHERITORS, true);
      return this;
    }

    public BeanClass effectiveBeanTypes() {
      myOptions = BitUtil.set(myOptions, EFFECTIVE_BEAN_TYPES, true);
      return this;
    }

    @NotNull
    public PsiType getSearchType() {
      return myType;
    }

    public boolean isWithInheritors() {
      return BitUtil.isSet(myOptions, WITH_INHERITORS);
    }

    public boolean isEffectiveBeanTypes() {
      return BitUtil.isSet(myOptions, EFFECTIVE_BEAN_TYPES);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanClass)) return false;

      BeanClass beanClass = (BeanClass)o;

      if (myOptions != beanClass.myOptions) return false;
      if (!myType.isValid() || !beanClass.getSearchType().isValid()) return false;
      if (!myType.equals(beanClass.getSearchType())) return false;
      return true;
    }

    @Override
    public int hashCode() {
      int result = myType.hashCode();
      result = 31 * result + myOptions;
      return result;
    }
  }

  @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
  private static final FactoryMap<String, BeanName> ourCachedBeanNames = new ConcurrentFactoryMap<String, BeanName>() {

    protected Map<String, BeanName> createMap() {
      return ContainerUtil.createConcurrentWeakKeyWeakValueMap();
    }

    @Override
    protected BeanName create(String key) {
      return new BeanName(key);
    }
  };

  public static BeanName byName(@NotNull String beanName) {
    return ourCachedBeanNames.get(beanName);
  }

  public static class BeanName extends SpringModelSearchParameters {

    @NotNull
    private final String myBeanName;

    private BeanName(@NotNull String beanName) {
      myBeanName = StringUtil.isNotEmpty(beanName) ? beanName : "";
    }

    @Override
    public boolean canSearch() {
      return !myBeanName.isEmpty();
    }

    @NotNull
    public String getBeanName() {
      return myBeanName;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanName)) return false;

      BeanName name = (BeanName)o;

      if (!myBeanName.equals(name.myBeanName)) return false;

      return true;
    }

    @Override
    public int hashCode() {
      return myBeanName.hashCode();
    }
  }
}
