/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.highlighting.jam;

import com.intellij.codeInspection.BaseJavaLocalInspectionTool;
import com.intellij.codeInspection.ProblemHighlightType;
import com.intellij.codeInspection.ProblemsHolder;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiAnnotationMemberValue;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.utils.SpringCommonUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class SpringBeanPointerResolveInspection extends BaseJavaLocalInspectionTool {
  protected static boolean isPlainJavaFileInSpringModule(@NotNull PsiElement psiElement) {
    if (!JamCommonUtil.isPlainJavaFile(psiElement.getContainingFile())) {
      return false;
    }
    final Module module = ModuleUtilCore.findModuleForPsiElement(psiElement);
    return SpringCommonUtils.hasSpringFacet(module);
  }

  public static void checkBeanPointerResolve(ProblemsHolder holder, JamStringAttributeElement<SpringBeanPointer> element, @Nullable String beanType) {
    final String beanName = element.getStringValue();
    if (beanName != null) {
      final PsiAnnotationMemberValue memberValue = element.getPsiElement();
      if (memberValue != null) {
        final SpringBeanPointer value = element.getValue();
        if (value == null) {
          holder.registerProblem(memberValue, SpringApiBundle.message("model.bean.error.message", beanName),
                                 ProblemHighlightType.GENERIC_ERROR_OR_WARNING);
        }
        else {
          if (StringUtil.isNotEmpty(beanType)) {
            final PsiClass psiClass = value.getBeanClass();
            if (psiClass != null && !InheritanceUtil.isInheritor(psiClass, beanType)) {
              holder.registerProblem(memberValue,
                                     SpringApiBundle.message("bean.must.be.of.type", beanType),
                                     ProblemHighlightType.GENERIC_ERROR_OR_WARNING);
            }
          }
        }
      }
    }
  }
}

