package aQute.service.library;

import java.util.regex.*;

import aQute.bnd.version.*;
import aQute.service.library.Library.Revision;
import aQute.service.library.Library.RevisionRef;

public class LibraryVersion implements Comparable<LibraryVersion> {

	final long		major, minor, micro;
	final String	qualifier;
	final String	input;

	static Pattern	fuzzyVersion	= Pattern.compile("(\\d+)(\\.(\\d+)(\\.(\\d+))?)?([^a-zA-Z0-9](.*))?",
											Pattern.DOTALL);
	static Pattern	fuzzyModifier	= Pattern.compile("(\\d+[.-])*(.*)", Pattern.DOTALL);

	static Pattern	nummeric		= Pattern.compile("\\d*");

	public LibraryVersion(String baseline, String qualifier) {
		this(qualifier == null ? baseline : baseline + "." + cleanupModifier(qualifier));
	}

	public LibraryVersion(RevisionRef ref) {
		this(ref.version);
	}

	public LibraryVersion(Revision revision) {
		this(revision.version);
	}

	public LibraryVersion(String version) {
		this.input = version;
		if (version.trim().isEmpty())
			version = "0";

		Matcher m = fuzzyVersion.matcher(version);
		if (!m.matches()) {
			m = fuzzyVersion.matcher("0.0.0." + cleanupModifier(version));
			if (!m.matches()) {
				throw new IllegalArgumentException("Really screwed up version " + version);
			}
		}

		String major = removeLeadingZeroes(m.group(1));
		String minor = removeLeadingZeroes(m.group(3));
		String micro = removeLeadingZeroes(m.group(5));
		String qualifier = m.group(7);

		if (qualifier == null) {
			if (!isInteger(minor)) {
				qualifier = minor;
				minor = "0";
				micro = "0";
			} else if (!isInteger(micro)) {
				qualifier = micro;
				micro = "0";
			}
		}

		this.major = toLong(major);
		this.minor = toLong(minor);
		this.micro = toLong(micro);
		this.qualifier = cleanupModifier(qualifier);
	}

	private long toLong(String s) {
		try {
			return Long.parseLong(s);
		}
		catch (Exception e) {
			return 0;
		}
	}

	public static boolean isValid(String version) {
		Matcher m = fuzzyVersion.matcher(version);
		return m.matches();
	}

	public String toString() {
		return input;
	}

	public String getBaseline() {
		return major + "." + minor + "." + micro;
	}

	public String getQualifier() {
		return qualifier;
	}

	public String toOSGi() {
		if (qualifier != null)
			return getBaseline() + "." + qualifier;
		return getBaseline();
	}

	public boolean equals(Object other) {
		if (this == other)
			return true;

		if (!(other instanceof LibraryVersion))
			return false;

		return compareTo((LibraryVersion) other) == 0;
	}

	@Override
	public int hashCode() {
		return (int) (major * 97 ^ minor * 13 ^ micro + (qualifier == null ? 97 : qualifier.hashCode()));
	}

	@Override
	public int compareTo(LibraryVersion other) {
		if (other == this)
			return 0;

		LibraryVersion o = other;
		if (major != o.major)
			return major > o.major ? 1 : -1;

		if (minor != o.minor)
			return minor > o.minor ? 1 : -1;

		if (micro != o.micro)
			return micro > o.micro ? 1 : -1;

		int c = 0;
		if (qualifier != null)
			c = 1;
		if (o.qualifier != null)
			c += 2;

		switch (c) {
			case 0 :
				return 0;
			case 1 :
				return 1;
			case 2 :
				return -1;
		}
		return qualifier.compareTo(o.qualifier);
	}

	/**
	 * Clean up version parameters. Other builders use more fuzzy definitions of
	 * the version syntax. This method cleans up such a version to match an OSGi
	 * version.
	 * 
	 * @param VERSION_STRING
	 * @return
	 */

	/**
	 * TRhe cleanup version got confused when people used numeric dates like
	 * 201209091230120 as qualifiers. These are too large for Integers. This
	 * method checks if the all digit string fits in an integer.
	 * 
	 * <pre>
	 * maxint = 2,147,483,647 = 10 digits
	 * </pre>
	 * 
	 * @param integer
	 * @return if this fits in an integer
	 */
	private static boolean isInteger(String minor) {
		return minor.length() < 10 || (minor.length() == 10 && minor.compareTo("2147483647") < 0);
	}

	private static String removeLeadingZeroes(String group) {
		if (group == null)
			return "0";

		int n = 0;
		while (n < group.length() - 1 && group.charAt(n) == '0')
			n++;
		if (n == 0)
			return group;

		return group.substring(n);
	}

	static String cleanupModifier(String modifier) {
		if (modifier == null)
			return null;

		StringBuilder sb = new StringBuilder();

		for (int i = 0; i < modifier.length(); i++) {
			char c = modifier.charAt(i);
			if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || c == '_' || c == '-')
				sb.append(c);
		}
		return sb.toString();
	}

	public Object getWithoutQualifier() {
		return major + "." + minor + "." + micro;
	}

	public Version getVersion() {
		return new Version((int) major, (int) minor, (int) micro, qualifier);
	}
}
