// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.codeInspection.ex;

import com.intellij.codeInspection.*;
import com.intellij.codeInspection.reference.RefGraphAnnotator;
import com.intellij.codeInspection.reference.RefManagerImpl;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class GlobalInspectionToolWrapper extends InspectionToolWrapper<GlobalInspectionTool, InspectionEP> {
  private static final Logger LOG = Logger.getInstance(GlobalInspectionToolWrapper.class);

  public GlobalInspectionToolWrapper(@NotNull GlobalInspectionTool tool) {
    super(tool, InspectionEP.GLOBAL_INSPECTION.getByKey(tool.getShortName(), GlobalInspectionToolWrapper.class, InspectionEP::getShortName));
  }

  public GlobalInspectionToolWrapper(@NotNull GlobalInspectionTool tool, @NotNull InspectionEP ep) {
    super(tool, ep);
  }

  public GlobalInspectionToolWrapper(@NotNull InspectionEP ep) {
    super(ep);
  }

  private GlobalInspectionToolWrapper(@NotNull GlobalInspectionToolWrapper other) {
    super(other);
  }

  @Override
  public @NotNull GlobalInspectionToolWrapper createCopy() {
    return new GlobalInspectionToolWrapper(this);
  }

  @Override
  public void initialize(@NotNull GlobalInspectionContext context) {
    RefManagerImpl refManager = (RefManagerImpl)context.getRefManager();
    final RefGraphAnnotator annotator = getTool().getAnnotator(refManager);
    if (annotator != null) {
      refManager.registerGraphAnnotator(annotator);
    }
    super.initialize(context);
  }

  @Override
  public JobDescriptor @NotNull [] getJobDescriptors(@NotNull GlobalInspectionContext context) {
    GlobalInspectionTool tool = getTool();
    JobDescriptor[] additionalJobs = ObjectUtils.notNull(tool.getAdditionalJobs(context), JobDescriptor.EMPTY_ARRAY);
    StdJobDescriptors stdJobDescriptors = context.getStdJobDescriptors();
    if (tool.isGraphNeeded()) {
      additionalJobs = additionalJobs.length == 0 ? stdJobDescriptors.BUILD_GRAPH_ONLY :
                       ArrayUtil.append(additionalJobs, stdJobDescriptors.BUILD_GRAPH);
    }
    if (tool.isGlobalSimpleInspectionTool()) {
      // if we run e.g., just "Annotator" simple global tool then myJobDescriptors are empty but LOCAL_ANALYSIS is used from inspectFile()
      additionalJobs = additionalJobs.length == 0 ? stdJobDescriptors.LOCAL_ANALYSIS_ARRAY :
                       ArrayUtil.contains(stdJobDescriptors.LOCAL_ANALYSIS, additionalJobs) ? additionalJobs :
                       ArrayUtil.append(additionalJobs, stdJobDescriptors.LOCAL_ANALYSIS);
    }
    return additionalJobs;
  }

  public boolean worksInBatchModeOnly() {
    return getTool().worksInBatchModeOnly();
  }

  public @Nullable LocalInspectionToolWrapper getSharedLocalInspectionToolWrapper() {
    final LocalInspectionTool sharedTool = getTool().getSharedLocalInspectionTool();
    if (sharedTool == null) {
      LOG.assertTrue(!isCleanupTool(), "Global cleanup tool MUST have shared local tool. The tool short name: " + getShortName());
      return null;
    }
    return new LocalInspectionToolWrapper(sharedTool){
      @Override
      public @NotNull String getDisplayName() {
        return GlobalInspectionToolWrapper.this.getDisplayName();
      }

      @Override
      public @Nullable String getLanguage() {
        return GlobalInspectionToolWrapper.this.getLanguage(); // inherit "language=" xml tag from the global inspection EP
      }
    };
  }
}
