// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package com.intellij.codeInsight.lookup;

import com.intellij.codeInsight.completion.CompletionService;
import com.intellij.codeInsight.completion.LookupElementListPresenter;
import com.intellij.codeInsight.completion.PrefixMatcher;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.Pair;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Predicate;

/**
 * An object holding the items to be shown in a {@link Lookup} and determining their order.
 * If accessed from multiple threads, it needs to take care of proper synchronization itself.
 */
public abstract class LookupArranger implements WeighingContext {
  protected final List<LookupElement> myItems = new ArrayList<>();
  private final List<LookupElement> myMatchingItems = new ArrayList<>();
  private final List<LookupElement> myExactPrefixItems = new ArrayList<>();
  private final List<LookupElement> myInexactPrefixItems = new ArrayList<>();
  private final List<LookupElement> myTopPriorityItems = new ArrayList<>();
  private final Key<PrefixMatcher> myMatcherKey = Key.create("LookupArrangerMatcher");
  private volatile @Nullable Predicate<LookupElement> myAdditionalMatcher;
  private volatile @NotNull String myAdditionalPrefix = "";

  public void addElement(@NotNull LookupElement item, @NotNull LookupElementPresentation presentation) {
    myItems.add(item);
    updateCache(item);
  }

  private void updateCache(@NotNull LookupElement item) {
    if (!prefixMatches(item)) {
      return;
    }
    myMatchingItems.add(item);

    if (supportCustomCaches(item)) {
      return;
    }

    if (isTopPriorityItem(item)) {
      myTopPriorityItems.add(item);
    }
    if (isPrefixItem(item, true)) {
      myExactPrefixItems.add(item);
    } else if (isPrefixItem(item, false)) {
      myInexactPrefixItems.add(item);
    }
  }

  /**
   * Determines whether custom caching for the specified {@code LookupElement} is supported.
   *
   * @param item the {@code LookupElement} to check for custom cache support and process it, or {@code null} if no specific element is provided
   * @return {@code true} if custom caching is supported for the given element; {@code false} otherwise
   */
  @ApiStatus.Internal
  protected boolean supportCustomCaches(@Nullable LookupElement item) {
    return false;
  }

  /**
   * Registers a custom matcher for the specified lookup element.
   *
   * @param item    the lookup element to register the matcher for
   * @param matcher the matcher to register
   */
  public void registerMatcher(@NotNull LookupElement item, @NotNull PrefixMatcher matcher) {
    item.putUserData(myMatcherKey, matcher);
  }

  /**
   * Registers an additional matcher for the lookup.
   * Every item is checked by this matcher.
   *
   * @param matcher an additional matcher to register
   */
  @ApiStatus.Internal
  @ApiStatus.Experimental
  public void registerAdditionalMatcher(@NotNull Predicate<LookupElement> matcher) {
    myAdditionalMatcher = matcher;
  }

  @ApiStatus.Internal
  @ApiStatus.Experimental
  public @Nullable Predicate<LookupElement> getAdditionalMatcher() {
    return myAdditionalMatcher;
  }

  @Override
  public @NotNull String itemPattern(@NotNull LookupElement element) {
    // This method is not synchronized in BaseCompletionLookupArranger.
    // The only shared state accessed here is myAdditionalPrefix,
    // which is declared as volatile and accessed atomically.
    String prefix = itemMatcher(element).getPrefix();
    String additionalPrefix = myAdditionalPrefix;
    return additionalPrefix.isEmpty() ? prefix : prefix + additionalPrefix;
  }

  @Override
  public @NotNull PrefixMatcher itemMatcher(@NotNull LookupElement item) {
    PrefixMatcher matcher = item.getUserData(myMatcherKey);
    if (matcher == null) {
      throw new AssertionError("Item not in lookup: item=" + item + "; lookup items=" + myItems);
    }
    return matcher;
  }

  private boolean prefixMatches(@NotNull LookupElement item) {
    PrefixMatcher matcher = itemMatcher(item);
    Predicate<LookupElement> additionalMatcher = myAdditionalMatcher;
    if (additionalMatcher != null && !additionalMatcher.test(item)) {
      return false;
    }
    if (!myAdditionalPrefix.isEmpty()) {
      matcher = matcher.cloneWithPrefix(matcher.getPrefix() + myAdditionalPrefix);
    }
    return matcher.prefixMatches(item);
  }

  public void itemSelected(@Nullable LookupElement lookupItem, char completionChar) {
  }

  public void prefixReplaced(@NotNull Lookup lookup, @NotNull String newPrefix) {
    ArrayList<LookupElement> itemCopy = new ArrayList<>(myItems);
    myItems.clear();
    for (LookupElement item : itemCopy) {
      if (item.isValid()) {
        PrefixMatcher matcher = itemMatcher(item).cloneWithPrefix(newPrefix);
        if (matcher.prefixMatches(item)) {
          item.putUserData(myMatcherKey, matcher);
          myItems.add(item);
        }
      }
    }

    prefixChanged(lookup);
  }

  public void prefixChanged(@NotNull Lookup lookup) {
    myAdditionalPrefix = ((LookupElementListPresenter)lookup).getAdditionalPrefix();
    rebuildItemCache();
  }

  @ApiStatus.Internal
  protected void rebuildItemCache() {
    myMatchingItems.clear();
    myExactPrefixItems.clear();
    myInexactPrefixItems.clear();
    myTopPriorityItems.clear();

    for (LookupElement item : myItems) {
      updateCache(item);
    }
  }

  /**
   * Returns true, if item must be placed at the very top of the lookup ignoring sorting order.
   * It is placed before fully matched lookup items.
   */
  @ApiStatus.Internal
  protected boolean isTopPriorityItem(@Nullable LookupElement item) {
    return false;
  }

  protected List<LookupElement> retainItems(@NotNull Set<LookupElement> retained) {
    List<LookupElement> filtered = new ArrayList<>();
    List<LookupElement> removed = new ArrayList<>();
    for (LookupElement item : myItems) {
      (retained.contains(item) ? filtered : removed).add(item);
    }
    myItems.clear();
    myItems.addAll(filtered);

    rebuildItemCache();
    return removed;
  }

  /**
   * Returns a pair of lists: the first one contains the sorted items to be shown in the lookup,
   * the second one contains the index of the item to be selected.
   *
   * @param lookup the lookup to arrange items for
   *
   * @return a pair of lists containing the sorted items to be shown in the lookup and the index of the item to be selected
   */
  public abstract @NotNull Pair<List<LookupElement>, Integer> arrangeItems(@NotNull Lookup lookup, boolean onExplicitAction);

  public abstract @NotNull LookupArranger createEmptyCopy();

  protected @NotNull List<LookupElement> getPrefixItems(boolean exactly) {
    return Collections.unmodifiableList(exactly ? myExactPrefixItems : myInexactPrefixItems);
  }

  @ApiStatus.Internal
  protected @NotNull List<LookupElement> getTopPriorityItems() {
    return Collections.unmodifiableList(myTopPriorityItems);
  }

  protected boolean isPrefixItem(@NotNull LookupElement item, final boolean exactly) {
    final String pattern = itemPattern(item);
    for (String s : item.getAllLookupStrings()) {
      if (!s.equalsIgnoreCase(pattern)) continue;

      if (!item.isCaseSensitive() || !exactly || s.equals(pattern)) {
        return true;
      }
    }
    return false;
  }

  public @NotNull List<LookupElement> getMatchingItems() {
    return myMatchingItems;
  }

  /**
   * @param items the items to give relevance weight for
   * @param hideSingleValued whether criteria that gave same values for all items should be skipped
   * @return for each item, an (ordered) map of criteria used for lookup relevance sorting
   * along with the objects representing the weights in these criteria
   */
  public @NotNull Map<LookupElement, List<Pair<String, Object>>> getRelevanceObjects(@NotNull Iterable<? extends LookupElement> items,
                                                                                     boolean hideSingleValued) {
    return Collections.emptyMap();
  }

  /**
   * Called when the prefix has been truncated farther than the additional prefix typed while the lookup was visible.
   * Usually, we want to restart completion in this case.
   */
  public void prefixTruncated(@NotNull LookupEx lookup, int hideOffset) {
    lookup.hideLookup(false);
  }

  public boolean isCompletion() {
    return false;
  }

  public static class DefaultArranger extends LookupArranger {
    @Override
    public @NotNull Pair<List<LookupElement>, Integer> arrangeItems(@NotNull Lookup lookup, boolean onExplicitAction) {
      LinkedHashSet<LookupElement> result = new LinkedHashSet<>();
      result.addAll(getPrefixItems(true));
      result.addAll(getPrefixItems(false));

      List<LookupElement> items = getMatchingItems();
      for (LookupElement item : items) {
        if (CompletionService.isStartMatch(item, this)) {
          result.add(item);
        }
      }
      result.addAll(items);
      ArrayList<LookupElement> list = new ArrayList<>(result);
      int selected = !lookup.isSelectionTouched() && onExplicitAction ? 0 : list.indexOf(lookup.getCurrentItem());
      return new Pair<>(list, Math.max(selected, 0));
    }

    @Override
    public @NotNull LookupArranger createEmptyCopy() {
      return new DefaultArranger();
    }
  }
}
