// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.xdebugger.impl.evaluate;

import com.intellij.openapi.Disposable;
import com.intellij.openapi.editor.ex.EditorEx;
import com.intellij.openapi.editor.ex.util.EditorUtil;
import com.intellij.openapi.keymap.KeymapUtil;
import com.intellij.openapi.project.Project;
import com.intellij.ui.components.JBLabel;
import com.intellij.util.ui.JBUI;
import com.intellij.util.ui.UIUtil;
import com.intellij.util.ui.components.BorderLayoutPanel;
import com.intellij.xdebugger.XDebuggerBundle;
import com.intellij.xdebugger.XExpression;
import com.intellij.xdebugger.XSourcePosition;
import com.intellij.xdebugger.evaluation.XDebuggerEditorsProvider;
import com.intellij.xdebugger.impl.ui.XDebuggerEditorBase;
import com.intellij.xdebugger.impl.ui.XDebuggerExpressionComboBox;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.awt.*;

public class ExpressionInputComponent extends EvaluationInputComponent {
  private final XDebuggerEditorBase myExpressionEditor;
  private final ExpressionInputForm myMainForm = new ExpressionInputForm();

  public ExpressionInputComponent(final @NotNull Project project,
                                  @NotNull XDebuggerEditorsProvider editorsProvider,
                                  @Nullable String historyId,
                                  final @Nullable XSourcePosition sourcePosition,
                                  @Nullable XExpression expression,
                                  @NotNull Disposable parentDisposable,
                                  boolean showHelp) {
    super(XDebuggerBundle.message("xdebugger.dialog.title.evaluate.expression"));
    BorderLayoutPanel expressionPanel = JBUI.Panels.simplePanel();
    myExpressionEditor = new XDebuggerExpressionComboBox(project, editorsProvider, historyId, sourcePosition, true, false) {
      @Override
      protected void prepareEditor(EditorEx editor) {
        super.prepareEditor(editor);
        Font font = EditorUtil.getEditorFont();
        editor.getColorsScheme().setEditorFontName(font.getFontName());
        editor.getColorsScheme().setEditorFontSize(font.getSize());
        editor.getSettings().setLineCursorWidth(EditorUtil.getDefaultCaretWidth());
      }
    };
    myExpressionEditor.setExpression(expression);
    expressionPanel.addToCenter(myExpressionEditor.getComponent());
    final JBLabel help = new JBLabel(XDebuggerBundle.message("xdebugger.evaluate.addtowatches.hint",
                                                             KeymapUtil.getKeystrokeText(XDebuggerEvaluationDialog.getAddWatchKeystroke())),
                                     SwingConstants.RIGHT);
    help.setBorder(JBUI.Borders.empty(2, 0, 6, 0));
    help.setComponentStyle(UIUtil.ComponentStyle.SMALL);
    help.setFontColor(UIUtil.FontColor.BRIGHTER);
    expressionPanel.addToBottom(help);
    help.setVisible(showHelp);

    myMainForm.addExpressionComponent(expressionPanel);
    myMainForm.addLanguageComponent(myExpressionEditor.getLanguageChooser());
  }

  @Override
  public void addComponent(JPanel contentPanel, JPanel resultPanel) {
    contentPanel.add(resultPanel, BorderLayout.CENTER);
    contentPanel.add(myMainForm.getMainPanel(), BorderLayout.NORTH);
  }

  @Override
  public JPanel getMainComponent() {
    return myMainForm.getMainPanel();
  }

  @Override
  public @NotNull XDebuggerEditorBase getInputEditor() {
    return myExpressionEditor;
  }
}
