// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.platform.eel.provider

import com.intellij.openapi.diagnostic.logger
import com.intellij.openapi.diagnostic.trace
import com.intellij.openapi.project.Project
import com.intellij.platform.eel.EelDescriptor
import com.intellij.platform.eel.EelPathBoundDescriptor
import com.intellij.platform.eel.annotations.MultiRoutingFileSystemPath
import com.intellij.platform.eel.isPosix
import com.intellij.platform.eel.path.EelPath
import com.intellij.platform.eel.path.EelPathException
import org.jetbrains.annotations.ApiStatus
import java.nio.file.FileSystems
import java.nio.file.Path

/**
 * Converts [EelPath], which is likely a path from a remote machine, to a [Path] for the local machine.
 *
 * Example:
 * ```kotlin
 * EelPath.parse("/home/user", getWslDescriptorSomewhere()).asNioPath() ==
 *     Path.of("""\\wsl.localhost\Ubuntu\home\user""")
 *
 * EelPath.parse("/home/user", getDocekrDescriptorSomewhere()).asNioPath() ==
 *     Path.of("""\\docker\f00b4r\home\user""")
 * ```
 *
 * @throws IllegalArgumentException if the Eel API for [this] does not have a corresponding [java.nio.file.FileSystem]
 */
@Throws(IllegalArgumentException::class)
@ApiStatus.Internal
fun EelPath.asNioPath(): @MultiRoutingFileSystemPath Path {
  return asNioPathOrNull()
         ?: throw IllegalArgumentException("Could not convert $this to NIO path, descriptor is $descriptor")
}

@Throws(IllegalArgumentException::class)
@ApiStatus.Internal
@Deprecated("Use asNioPath() instead", replaceWith = ReplaceWith("asNioPath()"))
fun EelPath.asNioPath(project: Project?): @MultiRoutingFileSystemPath Path {
  return asNioPath()
}

/** See docs for [asNioPath] */
@Deprecated("It never returns null anymore")
@ApiStatus.Internal
fun EelPath.asNioPathOrNull(): @MultiRoutingFileSystemPath Path? {
  if (descriptor === LocalEelDescriptor) {
    return Path.of(toString())
  }

  // Comparing strings because `Path.of("\\wsl.localhost\distro\").equals(Path.of("\\wsl$\distro\")) == true`
  // If the project works with `wsl$` paths, this function must return `wsl$` paths, and the same for `wsl.localhost`.
  val root = (descriptor as? EelPathBoundDescriptor)?.rootPath ?: return null

  LOG.trace {
    "asNioPathOrNull():" +
    " path=$this" +
    " descriptor=$descriptor" +
    " rootPath=$root"
  }

  @MultiRoutingFileSystemPath
  val result = parts.fold(root, Path::resolve)
  LOG.trace {
    "asNioPathOrNull(): path=$this basePath=$root result=$result"
  }
  return result
}

/**
 * Converts a path generated by the default NIO filesystem to [EelPath].
 *
 * Example:
 * ```kotlin
 * Path.of("""C:\Windows""").asEelPath() ==
 *     EelPath.parse("""C:\Windows""", LocalEelDescriptor)
 *
 * Path.of("""\\wsl$\Ubuntu\usr""").asEelPath() ==
 *     EelPath.parse("/usr", someWslDescriptor)
 * ```
 *
 * @throws IllegalArgumentException if the passed path cannot be mapped to a path corresponding to Eel.
 * It can happen if [this] belongs to a [java.nio.file.FileSystem] that was not registered as a backend of `MultiRoutingFileSystemProvider`
 *
 * @throws EelPathException if the passed path is not an absolute path.
 */
@Throws(IllegalArgumentException::class, EelPathException::class)
@ApiStatus.Internal
fun Path.asEelPath(): EelPath {
  return asEelPath(getEelDescriptor())
}

/**
 * [descriptor] should be exactly `this.getEelDescriptor()`. This method exists only to avoid calling `getEelDescriptor()` twice.
 */
@Throws(IllegalArgumentException::class, EelPathException::class)
@ApiStatus.Internal
fun Path.asEelPath(descriptor: EelDescriptor): EelPath {
  if (fileSystem != FileSystems.getDefault()) {
    throw IllegalArgumentException("Could not convert $this to EelPath: the path does not belong to the default NIO FileSystem")
  }
  when (descriptor) {
    is LocalEelDescriptor -> return EelPath.parse(toString(), descriptor)
    is EelPathBoundDescriptor if (descriptor.osFamily.isPosix) -> {
      val root = descriptor.rootPath
      val relative = root.relativize(this)
      return relative.fold(EelPath.parse("/", descriptor)) { path, part ->
        part.toString().takeIf { it.isNotEmpty() }?.let { path.getChild(it) } ?: path
      }
    }
    is EelPathBoundDescriptor -> {
      TODO() // on Windows, we need additional logic to guess the new root
    }
    else -> {
      throw NoSuchElementException("Cannot find a root for $this")
    }
  }
}

@ApiStatus.Internal
fun EelDescriptor.mountProvider(): EelMountProvider? {
  return EelProvider.EP_NAME.extensionList
    .firstNotNullOfOrNull { eelProvider ->
      eelProvider.getMountProvider(eelDescriptor = this)
    }
}

@ApiStatus.Internal
fun EelDescriptor.routingPrefixes(): Set<Path> {
  return EelProvider.EP_NAME.extensionList
    .flatMapTo(HashSet()) { eelProvider ->
      eelProvider.getCustomRoots(this)?.map(Path::of) ?: emptySet()
    }
}

private class EelNioBridgeService

private val LOG = logger<EelNioBridgeService>()