/*
 * Copyright 2000-2009 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.diagnostic.logging;

import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.io.FileUtilRt;
import com.intellij.psi.search.GlobalSearchScope;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.Arrays;

public abstract class LogConsoleImpl extends LogConsoleBase {
  private final @NlsSafe String myPath;
  private final @NotNull File myFile;
  private final @NotNull Charset myCharset;
  private FileSnapshot myOldSnapshot;

  public LogConsoleImpl(Project project,
                        @NotNull File file,
                        @NotNull Charset charset,
                        long skippedContents,
                        @NlsContexts.TabTitle @NotNull String title,
                        final boolean buildInActions,
                        final GlobalSearchScope searchScope) {
    super(project, getReader(file, charset, skippedContents), title, buildInActions, new DefaultLogFilterModel(project),
          searchScope);
    myPath = file.getAbsolutePath();
    myFile = file;
    myCharset = charset;
    myOldSnapshot = new FileSnapshot();
  }

  private static @Nullable Reader getReader(@NotNull File file, @NotNull Charset charset, long skippedContents) {
    try {
      try {
        @SuppressWarnings("IOResourceOpenedButNotSafelyClosed")
        InputStream inputStream = Files.newInputStream(file.toPath());
        //do not skip forward
        if (file.length() >= skippedContents) {
          long skipped = 0;
          while (skipped < skippedContents) {
            skipped += inputStream.skip(skippedContents - skipped);
          }
        }
        return new BufferedReader(new InputStreamReader(inputStream, charset));
      }
      catch (FileNotFoundException ignored) {
        if (FileUtilRt.createIfNotExists(file)) {
          return new BufferedReader(new InputStreamReader(Files.newInputStream(file.toPath()), charset));
        }
        return null;
      }
    }
    catch (Throwable ignored) {
      return null;
    }
  }

  @Override
  public @Nullable String getTooltip() {
    return myPath;
  }

  public String getPath() {
    return myPath;
  }

  @Override
  protected @Nullable BufferedReader updateReaderIfNeeded(@Nullable BufferedReader reader) throws IOException {
    if (reader == null) {
      return null;
    }

    FileSnapshot snapshot = new FileSnapshot();
    if (myOldSnapshot.rolloverDetected(snapshot)) {
      reader.close();
      //noinspection IOResourceOpenedButNotSafelyClosed
      reader = new BufferedReader(new InputStreamReader(Files.newInputStream(myFile.toPath()), myCharset));
    }
    myOldSnapshot = snapshot;
    return reader;
  }

  private final class FileSnapshot {
    final long length;
    final byte[] firstBytes;

    FileSnapshot() {
      this.length = myFile.length();

      byte[] bytes = new byte[20];
      try (InputStream stream = Files.newInputStream(myFile.toPath())) {
        //noinspection ResultOfMethodCallIgnored
        stream.read(bytes);
      }
      catch (IOException ignore) {
      }
      this.firstBytes = bytes;
    }

    boolean rolloverDetected(FileSnapshot current) {
      return current.length < length || !Arrays.equals(firstBytes, current.firstBytes);
    }
  }
}
