// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.execution;

import com.intellij.execution.configurations.RunConfiguration;
import com.intellij.execution.runners.ExecutionEnvironment;
import com.intellij.openapi.project.Project;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.messages.Topic;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

import java.util.Collections;
import java.util.List;

public abstract class ExecutionTargetManager {
  public static final Topic<ExecutionTargetListener> TOPIC = Topic.create("ExecutionTarget topic", ExecutionTargetListener.class);

  public static @NotNull ExecutionTargetManager getInstance(@NotNull Project project) {
    return project.getService(ExecutionTargetManager.class);
  }

  public static @NotNull ExecutionTarget getActiveTarget(@NotNull Project project) {
    return getInstance(project).getActiveTarget();
  }

  public static void setActiveTarget(@NotNull Project project, @NotNull ExecutionTarget target) {
    getInstance(project).setActiveTarget(target);
  }

  public static @NotNull List<ExecutionTarget> getTargetsToChooseFor(@NotNull Project project, @Nullable RunConfiguration configuration) {
    List<ExecutionTarget> result = getInstance(project).getTargetsFor(configuration);
    if (result.size() == 1 && DefaultExecutionTarget.INSTANCE.equals(result.get(0))) {
      return Collections.emptyList();
    }

    result = ContainerUtil.filter(result, target -> !target.isExternallyManaged());
    if (result.size() == 1 && DefaultExecutionTarget.INSTANCE.equals(result.get(0))) {
      return Collections.emptyList();
    }
    return result;
  }

  /**
   * @deprecated use {@link #canRun(RunConfiguration, ExecutionTarget)} instead
   */
  @Deprecated(forRemoval = true)
  public static boolean canRun(@Nullable RunnerAndConfigurationSettings settings, @Nullable ExecutionTarget target) {
    return canRun(settings != null ? settings.getConfiguration() : null, target);
  }

  public static boolean canRun(@Nullable RunConfiguration configuration, @Nullable ExecutionTarget target) {
    if (configuration == null || target == null) {
      return false;
    }
    else {
      return getInstance(configuration.getProject()).doCanRun(configuration, target);
    }
  }

  public static boolean canRun(@NotNull ExecutionEnvironment environment) {
    RunnerAndConfigurationSettings settings = environment.getRunnerAndConfigurationSettings();
    return settings != null && canRun(settings.getConfiguration(), environment.getExecutionTarget());
  }

  public abstract boolean doCanRun(@Nullable RunConfiguration configuration, @NotNull ExecutionTarget target);

  public static void update(@NotNull Project project) {
    getInstance(project).update();
  }

  public abstract @NotNull ExecutionTarget getActiveTarget();

  public abstract void setActiveTarget(@NotNull ExecutionTarget target);

  public abstract @NotNull @Unmodifiable List<ExecutionTarget> getTargetsFor(@Nullable RunConfiguration configuration);

  public abstract void update();

  public ExecutionTarget findTarget(RunConfiguration configuration) {
    ExecutionTarget target = getActiveTarget();
    if (canRun(configuration, target)) {
      return target;
    }

    List<ExecutionTarget> targets = getTargetsFor(configuration);
    return ContainerUtil.getFirstItem(targets);
  }
}
