// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package com.intellij.codeInsight.folding.impl;

import com.intellij.codeInsight.CodeInsightActionHandler;
import com.intellij.codeInsight.CodeInsightBundle;
import com.intellij.codeInsight.hint.HintManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.editor.FoldRegion;
import com.intellij.openapi.editor.ex.FoldingModelEx;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.ui.Messages;
import com.intellij.psi.PsiFile;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

@ApiStatus.Internal
public final class CollapseSelectionHandler implements CodeInsightActionHandler {
  private static final String ourPlaceHolderText = "...";
  private static final Logger LOG = Logger.getInstance(CollapseSelectionHandler.class);

  @Override
  public void invoke(@NotNull Project project, final @NotNull Editor editor, @NotNull PsiFile psiFile) {
    editor.getFoldingModel().runBatchFoldingOperation(
      () -> {
        final EditorFoldingInfo info = EditorFoldingInfo.get(editor);
        FoldingModelEx foldingModel = (FoldingModelEx) editor.getFoldingModel();
        int start = editor.getSelectionModel().getSelectionStart();
        int end = editor.getSelectionModel().getSelectionEnd();
        if (start < end) {
          Document doc = editor.getDocument();
          if (doc.getCharsSequence().charAt(end - 1) == '\n') {
            end--;
          }
          FoldRegion region;
          if ((region = FoldingUtil.findFoldRegion(editor, start, end)) != null) {
            if (info.getPsiElement(region) == null) {
              editor.getFoldingModel().removeFoldRegion(region);
              info.removeRegion(region);
            }
            else {
              HintManager.getInstance().showInformationHint(editor, CodeInsightBundle.message("collapse.selection.existing.autogenerated.region"));
            }
          }
          else {
            if (foldingModel.intersectsRegion(start, end)) {
              if (Messages.showDialog(project,
                                      CodeInsightBundle.message("collapse.selection.overlapping.warning.text"),
                                      CodeInsightBundle.message("collapse.selection.overlapping.warning.title"),
                                      new String[]{
                                        CodeInsightBundle.message("collapse.selection.overlapping.warning.ok"),
                                        CodeInsightBundle.message("collapse.selection.overlapping.warning.cancel")},
                                      1,
                                      Messages.getWarningIcon()) != 0) {
                return;
              }
              for (FoldRegion r : foldingModel.getAllFoldRegions()) {
                if (r.getStartOffset() < start && r.getEndOffset() > start && r.getEndOffset() < end ||
                    r.getStartOffset() > start && r.getStartOffset() < end && r.getEndOffset() > end) {
                  foldingModel.removeFoldRegion(r);
                }
              }
            }
            if (start < end) {
              region = foldingModel.addFoldRegion(start, end, ourPlaceHolderText);
              LOG.assertTrue(region != null, "Fold region is not created for (" +start+","+end+ "). Folding model: " + foldingModel + foldingModel.getClass());
              region.setExpanded(false);
              int offset = Math.min(start + ourPlaceHolderText.length(), doc.getTextLength());
              editor.getCaretModel().moveToOffset(offset);
            }
          }
        }
        else {
          FoldRegion[] regions = FoldingUtil.getFoldRegionsAtOffset(editor, editor.getCaretModel().getOffset());
          if (regions.length > 0) {
            FoldRegion region = regions[0];
            if (info.getPsiElement(region) == null) {
              editor.getFoldingModel().removeFoldRegion(region);
              info.removeRegion(region);
            }
            else {
              region.setExpanded(!region.isExpanded());
            }
          }
        }
      }
    );
  }

  public static boolean isEnabled(@NotNull Editor editor) {
    return editor.getSelectionModel().getSelectionStart() < editor.getSelectionModel().getSelectionEnd() ||
           FoldingUtil.getFoldRegionsAtOffset(editor, editor.getCaretModel().getOffset()).length > 0;
  }

  @Override
  public boolean startInWriteAction() {
    return false;
  }
}
