// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.wm.impl.content;

import com.intellij.ide.IdeBundle;
import com.intellij.openapi.actionSystem.ActionToolbar;
import com.intellij.openapi.ui.popup.ListPopup;
import com.intellij.openapi.util.NlsActions;
import com.intellij.ui.ExperimentalUI;
import com.intellij.ui.MouseDragHelper;
import com.intellij.ui.awt.RelativePoint;
import com.intellij.ui.content.ContentManager;
import com.intellij.util.ui.JBUI;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.jetbrains.annotations.NotNull;

import java.awt.*;

final class ComboContentLayout extends ContentLayout {
  ContentComboLabel comboLabel;

  ComboContentLayout(ToolWindowContentUi ui) {
    super(ui);
  }

  @Override
  public void init(@NotNull ContentManager contentManager) {
    reset();

    idLabel = new BaseLabel(ui, ExperimentalUI.isNewUI());
    MouseDragHelper.setComponentDraggable(idLabel, true);
    comboLabel = new ContentComboLabel(this);
  }

  @Override
  public void reset() {
    idLabel = null;
    comboLabel = null;
  }

  @Override
  public void layout() {
    Rectangle bounds = ui.getTabComponent().getBounds();
    Dimension idSize = isIdVisible() ? idLabel.getPreferredSize() : JBUI.emptySize();
    int eachX = 0;
    int eachY = 0;

    idLabel.setBounds(eachX, eachY, idSize.width, bounds.height);
    eachX += idSize.width;

    Dimension comboSize = comboLabel.getPreferredSize();
    int nonLabelWidth = getTabToolbarPreferredWidth();
    int spaceLeft = bounds.width - eachX - nonLabelWidth - (isToDrawCombo() && isIdVisible() ? 3 : 0);

    int width = comboSize.width;
    if (width > spaceLeft) {
      width = spaceLeft;
    }

    comboLabel.setBounds(eachX, eachY, width, bounds.height);
    eachX += width;

    // Tab toolbar is positioned at the end.
    ActionToolbar tabToolbar = ui.getTabToolbar();
    if (tabToolbar != null) {
      JComponent component = tabToolbar.getComponent();
      Dimension size = component.getPreferredSize();
      component.setBounds(eachX, eachY + (bounds.height - size.height) / 2, size.width, size.height);
      eachX += component.getWidth();
    }
  }

  @Override
  public int getMinimumWidth() {
    return (idLabel == null ? 0 : idLabel.getPreferredSize().width) + getTabToolbarPreferredWidth();
  }

  @Override
  public void update() {
    updateIdLabel(idLabel);
    comboLabel.update();
  }

  @Override
  public void rebuild() {
    JPanel tabComponent = ui.getTabComponent();
    tabComponent.removeAll();

    tabComponent.add(idLabel);
    ToolWindowContentUi.initMouseListeners(idLabel, ui, true);

    tabComponent.add(comboLabel);
    ToolWindowContentUi.initMouseListeners(comboLabel, ui, false);

    ui.connectTabToolbar();
  }

  boolean isToDrawCombo() {
    return ui.getContentManager().getContentCount() > 1;
  }

  @Override
  public void showContentPopup(ListPopup listPopup) {
    final int width = comboLabel.getSize().width;
    listPopup.setMinimumSize(new Dimension(width, 0));
    listPopup.show(new RelativePoint(comboLabel, new Point(0, comboLabel.getHeight())));
  }

  @Override
  public @NlsActions.ActionText String getCloseActionName() {
    return IdeBundle.message("action.ComboContentLayout.close.view.text");
  }

  @Override
  public @NlsActions.ActionText String getCloseAllButThisActionName() {
    return IdeBundle.message("action.ComboContentLayout.close.other.views.text");
  }

  @Override
  public @NlsActions.ActionText String getPreviousContentActionName() {
    return IdeBundle.message("action.ComboContentLayout.select.previous.view.text");
  }

  @Override
  public @NlsActions.ActionText String getNextContentActionName() {
    return IdeBundle.message("action.ComboContentLayout.select.next.view.text");
  }
}
