// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.toolWindow

import com.intellij.openapi.Disposable
import com.intellij.openapi.ui.FrameWrapper
import com.intellij.openapi.ui.popup.Balloon
import com.intellij.openapi.wm.ToolWindowAnchor
import com.intellij.openapi.wm.WindowInfo
import com.intellij.openapi.wm.impl.FloatingDecorator
import com.intellij.openapi.wm.impl.ToolWindowExternalDecorator
import com.intellij.openapi.wm.impl.ToolWindowImpl
import com.intellij.ui.drag.DragButton
import com.intellij.ui.drag.asDragButton
import javax.swing.Icon
import javax.swing.JComponent

internal interface StripeButtonManager {
  val id: String
  val toolWindow: ToolWindowImpl
  val windowDescriptor: WindowInfo

  fun updateState(toolWindow: ToolWindowImpl)

  fun updatePresentation()

  fun updateIcon(icon: Icon?)

  fun remove(anchor: ToolWindowAnchor, split: Boolean)

  fun getComponent(): JComponent
  
  fun asDragButton(): DragButton = getComponent().asDragButton()
}

internal class ToolWindowEntry(stripeButton: StripeButtonManager?,
                               @JvmField val toolWindow: ToolWindowImpl,
                               @JvmField val disposable: Disposable) {
  var stripeButton: StripeButtonManager? = stripeButton
    set(value) {
      if (value == null) {
        assert(field != null)
      }
      else {
        assert(field == null)
      }
      field = value
    }

  @JvmField
  var floatingDecorator: FloatingDecorator? = null

  @JvmField
  var windowedDecorator: FrameWrapper? = null

  val externalDecorator: ToolWindowExternalDecorator?
    get() = toolWindow.decorator?.getExternalDecorator(toolWindow.type)

  @JvmField
  var balloon: Balloon? = null

  val id: String
    get() = toolWindow.id

  val readOnlyWindowInfo: WindowInfo
    get() = toolWindow.windowInfo

  fun removeStripeButton() {
    removeStripeButton(toolWindow.anchor, toolWindow.isSplitMode)
  }

  fun removeStripeButton(anchor: ToolWindowAnchor, split: Boolean) {
    val stripeButton = stripeButton ?: return
    this.stripeButton = null
    stripeButton.remove(anchor, split)
  }

  fun applyWindowInfo(newInfo: WindowInfo) {
    toolWindow.applyWindowInfo(newInfo)
    // must be applied _after_ updating tool window layout info
    val stripeButton = stripeButton ?: return
    stripeButton.updateState(toolWindow)
  }

  override fun toString(): String {
    return "ToolWindowEntry(toolWindowId=${toolWindow.id})"
  }
}