// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.ui.components;

import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.ui.ComponentUtil;
import com.intellij.ui.DocumentAdapter;
import com.intellij.util.ObjectUtils;
import com.intellij.util.ui.ComponentWithEmptyText;
import com.intellij.util.ui.JBInsets;
import com.intellij.util.ui.JBSwingUtilities;
import com.intellij.util.ui.StatusText;
import com.intellij.util.ui.accessibility.AccessibleContextUtil;
import org.jetbrains.annotations.NotNull;

import javax.accessibility.AccessibleContext;
import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.text.Document;
import java.awt.*;

public class JBTextArea extends JTextArea implements ComponentWithEmptyText {
  private final TextComponentEmptyText myEmptyText;

  public JBTextArea() {
    this(null, null, 0, 0);
  }

  public JBTextArea(@NlsContexts.DetailedDescription String text) {
    this(null, text, 0, 0);
  }

  public JBTextArea(int rows, int columns) {
    this(null, null, rows, columns);
  }

  public JBTextArea(@NlsContexts.DetailedDescription String text, int rows, int columns) {
    this(null, text, rows, columns);
  }

  public JBTextArea(Document doc) {
    this(doc, null, 0, 0);
  }

  public JBTextArea(Document doc, @NlsContexts.DetailedDescription String text, int rows, int columns) {
    super(doc, text, rows, columns);

    myEmptyText = new TextComponentEmptyText(this, true) {
      @Override
      protected Rectangle getTextComponentBound() {
        Insets insets = ObjectUtils.notNull(getInsets(), JBInsets.emptyInsets());
        Insets margin = ObjectUtils.notNull(getMargin(), JBInsets.emptyInsets());
        Insets ipad = getComponent().getIpad();
        Dimension size = getSize();
        int left = insets.left + margin.left - ipad.left - 1;
        int top = insets.top + margin.top - ipad.top + 1;
        int right = size.width - (insets.right + margin.right - ipad.right);
        return new Rectangle(left, top, right - left, getRowHeight());
      }
    };

    getDocument().addDocumentListener(new DocumentAdapter() {
      @Override
      protected void textChanged(@NotNull DocumentEvent e) {
        invalidate();
        revalidate();
        repaint();
      }
    });
  }

  @Override
  public void updateUI() {
    super.updateUI();
    if (getParent() != null) myEmptyText.resetFont();
  }

  @Override
  protected Graphics getComponentGraphics(Graphics graphics) {
    return JBSwingUtilities.runGlobalCGTransform(this, super.getComponentGraphics(graphics));
  }

  @Override
  public Dimension getPreferredSize() {
    if (isPreferredSizeSet()) return super.getPreferredSize();
    int width = 0;
    FontMetrics fontMetrics = getFontMetrics(getFont());
    for (String line : getText().split("\n")) {
      width = Math.max(width, fontMetrics.stringWidth(line));
    }
    Dimension d = super.getPreferredSize();
    Insets insets = getInsets();
    d.width = Math.min(d.width, width + insets.left + insets.right);
    return d;
  }

  @Override
  public void scrollRectToVisible(Rectangle r) {
    JViewport viewport = ComponentUtil.getParentOfType((Class<? extends JViewport>)JViewport.class, (Component)this);
    if (viewport != null) {
      r = SwingUtilities.convertRectangle(this, r, viewport);
      viewport.scrollRectToVisible(r);
    }
  }

  @Override
  public @NotNull StatusText getEmptyText() {
    return myEmptyText;
  }

  @Override
  @SuppressWarnings("DuplicatedCode")
  protected void paintComponent(Graphics g) {
    super.paintComponent(g);

    if (!myEmptyText.getStatusTriggerText().isEmpty() && myEmptyText.isStatusVisible()) {
      g.setColor(getBackground());

      Rectangle rect = new Rectangle(getSize());
      JBInsets.removeFrom(rect, getInsets());
      JBInsets.removeFrom(rect, getMargin());
      ((Graphics2D)g).fill(rect);

      g.setColor(getForeground());
    }

    myEmptyText.paintStatusText(g);
  }

  @Override
  public AccessibleContext getAccessibleContext() {
    if (accessibleContext == null) {
      accessibleContext = new AccessibleJBTextArea();
    }
    return accessibleContext;
  }

  private class AccessibleJBTextArea extends AccessibleJTextArea {
    @Override
    public String getAccessibleDescription() {
      String description = super.getAccessibleDescription();
      if (description == null && StringUtil.isEmpty(getText())) {
        //noinspection HardCodedStringLiteral
        String emptyText = myEmptyText.toString();
        if (!emptyText.isEmpty()) {
          return AccessibleContextUtil.getUniqueDescription(this, emptyText);
        }
      }
      return description;
    }
  }
}
