// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.ide.actions.runAnything.activity;

import com.intellij.execution.ExecutionException;
import com.intellij.execution.Executor;
import com.intellij.execution.configurations.GeneralCommandLine;
import com.intellij.execution.configurations.PtyCommandLine;
import com.intellij.execution.runners.ExecutionEnvironmentBuilder;
import com.intellij.icons.AllIcons;
import com.intellij.ide.IdeBundle;
import com.intellij.ide.actions.runAnything.RunAnythingAction;
import com.intellij.ide.actions.runAnything.RunAnythingCache;
import com.intellij.ide.actions.runAnything.commands.RunAnythingCommandCustomizer;
import com.intellij.ide.actions.runAnything.execution.RunAnythingRunProfile;
import com.intellij.openapi.actionSystem.CommonDataKeys;
import com.intellij.openapi.actionSystem.DataContext;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.ui.Messages;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.execution.ParametersListUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Collection;

import static com.intellij.ide.actions.runAnything.RunAnythingUtil.*;

public abstract class RunAnythingCommandProvider extends RunAnythingProviderBase<String> {
  @Override
  public void execute(@NotNull DataContext dataContext, @NotNull String value) {
    VirtualFile workDirectory = dataContext.getData(CommonDataKeys.VIRTUAL_FILE);
    Executor executor = dataContext.getData(RunAnythingAction.EXECUTOR_KEY);
    LOG.assertTrue(workDirectory != null);
    LOG.assertTrue(executor != null);

    runCommand(workDirectory, value, executor, dataContext);
  }

  public static void runCommand(@NotNull VirtualFile workDirectory,
                                @NotNull String commandString,
                                @NotNull Executor executor,
                                @NotNull DataContext dataContext) {
    final Project project = CommonDataKeys.PROJECT.getData(dataContext);
    LOG.assertTrue(project != null);

    Collection<String> commands = RunAnythingCache.getInstance(project).getState().getCommands();
    commands.remove(commandString);
    commands.add(commandString);

    dataContext = RunAnythingCommandCustomizer.customizeContext(dataContext);

    GeneralCommandLine initialCommandLine =
      new GeneralCommandLine(ParametersListUtil.parse(commandString, false, true))
        .withParentEnvironmentType(GeneralCommandLine.ParentEnvironmentType.CONSOLE)
        .withWorkDirectory(workDirectory.getPath());

    GeneralCommandLine commandLine = RunAnythingCommandCustomizer.customizeCommandLine(dataContext, workDirectory, initialCommandLine);
    try {
      RunAnythingRunProfile runAnythingRunProfile = new RunAnythingRunProfile(
        Registry.is("run.anything.use.pty", false) ? new PtyCommandLine(commandLine) : commandLine,
        commandString
      );
      ExecutionEnvironmentBuilder.create(project, executor, runAnythingRunProfile)
                                 .dataContext(dataContext)
                                 .buildAndExecute();
    }
    catch (ExecutionException e) {
      LOG.warn(e);
      Messages.showInfoMessage(project, e.getMessage(), IdeBundle.message("run.anything.console.error.title"));
    }
  }

  @Override
  public @Nullable String getAdText() {
    return getAdContextText() + ", " + getAdDebugText() + ", " + getAdDeleteCommandText();
  }

  @Override
  public @NotNull String getCommand(@NotNull String value) {
    return value;
  }

  @Override
  public @Nullable Icon getIcon(@NotNull String value) {
    return AllIcons.Actions.RunAnything;
  }
}