// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.codeInsight.daemon.impl;

import com.intellij.codeHighlighting.HighlightDisplayLevel;
import com.intellij.codeHighlighting.Pass;
import com.intellij.codeInsight.daemon.DaemonBundle;
import com.intellij.codeInsight.daemon.HighlightDisplayKey;
import com.intellij.codeInsight.daemon.impl.analysis.DaemonTooltipsUtil;
import com.intellij.codeInsight.intention.EmptyIntentionAction;
import com.intellij.codeInsight.intention.IntentionAction;
import com.intellij.codeInspection.*;
import com.intellij.codeInspection.ProblemDescriptorUtil.ProblemPresentation;
import com.intellij.codeInspection.ex.*;
import com.intellij.diagnostic.PluginException;
import com.intellij.injected.editor.DocumentWindow;
import com.intellij.lang.Language;
import com.intellij.lang.annotation.HighlightSeverity;
import com.intellij.lang.annotation.ProblemGroup;
import com.intellij.lang.injection.InjectedLanguageManager;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.colors.TextAttributesKey;
import com.intellij.openapi.editor.markup.TextAttributes;
import com.intellij.openapi.editor.markup.UnmodifiableTextAttributes;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.project.*;
import com.intellij.openapi.util.*;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.profile.codeInspection.ProjectInspectionProfileManager;
import com.intellij.psi.FileViewProvider;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.impl.source.tree.injected.InjectedFileViewProvider;
import com.intellij.psi.util.PsiUtilCore;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.Interner;
import com.intellij.workspaceModel.core.fileIndex.WorkspaceFileIndex;
import com.intellij.xml.util.XmlStringUtil;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Function;

@ApiStatus.Internal
final class LocalInspectionsPass extends ProgressableTextEditorHighlightingPass implements PossiblyDumbAware {
  private static final Logger LOG = Logger.getInstance(LocalInspectionsPass.class);
  private final @NotNull TextRange myPriorityRange;
  private final boolean myIgnoreSuppressed;
  private final HighlightInfoUpdater myHighlightInfoUpdater;
  private volatile List<? extends HighlightInfo> myInfos = Collections.emptyList(); // updated atomically
  private final InspectionProfileWrapper myProfileWrapper;
  // toolId -> suppressed elements (for which tool.isSuppressedFor(element) == true)
  private final Map<String, Set<PsiElement>> mySuppressedElements = new ConcurrentHashMap<>();
  private final boolean myInspectInjectedPsi;

  LocalInspectionsPass(@NotNull PsiFile psiFile,
                       @NotNull Document document,
                       @NotNull TextRange restrictRange,
                       @NotNull TextRange priorityRange,
                       boolean ignoreSuppressed,
                       @NotNull HighlightInfoUpdater highlightInfoUpdater,
                       boolean inspectInjectedPsi) {
    super(psiFile.getProject(), document, DaemonBundle.message("pass.inspection"), psiFile, null, restrictRange, true, HighlightInfoProcessor.getEmpty());
    myHighlightInfoUpdater = highlightInfoUpdater;
    assert psiFile.isPhysical() : "can't inspect non-physical file: " + psiFile + "; " + psiFile.getVirtualFile();
    myPriorityRange = priorityRange;
    myIgnoreSuppressed = ignoreSuppressed;
    setId(Pass.LOCAL_INSPECTIONS);

    InspectionProfileImpl profileToUse = ProjectInspectionProfileManager.getInstance(myProject).getCurrentProfile();
    Function<? super InspectionProfile, ? extends InspectionProfileWrapper>
      custom = InspectionProfileWrapper.getCustomInspectionProfileWrapper(psiFile);
    myProfileWrapper = custom == null ? new InspectionProfileWrapper(profileToUse) : custom.apply(profileToUse);
    assert myProfileWrapper != null;
    myInspectInjectedPsi = inspectInjectedPsi;
  }

  private @NotNull PsiFile getFile() {
    return myFile;
  }

  @Override
  protected void collectInformationWithProgress(@NotNull ProgressIndicator progress) {
    List<HighlightInfo> fileInfos = Collections.synchronizedList(new ArrayList<>());
    List<? extends LocalInspectionToolWrapper> toolWrappers = getInspectionTools(myProfileWrapper);
    var result = new Object() {
      List<? extends InspectionRunner.InspectionContext> resultContexts = List.of();
      List<PsiFile> injectedFragments = List.of();
    };

    VirtualFile virtualFile = getFile().getVirtualFile();
    boolean indexable = !virtualFile.isInLocalFileSystem() || WorkspaceFileIndex.getInstance(myProject).isIndexable(virtualFile);
    DumbToolWrapperCondition dumbToolWrapperCondition = new DumbToolWrapperCondition(isDumbMode(), indexable);

    if (!toolWrappers.isEmpty()) {
      if (PassExecutorService.LOG.isDebugEnabled()) {
        PassExecutorService.log(progress, this, "toolWrappers: ",toolWrappers.size());
      }
      Consumer<? super ManagedHighlighterRecycler> withRecycler = invalidPsiRecycler -> {
        InspectionRunner.ApplyIncrementallyCallback applyIncrementallyCallback = (descriptors, holder, visitingPsiElement, shortName) -> {
          List<HighlightInfo> allInfos = descriptors.isEmpty() ? null : new ArrayList<>(descriptors.size());
          for (ProblemDescriptor descriptor : descriptors) {
            PsiElement descriptorPsiElement = descriptor.getPsiElement();
            if (LOG.isTraceEnabled()) {
              HighlightDisplayKey key = HighlightDisplayKey.find(holder.myToolWrapper.getShortName());
              LOG.trace("collectInformationWithProgress:applyIncrementallyCallback: toolId:" + holder.myToolWrapper.getShortName() + ": " +
                        descriptor + "; psi:" + descriptorPsiElement + "; isEnabled:" +
                        myProfileWrapper.getInspectionProfile().isToolEnabled(key, getFile()));
            }
            if (descriptorPsiElement != null) {
              createHighlightsForDescriptor(descriptor, descriptorPsiElement, holder.myToolWrapper, info -> {
                synchronized (holder.toolInfos) {
                  holder.toolInfos.add(info);
                  allInfos.add(info);
                }
              });
            }
          }
          myHighlightInfoUpdater.psiElementVisited(shortName, visitingPsiElement, ContainerUtil.notNullize(allInfos), getDocument(), holder.getFile(),
                                                   myProject, getHighlightingSession(), invalidPsiRecycler);
          if (allInfos != null) {
            fileInfos.addAll(allInfos);
          }
        };

        Consumer<InspectionRunner.InspectionContext> contextFinishedCallback = context -> {
          InspectionRunner.InspectionProblemHolder holder = context.holder();
          Collection<HighlightInfo> infos;
          synchronized (holder.toolInfos) {
            infos = new ArrayList<>(holder.toolInfos);
          }
          if (LOG.isTraceEnabled()) {
            LOG.trace("contextFinishedCallback: " + context.tool() + "; toolId:" + context.tool().getShortName() + "; " +
                      infos + "; " + context.elementsInside().size() + "/" + context.elementsOutside().size()+" elements");
          }
        };

        InspectionRunner runner = new InspectionRunner(getFile(), myRestrictRange, myPriorityRange, myInspectInjectedPsi, true,
                                                       isDumbMode(), progress, myIgnoreSuppressed, myProfileWrapper, mySuppressedElements);


        result.resultContexts = runner.inspect(toolWrappers,
                                               ((HighlightingSessionImpl)getHighlightingSession()).getMinimumSeverity(),
                                               true,
                                               applyIncrementallyCallback,
                                               contextFinishedCallback,
                                               wrapper -> dumbToolWrapperCondition.value(wrapper) &&
                                                          !wrapper.getTool().isSuppressedFor(getFile())
        );
        myInfos = fileInfos;
        result.injectedFragments = runner.getInjectedFragments();
      };
      if (myHighlightInfoUpdater instanceof HighlightInfoUpdaterImpl impl) {
        impl.runWithInvalidPsiRecycler(getHighlightingSession(), HighlightInfoUpdaterImpl.WhatTool.INSPECTION, withRecycler);
      }
      else {
        ManagedHighlighterRecycler.runWithRecycler(getHighlightingSession(), withRecycler);
      }
    }

    if (myHighlightInfoUpdater instanceof HighlightInfoUpdaterImpl impl) {
      Set<Pair<Object, PsiFile>> actualToolsRun = ContainerUtil.map2Set(result.resultContexts, context -> Pair.create(context.tool().getShortName(), context.psiFile()));

      Set<? extends String> inactiveSmartOnlyToolIds = dumbToolWrapperCondition.getInactiveToolWrapperIds();
      BiPredicate<? super Object, ? super PsiFile> keepToolIdPredicate = (toolId, psiFile) -> {
        if (!HighlightInfoUpdaterImpl.isInspectionToolId(toolId)) {
          return true;
        }

        if (actualToolsRun.contains(Pair.create(toolId, psiFile))) {
          return true;
        }
        // keep highlights from smart-only inspections in dumb mode
        return inactiveSmartOnlyToolIds.contains(toolId);
      };
      impl.removeHighlightsForObsoleteTools(getHighlightingSession(), result.injectedFragments, keepToolIdPredicate);
      impl.removeWarningsInsideErrors(result.injectedFragments, getDocument(), getHighlightingSession());  // must be the last
    }
  }

  private static final TextAttributes NONEMPTY_TEXT_ATTRIBUTES = new UnmodifiableTextAttributes(){
    @Override
    public boolean isEmpty() {
      return false;
    }
  };

  private HighlightInfo.Builder highlightInfoFromDescriptor(@NotNull ProblemDescriptor problemDescriptor,
                                                            @NotNull HighlightInfoType highlightInfoType,
                                                            @NotNull @NlsContexts.DetailedDescription String message,
                                                            @Nullable @NlsContexts.Tooltip String toolTip,
                                                            @NotNull PsiElement psiElement,
                                                            @NotNull List<? extends IntentionAction> quickFixes,
                                                            @NotNull HighlightDisplayKey key,
                                                            @Nullable EditorColorsScheme editorColorsScheme,
                                                            @NotNull SeverityRegistrar severityRegistrar) {
    TextRange textRange = ((ProblemDescriptorBase)problemDescriptor).getTextRange();
    if (textRange == null) {
      return null;
    }
    boolean isFileLevel = psiElement instanceof PsiFile && textRange.equals(psiElement.getTextRange());

    HighlightSeverity severity = highlightInfoType.getSeverity(psiElement);
    TextAttributesKey attributesKey = ((ProblemDescriptorBase)problemDescriptor).getEnforcedTextAttributes();
    if (problemDescriptor.getHighlightType() == ProblemHighlightType.GENERIC_ERROR_OR_WARNING && attributesKey == null) {
      attributesKey = myProfileWrapper.getInspectionProfile().getEditorAttributes(key.getShortName(), getFile());
    }
    TextAttributes attributes = attributesKey == null || editorColorsScheme == null || severity.getName().equals(attributesKey.getExternalName())
                                ? severityRegistrar.getTextAttributesBySeverity(severity)
                                : editorColorsScheme.getAttributes(attributesKey);
    if (attributesKey != null && (attributes == null || attributes.isEmpty())) {
      attributes = severityRegistrar.getCustomSeverityTextAttributes(attributesKey);
    }
    HighlightInfo.Builder b = HighlightInfo.newHighlightInfo(highlightInfoType)
      .range(psiElement, textRange.getStartOffset(), textRange.getEndOffset())
      .description(message)
      .severity(severity)
      .group(HighlightInfoUpdaterImpl.MANAGED_HIGHLIGHT_INFO_GROUP)
      ;
    if (toolTip != null) b.escapedToolTip(toolTip);
    if (HighlightSeverity.INFORMATION.equals(severity) && attributes == null && toolTip == null && !quickFixes.isEmpty()) {
      // Hack to avoid filtering this info out in HighlightInfoFilterImpl even though its attributes are empty.
      // But it has quick fixes, so it needs to be created.
      attributes = NONEMPTY_TEXT_ATTRIBUTES;
    }
    if (attributesKey != null) b.textAttributes(attributesKey);
    if (attributes != null) b.textAttributes(attributes);
    if (problemDescriptor.isAfterEndOfLine()) b.endOfLine();
    if (isFileLevel) b.fileLevelAnnotation();
    if (problemDescriptor.getProblemGroup() != null) b.problemGroup(problemDescriptor.getProblemGroup());

    return b;
  }

  private final InjectedLanguageManager myInjectedLanguageManager = InjectedLanguageManager.getInstance(myProject);
  private final PsiDocumentManager documentManager = PsiDocumentManager.getInstance(myProject);
  private final Set<Pair<TextRange, String>> emptyActionRegistered = Collections.synchronizedSet(new HashSet<>());

  private void createHighlightsForDescriptor(@NotNull ProblemDescriptor descriptor,
                                             @NotNull PsiElement psiElement,
                                             @NotNull LocalInspectionToolWrapper toolWrapper,
                                             @NotNull Consumer<? super HighlightInfo> infoProcessor) {
    String originalShortName = toolWrapper.getShortName();
    ApplicationManager.getApplication().assertIsNonDispatchThread();
    if (descriptor instanceof ProblemDescriptorWithReporterName name) {
      String reportingToolName = name.getReportingToolShortName();
      toolWrapper = (LocalInspectionToolWrapper)myProfileWrapper.getInspectionTool(reportingToolName, psiElement);
    }
    if (myIgnoreSuppressed && toolWrapper.getTool().isSuppressedFor(psiElement)) {
      registerSuppressedElements(psiElement, toolWrapper.getID(), toolWrapper.getAlternativeID(), mySuppressedElements);
      return;
    }

    PsiFile psiFile = psiElement.getContainingFile();

    HighlightDisplayKey displayKey = toolWrapper.getDisplayKey();
    if (displayKey == null) {
      LOG.error("getDisplayKey() is null for " + toolWrapper + " (" + toolWrapper.getTool() + " ; " + toolWrapper.getTool().getClass() + ")");
      return;
    }
    HighlightSeverity severity = myProfileWrapper.getErrorLevel(displayKey, psiFile).getSeverity();

    createHighlightsForDescriptor(emptyActionRegistered, psiFile, toolWrapper, originalShortName, severity, descriptor, psiElement, infoProcessor);
  }

  @Override
  protected void applyInformationWithProgress() {
    ((HighlightingSessionImpl)getHighlightingSession()).applyFileLevelHighlightsRequests();
  }

  private void createHighlightsForDescriptor(@NotNull Set<? super Pair<TextRange, String>> emptyActionRegistered,
                                             @NotNull PsiFile psiFile,
                                             @NotNull LocalInspectionToolWrapper toolWrapper,
                                             @NotNull String originalShortName,
                                             @NotNull HighlightSeverity severity,
                                             @NotNull ProblemDescriptor descriptor,
                                             @NotNull PsiElement element,
                                             @NotNull Consumer<? super HighlightInfo> outInfos) {
    SeverityRegistrar severityRegistrar = myProfileWrapper.getProfileManager().getSeverityRegistrar();
    HighlightInfoType level = ProblemDescriptorUtil.highlightTypeFromDescriptor(descriptor, severity, severityRegistrar);
    ProblemPresentation presentation = ProblemDescriptorUtil.renderDescriptor(descriptor, element, ProblemDescriptorUtil.NONE);
    String message = presentation.getDescription();

    ProblemGroup problemGroup = descriptor.getProblemGroup();
    String problemName = problemGroup != null ? problemGroup.getProblemName() : null;
    String shortName = problemName != null ? problemName : toolWrapper.getShortName();
    HighlightDisplayKey key = HighlightDisplayKey.find(shortName);
    InspectionProfile inspectionProfile = myProfileWrapper.getInspectionProfile();
    if (!inspectionProfile.isToolEnabled(key, psiFile)) {
      return;
    }

    HighlightInfoType type = new InspectionHighlightInfoType(level, element);
    String plainMessage = message.startsWith("<html>")
                          ? StringUtil.unescapeXmlEntities(XmlStringUtil.stripHtml(message).replaceAll("<[^>]*>", ""))
                            .replaceAll("&nbsp;|&#32;", " ")
                          : message;

    @NlsSafe String tooltip = null;
    if (descriptor.showTooltip()) {
      String rendered = presentation.getTooltip();
      tooltip = tooltips.intern(DaemonTooltipsUtil.getWrappedTooltip(rendered, shortName,
                                                                     showToolDescription(toolWrapper)));
    }
    List<IntentionAction> fixes = getQuickFixes(key, descriptor, emptyActionRegistered);
    HighlightInfo.Builder builder = highlightInfoFromDescriptor(descriptor, type, plainMessage, tooltip, element, fixes, key, getColorsScheme(), severityRegistrar);
    if (builder == null) {
      return;
    }
    registerQuickFixes(builder, fixes, shortName);

    PsiFile context = getTopLevelFileInBaseLanguage(element, psiFile.getProject());
    PsiFile myContext = getTopLevelFileInBaseLanguage(psiFile, psiFile.getProject());
    if (context != myContext) {
      String errorMessage = "Reported element " + element + " ("+element.getClass()+")"+
                            " is not from the file the inspection '" + shortName +
                            "' (" + toolWrapper.getTool().getClass() +
                            ") was invoked for. Message: '" + descriptor + "'.\nElement containing file: " +
                            PsiUtilCore.getVirtualFile(context) + "\nInspection invoked for the file: " + PsiUtilCore.getVirtualFile(myContext) + "\n";
      PluginException.logPluginError(LOG, errorMessage, null, toolWrapper.getTool().getClass());
    }
    boolean isInInjected = myInspectInjectedPsi && psiFile.getViewProvider() instanceof InjectedFileViewProvider;
    HighlightInfo info = builder.create();

    if (info == null || !UpdateHighlightersUtil.HighlightInfoPostFilters.accept(myProject, info)) {
      return;
    }
    info.setToolId(originalShortName); // toolId must be consistent with the tool which actually ran it
    info.setGroup(HighlightInfoUpdaterImpl.MANAGED_HIGHLIGHT_INFO_GROUP);
    if (isInInjected) {
      Document documentRange = documentManager.getDocument(psiFile);
      if (documentRange != null) {
        injectToHost(psiFile, documentRange, element, fixes, info, shortName, outInfos);
      }
    }
    else {
      outInfos.accept(info);
    }
  }

  private static void registerSuppressedElements(@NotNull PsiElement element,
                                                 @NotNull String id,
                                                 @Nullable String alternativeID,
                                                 @NotNull Map<? super String, Set<PsiElement>> outSuppressedElements) {
    outSuppressedElements.computeIfAbsent(id, __ -> new HashSet<>()).add(element);
    if (alternativeID != null) {
      outSuppressedElements.computeIfAbsent(alternativeID, __ -> new HashSet<>()).add(element);
    }
  }

  private void injectToHost(@NotNull PsiFile psiFile,
                            @NotNull Document documentRange,
                            @NotNull PsiElement element,
                            @NotNull List<? extends IntentionAction> fixes,
                            @NotNull HighlightInfo info,
                            @NotNull String shortName,
                            @NotNull Consumer<? super HighlightInfo> outInfos) {
    // todo we got to separate our "internal" prefixes/suffixes from user-defined ones
    // todo in the latter case the errors should be highlighted, otherwise not
    List<TextRange> editables = myInjectedLanguageManager.intersectWithAllEditableFragments(psiFile, new TextRange(info.startOffset, info.endOffset));
    for (TextRange editable : editables) {
      TextRange hostRange = ((DocumentWindow)documentRange).injectedToHost(editable);
      int start = hostRange.getStartOffset();
      int end = hostRange.getEndOffset();
      HighlightInfo.Builder builder = HighlightInfo.newHighlightInfo(info.type).range(element, start, end);
      String description = info.getDescription();
      if (description != null) {
        builder.description(description);
      }
      String toolTip = info.getToolTip();
      if (toolTip != null) {
        builder.escapedToolTip(toolTip);
      }
      if (start != end || info.startOffset == info.endOffset) {
        registerQuickFixes(builder, fixes, shortName);
        HighlightInfo patched = builder.createUnconditionally();
        patched.markFromInjection();
        patched.setToolId(info.getToolId());
        patched.setGroup(HighlightInfoUpdaterImpl.MANAGED_HIGHLIGHT_INFO_GROUP);
        outInfos.accept(patched);
      }
    }
  }

  private static PsiFile getTopLevelFileInBaseLanguage(@NotNull PsiElement element, @NotNull Project project) {
    PsiFile psiFile = InjectedLanguageManager.getInstance(project).getTopLevelFile(element);
    FileViewProvider viewProvider = psiFile.getViewProvider();
    return viewProvider.getPsi(viewProvider.getBaseLanguage());
  }


  private static final Interner<String> tooltips = Interner.createWeakInterner();

  private static boolean showToolDescription(@NotNull LocalInspectionToolWrapper tool) {
    String staticDescription = tool.getStaticDescription();
    return staticDescription == null || !staticDescription.isEmpty();
  }

  private static void registerQuickFixes(@NotNull HighlightInfo.Builder builder,
                                         @NotNull List<? extends IntentionAction> quickFixes,
                                         @NotNull String shortName) {
    HighlightDisplayKey key = HighlightDisplayKey.find(shortName);
    for (IntentionAction quickFix : quickFixes) {
      builder.registerFix(quickFix, null, HighlightDisplayKey.getDisplayNameByKey(key), null, key);
    }
  }

  private static @NotNull List<IntentionAction> getQuickFixes(@NotNull HighlightDisplayKey key,
                                                              @NotNull ProblemDescriptor descriptor,
                                                              @NotNull Set<? super Pair<TextRange, String>> emptyActionRegistered) {
    List<IntentionAction> result = new SmartList<>();
    boolean needEmptyAction = true;
    QuickFix<?>[] fixes = descriptor.getFixes();
    if (fixes != null && fixes.length != 0) {
      for (int k = 0; k < fixes.length; k++) {
        QuickFix<?> fix = fixes[k];
        if (fix == null) {
          throw new IllegalStateException("Inspection " + key + " returns null quick fix in its descriptor: " + descriptor + "; array: " + Arrays.toString(fixes));
        }
        result.add(QuickFixWrapper.wrap(descriptor, k));
        needEmptyAction = false;
      }
    }
    HintAction hintAction = descriptor instanceof ProblemDescriptorImpl impl ? impl.getHintAction() : null;
    if (hintAction != null) {
      result.add(hintAction);
      needEmptyAction = false;
    }
    if (((ProblemDescriptorBase)descriptor).getEnforcedTextAttributes() != null) {
      needEmptyAction = false;
    }
    if (needEmptyAction && emptyActionRegistered.add(Pair.create(((ProblemDescriptorBase)descriptor).getTextRange(), key.getShortName()))) {
      String displayNameByKey = HighlightDisplayKey.getDisplayNameByKey(key);
      LOG.assertTrue(displayNameByKey != null, key.toString());

      result.add(new EmptyIntentionAction(displayNameByKey));
    }
    return result;
  }

  private @NotNull List<LocalInspectionToolWrapper> getInspectionTools(@NotNull InspectionProfileWrapper profile) {
    List<InspectionToolWrapper<?, ?>> toolWrappers = profile.getInspectionProfile().getInspectionTools(getFile());

    if (LOG.isDebugEnabled()) {
      // this triggers heavy class loading of all inspections, do not run if DEBUG not enabled
      InspectionProfileWrapper.checkInspectionsDuplicates(toolWrappers);
    }

    List<LocalInspectionToolWrapper> enabled = new ArrayList<>();
    Set<String> projectTypes = ProjectTypeService.getProjectTypeIds(myProject);
    boolean isTests = ApplicationManager.getApplication().isUnitTestMode();

    for (InspectionToolWrapper<?, ?> toolWrapper : toolWrappers) {
      ProgressManager.checkCanceled();

      if (!isTests && !toolWrapper.isApplicable(projectTypes)) continue;

      HighlightDisplayKey key = toolWrapper.getDisplayKey();
      if (!profile.isToolEnabled(key, getFile())) continue;
      if (HighlightDisplayLevel.DO_NOT_SHOW.equals(profile.getErrorLevel(key, getFile()))) continue;
      LocalInspectionToolWrapper wrapper;
      if (toolWrapper instanceof LocalInspectionToolWrapper local) {
        wrapper = local;
      }
      else {
        wrapper = ((GlobalInspectionToolWrapper)toolWrapper).getSharedLocalInspectionToolWrapper();
        if (wrapper == null) continue;
      }
      String language = wrapper.getLanguage();
      if (language != null && Language.findLanguageByID(language) == null) {
        continue; // filter out at least unknown languages
      }

      try {
        if (myIgnoreSuppressed
            && wrapper.isApplicable(getFile().getLanguage())
            && wrapper.getTool().isSuppressedFor(getFile())) {
          // inspections that do not match file language are excluded later in InspectionRunner.inspect
          continue;
        }
      }
      catch (IndexNotReadyException ex) {
        continue;
      }
      enabled.add(wrapper);
    }
    return enabled;
  }

  @Override
  public @NotNull List<HighlightInfo> getInfos() {
    return Collections.unmodifiableList(myInfos);
  }

  @Override
  public boolean isDumbAware() {
    return Registry.is("ide.dumb.aware.inspections");
  }

  @Override
  public String toString() {
    return super.toString() + (myPriorityRange.equals(myRestrictRange) ? "" : "; priorityRange="+myPriorityRange);
  }

  private static final class InspectionHighlightInfoType extends HighlightInfoType.HighlightInfoTypeImpl {
    InspectionHighlightInfoType(@NotNull HighlightInfoType level, @NotNull PsiElement element) {
      super(level.getSeverity(element), level.getAttributesKey());
    }

    @Override
    public boolean isInspectionHighlightInfoType() {
      return true;
    }
  }

  private static final class DumbToolWrapperCondition implements Condition<LocalInspectionToolWrapper> {
    private final boolean myDumbMode;
    private final boolean myVirtualFileIsIndexable;
    private final Set<String> myInactiveIds = ConcurrentHashMap.newKeySet();

    private DumbToolWrapperCondition(boolean isDumbMode, boolean virtualFileIsIndexable) {
      myDumbMode = isDumbMode;
      myVirtualFileIsIndexable = virtualFileIsIndexable;
    }

    @Override
    public boolean value(LocalInspectionToolWrapper wrapper) {
      if (!myDumbMode && myVirtualFileIsIndexable) return true;

      LocalInspectionTool tool = wrapper.getTool();
      if (DumbService.isDumbAware(tool)) {
        return true;
      }

      myInactiveIds.add(tool.getShortName());
      return false;
    }

    private @NotNull Set<? extends String> getInactiveToolWrapperIds() {
      return myInactiveIds;
    }
  }
}
