// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.projectRoots.impl

import com.intellij.openapi.Disposable
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.application.PathManager.DEFAULT_EXT
import com.intellij.openapi.application.runWriteAction
import com.intellij.openapi.components.ComponentManagerEx
import com.intellij.openapi.components.State
import com.intellij.openapi.components.Storage
import com.intellij.openapi.extensions.ExtensionPointListener
import com.intellij.openapi.extensions.PluginDescriptor
import com.intellij.openapi.project.ProjectBundle
import com.intellij.openapi.projectRoots.ProjectJdkTable
import com.intellij.openapi.projectRoots.Sdk
import com.intellij.openapi.projectRoots.SdkType
import com.intellij.openapi.projectRoots.SdkTypeId
import com.intellij.openapi.util.Disposer
import com.intellij.platform.eel.EelDescriptor
import com.intellij.platform.workspace.jps.serialization.impl.JpsGlobalEntitiesSerializers
import com.intellij.platform.workspace.storage.InternalEnvironmentName
import com.intellij.serviceContainer.ComponentManagerImpl
import com.intellij.workspaceModel.ide.impl.getInternalEnvironmentName
import com.intellij.workspaceModel.ide.impl.legacyBridge.sdk.SdkTableBridgeImpl
import com.intellij.workspaceModel.ide.legacyBridge.sdk.SdkTableImplementationDelegate
import org.jdom.Element
import org.jetbrains.annotations.ApiStatus
import org.jetbrains.annotations.NonNls
import org.jetbrains.annotations.TestOnly

// This annotation is needed only for support of the "export settings" action
@State(name = "ProjectJdkTable", storages = [Storage(value = JpsGlobalEntitiesSerializers.SDK_FILE_NAME + DEFAULT_EXT)], presentableName = ProjectJdkTableImpl.PresentableNameGetter::class)
open class ProjectJdkTableImpl: ProjectJdkTable(), EnvironmentScopedSdkTableOps {

  private val delegate: SdkTableImplementationDelegate

  private val cachedProjectJdks: MutableMap<String, Sdk> = HashMap()

  init {
    val componentManager = ApplicationManager.getApplication() as ComponentManagerEx
    componentManager.registerService(SdkTableImplementationDelegate::class.java, SdkTableBridgeImpl::class.java,
                                     ComponentManagerImpl.fakeCorePluginDescriptor, false)
    delegate = SdkTableImplementationDelegate.getInstance()

    SdkType.EP_NAME.addExtensionPointListener(object : ExtensionPointListener<SdkType> {
      override fun extensionAdded(extension: SdkType, pluginDescriptor: PluginDescriptor) {
        loadSdkType(extension)
      }

      override fun extensionRemoved(extension: SdkType, pluginDescriptor: PluginDescriptor) {
        forgetSdkType(extension)
      }
    }, null)
  }

  override fun findJdk(name: String): Sdk? = delegate.findSdkByName(name)

  override fun findJdk(name: String, type: String): Sdk? {
    val sdk = findJdk(name)
    if (sdk != null) return sdk
    return getCachedJdkOrTryCreateJdkUsingSystemProperty(name, type, InternalEnvironmentName.Local)
  }

  @ApiStatus.Internal
  override fun findJdk(name: String, eelDescriptor: EelDescriptor): Sdk? {
    return delegate.findSdkByName(name, environmentName = eelDescriptor.machine.getInternalEnvironmentName())
  }

  @ApiStatus.Internal
  override fun findJdk(name: String, type: String, eelDescriptor: EelDescriptor): Sdk? {
    val environmentName = eelDescriptor.machine.getInternalEnvironmentName()
    val sdk = delegate.findSdkByName(name, environmentName)
    if (sdk != null) return sdk
    return getCachedJdkOrTryCreateJdkUsingSystemProperty(name, type, environmentName)
  }

  private fun getCachedJdkOrTryCreateJdkUsingSystemProperty(name: String, type: String, environmentName: InternalEnvironmentName): Sdk? {
    val uniqueName = "$type.$name"
    val sdk = cachedProjectJdks[uniqueName]
    if (sdk != null) return sdk
    return when (environmentName) {
      InternalEnvironmentName.Local -> {
        val sdkPath = System.getProperty("jdk.$name")
        if (sdkPath == null) return null

        val sdkType = SdkType.findByName(type)
        if (sdkType != null && sdkType.isValidSdkHome(sdkPath)) {
          val createdSdk = delegate.createSdk(name, sdkType, sdkPath)
          sdkType.setupSdkPaths(createdSdk)
          cachedProjectJdks[uniqueName] = createdSdk
          return createdSdk
        }
        return null
      }
      is InternalEnvironmentName.Custom -> null
    }
  }

  @ApiStatus.Internal
  override fun createSdk(name: String, sdkType: SdkTypeId, eelDescriptor: EelDescriptor): Sdk {
    return delegate.createSdk(name, sdkType, eelDescriptor.machine.getInternalEnvironmentName())
  }

  override fun getAllJdks(): Array<Sdk> = delegate.getAllSdks().toTypedArray()

  override fun getSdksOfType(type: SdkTypeId): List<Sdk> = delegate.getAllSdks().filter { it.sdkType.name == type.name }

  override fun addJdk(sdk: Sdk) {
    ApplicationManager.getApplication().assertWriteAccessAllowed()
    delegate.addNewSdk(sdk)
  }

  override fun removeJdk(sdk: Sdk) {
    ApplicationManager.getApplication().assertWriteAccessAllowed()
    delegate.removeSdk(sdk)
    if (sdk is Disposable) {
      Disposer.dispose(sdk)
    }
  }

  override fun updateJdk(originalSdk: Sdk, modifiedSdk: Sdk) {
    ApplicationManager.getApplication().assertWriteAccessAllowed()
    delegate.updateSdk(originalSdk, modifiedSdk)
  }

  override fun createSdk(name: String, sdkType: SdkTypeId): Sdk = delegate.createSdk(name, sdkType, null)

  @TestOnly
  override fun saveOnDisk(): Unit = delegate.saveOnDisk()

  override fun getDefaultSdkType(): SdkTypeId = UnknownSdkType.getInstance("")

  override fun getSdkTypeByName(sdkTypeName: String): SdkTypeId {
    return SdkType.getAllTypeList().firstOrNull { it.name == sdkTypeName } ?: UnknownSdkType.getInstance(sdkTypeName)
  }

  private fun loadSdkType(newSdkType: SdkType) {
    allJdks.forEach { sdk ->
      val sdkType = sdk.sdkType
      if (sdkType is UnknownSdkType && sdkType.getName() == newSdkType.name && sdk is SdkBridge) {
        runWriteAction {
          sdk.changeType(newSdkType, saveSdkAdditionalData(sdk))
        }
      }
    }
  }

  private fun forgetSdkType(extension: SdkType) {
    val sdkToRemove = hashSetOf<Sdk>()
    allJdks.forEach { sdk ->
      val sdkType = sdk.sdkType
      if (sdkType === extension) {
        if (sdk is SdkBridge) {
          sdk.changeType(UnknownSdkType.getInstance(sdkType.getName()), saveSdkAdditionalData(sdk))
        }
        else {
          //sdk was dynamically added by a plugin, so we can only remove it
          sdkToRemove.add(sdk)
        }
      }
    }
    sdkToRemove.forEach { sdk ->
      ApplicationManager.getApplication().messageBus.syncPublisher(JDK_TABLE_TOPIC).jdkRemoved(sdk)
      removeJdk(sdk)
    }
  }

  private fun saveSdkAdditionalData(sdk: Sdk): Element? {
    val additionalData = sdk.sdkAdditionalData
    if (additionalData == null) return null
    val additionalDataElement = Element(ELEMENT_ADDITIONAL)
    sdk.sdkType.saveAdditionalData(additionalData, additionalDataElement)
    return additionalDataElement
  }

  internal class PresentableNameGetter : State.NameGetter() {
    override fun get(): String = ProjectBundle.message("sdk.table.settings")
  }

  companion object {
    @NonNls
    private const val ELEMENT_ADDITIONAL: String = "additional"
  }
}
