// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.codeInsight.navigation.impl

import com.intellij.codeInsight.navigation.fileLocation
import com.intellij.codeInsight.navigation.fileStatusAttributes
import com.intellij.model.Pointer
import com.intellij.openapi.vfs.newvfs.VfsPresentationUtil
import com.intellij.platform.backend.navigation.NavigationRequest
import com.intellij.platform.backend.navigation.NavigationTarget
import com.intellij.platform.backend.presentation.TargetPresentation
import com.intellij.psi.PsiFile
import com.intellij.psi.createSmartPointer

internal class PsiFileNavigationTarget(
  private val psiFile: PsiFile
) : NavigationTarget {

  override fun createPointer(): Pointer<out NavigationTarget> = Pointer.delegatingPointer(
    psiFile.createSmartPointer(), ::PsiFileNavigationTarget
  )

  override fun computePresentation(): TargetPresentation {
    val project = psiFile.project

    var builder = TargetPresentation
      .builder(psiFile.name)
      .icon(psiFile.getIcon(0))
      .containerText(psiFile.parent?.virtualFile?.presentableUrl)

    val file = psiFile.virtualFile
               ?: return builder.presentation()

    builder = builder
      .backgroundColor(VfsPresentationUtil.getFileBackgroundColor(project, file))
      .presentableTextAttributes(fileStatusAttributes(project, file)) // apply file error and file status highlighting to file name

    val locationAndIcon = fileLocation(project, file)
                          ?: return builder.presentation()
    builder = builder.locationText(locationAndIcon.text, locationAndIcon.icon)

    return builder.presentation()
  }

  override fun navigationRequest(): NavigationRequest? {
    return psiFile.navigationRequest()
  }
}
