// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
@file:Suppress("PropertyName", "FunctionName", "unused", "ObjectPropertyName")
@file:ApiStatus.Experimental

package com.intellij.platform.syntax.util.runtime

import com.intellij.platform.syntax.Logger
import com.intellij.platform.syntax.SyntaxElementType
import com.intellij.platform.syntax.SyntaxElementTypeSet
import com.intellij.platform.syntax.element.SyntaxTokenTypes
import com.intellij.platform.syntax.parser.SyntaxTreeBuilder
import com.intellij.platform.syntax.parser.WhitespacesAndCommentsBinder
import com.intellij.platform.syntax.parser.WhitespacesBinders
import com.intellij.platform.syntax.syntaxElementTypeSetOf
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._AND_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._COLLAPSE_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._LEFT_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._LEFT_INNER_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._NONE_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._NOT_
import com.intellij.platform.syntax.util.runtime.Modifiers.Companion._UPPER_
import com.intellij.platform.syntax.util.runtime.SyntaxGeneratedParserRuntime.Hook
import com.intellij.platform.syntax.util.runtime.SyntaxGeneratedParserRuntime.Parser
import com.intellij.util.containers.LimitedPool
import org.jetbrains.annotations.ApiStatus
import org.jetbrains.annotations.Contract
import org.jetbrains.annotations.NonNls
import kotlin.math.min
import kotlin.jvm.JvmInline
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmField

/**
 * This class is a runtime engine for parsers generated by [Grammar-Kit](https://github.com/JetBrains/Grammar-Kit).
 * <br></br>
 * Here is the original [
 * GeneratedParserUtilBase](https://github.com/JetBrains/Grammar-Kit/blob/master/support/org/intellij/grammar/parser/GeneratedParserUtilBase.java) code.
 * <br></br>
 * <br></br>
 * CAUTION: Authorized personnel only. Do not modify or refactor.
 *
 * @author gregsh
 */
private const val MAX_VARIANTS_SIZE = 10000
private const val MAX_VARIANTS_TO_DISPLAY = 50
private const val MAX_ERROR_TOKEN_TEXT = 20
private const val MAX_CHILDREN_IN_TREE = 10

private const val INITIAL_VARIANTS_SIZE = 1000
private const val VARIANTS_POOL_SIZE = 10000
private const val FRAMES_POOL_SIZE = 500

// here's the new section API for compact parsers and less IntelliJ platform API exposure
@ApiStatus.Experimental
@JvmInline
/**
 * Encapsulates an integer mask to provide functionality using bitwise operations for controlling parser behavior.
 *
 * @author Maxim Medvedev
 */
value class Modifiers private constructor(private val mask: Int) {
  
  infix fun and(other: Modifiers): Modifiers = Modifiers(mask and other.mask)

  infix fun or(other: Modifiers): Modifiers = Modifiers(mask or other.mask)

  companion object {
    val _NONE_: Modifiers = Modifiers(0x0)
    val _COLLAPSE_: Modifiers = Modifiers(0x1)
    val _LEFT_: Modifiers = Modifiers(0x2)
    val _LEFT_INNER_: Modifiers = Modifiers(0x4)
    val _AND_: Modifiers = Modifiers(0x8)
    val _NOT_: Modifiers = Modifiers(0x10)
    val _UPPER_: Modifiers = Modifiers(0x20)
  }
}

@ApiStatus.Experimental
@JvmField
val TOKEN_ADVANCER: Parser = Parser { parserRuntime: SyntaxGeneratedParserRuntime, level: Int -> parserRuntime.advanceToken(level) }

@ApiStatus.Experimental
@JvmField
val TRUE_CONDITION: Parser = Parser { parserRuntime: SyntaxGeneratedParserRuntime, level: Int -> true }

@ApiStatus.Experimental
@JvmField
val LEFT_BINDER: Hook<WhitespacesAndCommentsBinder> = object : Hook<WhitespacesAndCommentsBinder> {
  override fun run(parserRuntime: SyntaxGeneratedParserRuntime, marker: SyntaxTreeBuilder.Marker?, param: WhitespacesAndCommentsBinder?): SyntaxTreeBuilder.Marker? {
    marker?.setCustomEdgeTokenBinders(param, null)
    return marker
  }
}

@ApiStatus.Experimental
@JvmField
val RIGHT_BINDER: Hook<WhitespacesAndCommentsBinder> = object : Hook<WhitespacesAndCommentsBinder> {
  override fun run(parserRuntime: SyntaxGeneratedParserRuntime, marker: SyntaxTreeBuilder.Marker?, param: WhitespacesAndCommentsBinder?): SyntaxTreeBuilder.Marker? {
    marker?.setCustomEdgeTokenBinders(null, param)
    return marker
  }
}

@ApiStatus.Experimental
@JvmField
val WS_BINDERS: Hook<Array<WhitespacesAndCommentsBinder?>> = object : Hook<Array<WhitespacesAndCommentsBinder?>> {
  override fun run(parserRuntime: SyntaxGeneratedParserRuntime, marker: SyntaxTreeBuilder.Marker?, param: Array<WhitespacesAndCommentsBinder?>?): SyntaxTreeBuilder.Marker? {
    marker?.setCustomEdgeTokenBinders(param!![0], param[1])
    return marker
  }
}

@ApiStatus.Experimental
@JvmField
val DUMMY_BLOCK: SyntaxElementType = SyntaxElementType("DUMMY_BLOCK")

@ApiStatus.Experimental
class SyntaxGeneratedParserRuntime(
  val syntaxBuilder: SyntaxTreeBuilder,
  private val maxRecursionDepth: Int,
  private val isCaseSensitive: Boolean,
  private val braces: Collection<BracePair>,
  internal val LOG: Logger,
  val parserUserState: ParserUserState?,
) {
  private val error: ErrorState = ErrorState()

  internal var parser: (SyntaxElementType, SyntaxGeneratedParserRuntime) -> Unit = { _, _ -> }
  internal val MAX_RECURSION_LEVEL: Int get() = maxRecursionDepth
  internal val isLanguageCaseSensitive get() = isCaseSensitive
  internal val errorState get() = error

  fun init(parse: (SyntaxElementType, SyntaxGeneratedParserRuntime) -> Unit, extendsSets: Array<SyntaxElementTypeSet> = emptyArray()) {
    parser = parse
    errorState.initState(this, extendsSets)
  }

  interface ParserUserState

  fun interface Parser {
    fun parse(parserRuntime: SyntaxGeneratedParserRuntime, level: Int): Boolean
  }

  fun interface Hook<T> {
    @Contract("_,null,_->null")
    fun run(parserRuntime: SyntaxGeneratedParserRuntime, marker: SyntaxTreeBuilder.Marker?, param: T?): SyntaxTreeBuilder.Marker?
  }

  internal data class HookBatch<T>(val hook: Hook<T>, val param: T, val level: Int) {
    fun process(parserRuntime: SyntaxGeneratedParserRuntime, marker: SyntaxTreeBuilder.Marker?): SyntaxTreeBuilder.Marker? {
      return hook.run(parserRuntime, marker, param)
    }
  }

  internal class MyList<E>(initialCapacity: Int) {
    private var arrayList: ArrayList<E> = ArrayList(initialCapacity)
    val size: Int get() = arrayList.size

    fun get(index: Int): E {
      return arrayList[index]
    }

    fun setSize(fromIndex: Int) {
      arrayList.subList(fromIndex, this.size).clear()
    }

    fun add(e: E): Boolean {
      val size = this.size
      if (size >= MAX_VARIANTS_SIZE) {
        setSize(MAX_VARIANTS_SIZE / 4)
      }
      return arrayList.add(e)
    }
  }

  class ErrorState() {
    internal var currentFrame: Frame? = null
    internal val variants: MyList<Variant> = MyList(INITIAL_VARIANTS_SIZE)
    internal val unexpected: MyList<Variant> = MyList(INITIAL_VARIANTS_SIZE / 10)

    internal var predicateCount: Int = 0
    internal var level: Int = 0
    internal var predicateSign: Boolean = true
    internal var suppressErrors: Boolean = false
    internal var hooks: ArrayDeque<HookBatch<*>> = ArrayDeque(8)

    internal var extendsSets: Array<SyntaxElementTypeSet> = emptyArray()
    internal var braces: Array<BracePair> = emptyArray()
    internal var altMode: Boolean = false

    internal val VARIANTS: LimitedPool<Variant> = LimitedPool(VARIANTS_POOL_SIZE) { Variant() }
    internal val FRAMES: LimitedPool<Frame> = LimitedPool(FRAMES_POOL_SIZE) { Frame() }

    fun getExpected(position: Int, expected: Boolean): String {
      val list = if (expected) variants else unexpected
      val strings = HashSet<String>(list.size)
      for (i in 0 until list.size) {
        val variant: Variant = list.get(i)
        if (position == variant.position) {
          val text: String = variant.`object`.toString()
          if (text.isNotEmpty()) {
            strings.add(text)
          }
        }
      }

      fun shouldPutInQuotes(ch: Char): Boolean {
        return ch != '$' && ch != '_' && ch != '<' && !ch.isLetter()
      }

      return when {
        strings.size > MAX_VARIANTS_TO_DISPLAY -> {
          strings
            .asSequence()
            .sorted()
            .take(MAX_VARIANTS_TO_DISPLAY)
            .joinToString(separator = ", ", postfix = " ${SyntaxRuntimeBundle.message("parsing.error.and.ellipsis")}") { s ->
              if (shouldPutInQuotes(s[0])) "'$s'" else s
            }
        }
        strings.size > 1 -> {
          val sorted = strings.sorted().map { s -> if (shouldPutInQuotes(s[0])) "'$s'" else s }
          val last = sorted.last()
          sorted
            .dropLast(1)
            .joinToString(separator = ", ", postfix = " ${SyntaxRuntimeBundle.message("parsing.error.or")} $last")
        }
        else -> {
          strings.singleOrNull()?.let {
            if (shouldPutInQuotes(it[0])) "'$it'" else it
          }.orEmpty()
        }
      }
    }

    fun clearVariants(frame: Frame?) {
      clearVariants(true, frame?.variantCount ?: 0)
      if (frame != null) frame.lastVariantAt = -1
    }

    fun clearVariants(expected: Boolean, start: Int) {
      val list: MyList<Variant> = if (expected) variants else unexpected
      if (start < 0 || start >= list.size) return
      var i = start
      val len: Int = list.size
      while (i < len) {
        VARIANTS.recycle(list.get(i))
        i++
      }
      list.setSize(start)
    }

    fun typeExtends(child: SyntaxElementType?, parent: SyntaxElementType?): Boolean {
      if (child === parent) return true
      return extendsSets.any { it.contains(child) && it.contains(parent) }
    }

    fun initState(util: SyntaxGeneratedParserRuntime, extendsSets: Array<SyntaxElementTypeSet>) {
      this.extendsSets = extendsSets
      this.braces = util.braces.toTypedArray()
    }
  }

  class Frame {
    var parentFrame: Frame? = null
      private set

    var elementType: SyntaxElementType? = null // Left public because this field is modified extensively in SqlParser

    var offset: Int = 0
      private set

    var position: Int = 0
      private set

    var level: Int = 0
      private set

    var modifiers: Modifiers = _NONE_ // Left public because this field can be modified in many different ways

    var name: @NonNls String? = null
      private set

    var variantCount: Int = 0
      internal set

    var errorReportedAt: Int = 0
      internal set

    var lastVariantAt: Int = 0
      internal set

    var leftMarker: SyntaxTreeBuilder.Marker? = null
      internal set

    fun init(
      builder: SyntaxTreeBuilder,
      state: ErrorState,
      level_: Int,
      modifiers_: Modifiers,
      elementType_: SyntaxElementType?,
      name_: String?,
    ): Frame {
      parentFrame = state.currentFrame
      elementType = elementType_

      offset = builder.currentOffset
      position = builder.rawTokenIndex()
      level = level_
      modifiers = modifiers_
      name = name_
      variantCount = state.variants.size
      errorReportedAt = -1
      lastVariantAt = -1

      leftMarker = null
      return this
    }

    fun resetErrorReportedAt() {
      errorReportedAt = -1
    }

    override fun toString(): @NonNls String {
      val mod = if (modifiers == _NONE_) "_NONE_, "
      else ((if ((modifiers and _COLLAPSE_) != _NONE_) "_CAN_COLLAPSE_, " else "") +
            (if ((modifiers and _LEFT_) != _NONE_) "_LEFT_, " else "") +
            (if ((modifiers and _LEFT_INNER_) != _NONE_) "_LEFT_INNER_, " else "") +
            (if ((modifiers and _AND_) != _NONE_) "_AND_, " else "") +
            (if ((modifiers and _NOT_) != _NONE_) "_NOT_, " else "") +
            (if ((modifiers and _UPPER_) != _NONE_) "_UPPER_, " else ""))
      return "{$offset:$position:$level, $errorReportedAt, $mod$elementType, $name}"
    }
  }

  internal class Variant {
    var position: Int = 0
    var `object`: Any? = null

    fun init(pos: Int, o: Any?): Variant {
      position = pos
      `object` = o
      return this
    }

    override fun toString(): String {
      return "<$position, $`object`>"
    }
  }

  data class BracePair(
    val myLeftBrace: SyntaxElementType?,
    val myRightBrace: SyntaxElementType?,
    private val myStructural: Boolean,
  )
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.advanceToken(level: Int): Boolean {
  if (syntaxBuilder.eof()) return false
  syntaxBuilder.advanceLexer()
  return true
}

@ApiStatus.Experimental
fun create_token_set_(vararg tokenTypes: SyntaxElementType): SyntaxElementTypeSet {
  return syntaxElementTypeSetOf(*tokenTypes)
}

//private val MAX_RECURSION_LEVEL = StringUtil.parseInt(System.getProperty("grammar.kit.gpub.max.level"), 1000)

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.eof(level: Int): Boolean {
  return syntaxBuilder.eof()
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.current_position_(): Int {
  return syntaxBuilder.rawTokenIndex()
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.recursion_guard_(level: Int, funcName: String): Boolean {
  if (level > MAX_RECURSION_LEVEL) {
    syntaxBuilder.mark().error(SyntaxRuntimeBundle.message("parsing.error.maximum.recursion.level.reached.in", MAX_RECURSION_LEVEL, funcName))
    return false
  }
  return true
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.empty_element_parsed_guard_(funcName: String, pos: Int): Boolean {
  if (pos == current_position_()) {
    // sometimes this is a correct situation, therefore no explicit marker
    syntaxBuilder.error(SyntaxRuntimeBundle.message("parsing.error.empty.element.parsed.in.at.offset", funcName, syntaxBuilder.currentOffset))
    return false
  }
  return true
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.invalid_left_marker_guard_(marker: SyntaxTreeBuilder.Marker?, funcName: String?): Boolean {
  //builder.error("Invalid left marker encountered in " + funcName_ +" at offset " + builder.getCurrentOffset());
  val goodMarker = marker != null // && ((LighterASTNode)marker).getTokenType() != TokenType.ERROR_ELEMENT;
  if (!goodMarker) return false
  return errorState.currentFrame != null
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.leftMarkerIs(type: SyntaxElementType?): Boolean {
  val lastDoneMarker = syntaxBuilder.lastDoneMarker
  return lastDoneMarker?.getNodeType() === type
}

private fun SyntaxGeneratedParserRuntime.consumeTokens(smart: Boolean, pin: Int, vararg tokens: SyntaxElementType?): Boolean {
  var result = true
  var pinned = false
  var i = 0
  val tokensLength = tokens.size
  while (i < tokensLength) {
    if (pin > 0 && i == pin) pinned = result
    if (result || pinned) {
      val fast = smart && i == 0
      tokens[i]?.let {
        if (!(if (fast) consumeTokenFast(it) else consumeToken(it))) {
          result = false
          if (pin < 0 || pinned) report_error_(errorState, false)
        }
      }
    }
    i++
  }
  return pinned || result
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokens(pin: Int, vararg token: SyntaxElementType?): Boolean {
  return consumeTokens(false, pin, *token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokensSmart(pin: Int, vararg token: SyntaxElementType?): Boolean {
  return consumeTokens(true, pin, *token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.parseTokens(pin: Int, vararg tokens: SyntaxElementType?): Boolean {
  return parseTokens(false, pin, *tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.parseTokensSmart(pin: Int, vararg tokens: SyntaxElementType?): Boolean {
  return parseTokens(true, pin, *tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.parseTokens(smart: Boolean, pin: Int, vararg tokens: SyntaxElementType?): Boolean {
  val marker: SyntaxTreeBuilder.Marker = syntaxBuilder.mark()
  val result: Boolean = consumeTokens(smart, pin, *tokens)
  if (!result) {
    marker.rollbackTo()
  }
  else {
    marker.drop()
  }
  return result
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenSmart(token: SyntaxElementType): Boolean {
  return consumeTokenFast(token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenSmart(token: String): Boolean {
  return consumeTokenFast(token)
}

@ApiStatus.Experimental
@Contract(mutates = "param1")
fun SyntaxGeneratedParserRuntime.consumeToken(token: SyntaxElementType): Boolean {
  addVariantSmart(token, true)
  return consumeTokenFast(token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenFast(token: SyntaxElementType): Boolean {
  if (nextTokenIsFast(token)) {
    syntaxBuilder.advanceLexer()
    return true
  }
  return false
}

@ApiStatus.Experimental
@JvmOverloads
fun SyntaxGeneratedParserRuntime.consumeToken(text: String, caseSensitive: Boolean = isLanguageCaseSensitive): Boolean {
  addVariantSmart(text, true)
  var count: Int = nextTokenIsFast(text, caseSensitive)
  if (count > 0) {
    while (count-- > 0) syntaxBuilder.advanceLexer()
    return true
  }
  return false
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenFast(text: String): Boolean {
  var count: Int = nextTokenIsFast(text, isLanguageCaseSensitive)
  if (count > 0) {
    while (count-- > 0) syntaxBuilder.advanceLexer()
    return true
  }
  return false
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeToken(vararg tokens: SyntaxElementType): Boolean {
  addVariantSmart(tokens, true)
  return consumeTokenFast(*tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenSmart(vararg tokens: SyntaxElementType): Boolean {
  return consumeTokenFast(*tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.consumeTokenFast(vararg tokens: SyntaxElementType): Boolean {
  if (nextTokenIsFast(*tokens)) {
    syntaxBuilder.advanceLexer()
    return true
  }
  return false
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsFast(token: SyntaxElementType?): Boolean {
  return syntaxBuilder.tokenType === token
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsFast(vararg tokens: SyntaxElementType): Boolean {
  val tokenType: SyntaxElementType? = syntaxBuilder.tokenType
  return tokens.indexOf(tokenType) >= 0
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsFast(tokens: SyntaxElementTypeSet): Boolean {
  return tokens.contains(syntaxBuilder.tokenType)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsSmart(token: SyntaxElementType?): Boolean {
  return nextTokenIsFast(token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsSmart(vararg tokens: SyntaxElementType): Boolean {
  return nextTokenIsFast(*tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIs(frameName: String?, vararg tokens: SyntaxElementType): Boolean {
  val track = !errorState.suppressErrors && errorState.predicateCount < 2 && errorState.predicateSign
  return if (!track) nextTokenIsFast(*tokens) else nextTokenIsSlow(frameName, *tokens)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsSlow(frameName: String?, vararg tokens: SyntaxElementType): Boolean {
  val tokenType: SyntaxElementType? = syntaxBuilder.tokenType
  if (!frameName.isNullOrEmpty()) {
    addVariantInner(errorState, errorState.currentFrame, syntaxBuilder.rawTokenIndex(), frameName)
  }
  else {
    for (token in tokens) {
      addVariant(errorState, token)
    }
  }
  if (tokenType == null) return false
  return tokens.indexOf(tokenType) != -1
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIs(token: SyntaxElementType): Boolean {
  if (!addVariantSmart(token, false)) return true
  return nextTokenIsFast(token)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIs(tokenText: String): Boolean {
  if (!addVariantSmart(tokenText, false)) return true
  return nextTokenIsFast(tokenText, isLanguageCaseSensitive) > 0
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsFast(tokenText: String): Boolean {
  return nextTokenIsFast(tokenText, isLanguageCaseSensitive) > 0
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.nextTokenIsFast(tokenText: String, caseSensitive: Boolean): Int {
  val sequence: CharSequence = syntaxBuilder.text
  val offset: Int = syntaxBuilder.currentOffset
  val endOffset = offset + tokenText.length
  val subSequence = sequence.subSequence(offset, min(endOffset.toDouble(), sequence.length.toDouble()).toInt())
  if (!subSequence.contentEquals(tokenText, caseSensitive)) return 0

  var count = 0
  while (true) {
    val nextOffset: Int = syntaxBuilder.rawTokenTypeStart(++count)
    if (nextOffset > endOffset) {
      return -count
    }
    else if (nextOffset == endOffset) {
      break
    }
  }
  return count
}

private fun SyntaxGeneratedParserRuntime.addVariantSmart(token: Any, force: Boolean): Boolean {
  syntaxBuilder.eof()
  if (!errorState.suppressErrors && errorState.predicateCount < 2) {
    addVariant(errorState, token)
  }
  return true
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.addVariant(text: String) {
  addVariant(errorState, text)
}

private fun SyntaxGeneratedParserRuntime.addVariant(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  o: Any,
) {
  syntaxBuilder.eof() // skip whitespaces
  addVariantInner(state, state.currentFrame, syntaxBuilder.rawTokenIndex(), o)
}

private fun addVariantInner(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  frame: SyntaxGeneratedParserRuntime.Frame?,
  pos: Int,
  o: Any?,
) {
  val variant: SyntaxGeneratedParserRuntime.Variant = state.VARIANTS.alloc().init(pos, o)
  if (state.predicateSign) {
    state.variants.add(variant)
    if (frame != null && frame.lastVariantAt < pos) {
      frame.lastVariantAt = pos
    }
  }
  else {
    state.unexpected.add(variant)
  }
}

private fun SyntaxGeneratedParserRuntime.wasAutoSkipped(steps: Int): Boolean {
  for (i in -1 downTo -steps) {
    if (!isWhitespaceOrComment(syntaxBuilder.rawLookup(i))) return false
  }
  return true
}

// simple enter/exit methods pair that doesn't require a frame object
@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.enter_section_(): SyntaxTreeBuilder.Marker {
  reportFrameError(errorState)
  errorState.level++
  return syntaxBuilder.mark()
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.exit_section_(
  marker: SyntaxTreeBuilder.Marker?,
  elementType: SyntaxElementType?,
  result: Boolean,
) {
  close_marker_impl_(errorState.currentFrame, marker, elementType, result)
  run_hooks_impl_(errorState, if (result) elementType else null)
  errorState.level--
}

// complex enter/exit methods pair with a frame object
@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.enter_section_(level: Int, modifiers: Modifiers, frameName: String?): SyntaxTreeBuilder.Marker {
  return enter_section_(level, modifiers, null, frameName)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.enter_section_(level: Int, modifiers: Modifiers): SyntaxTreeBuilder.Marker = enter_section_(level, modifiers, null, null)

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.enter_section_(level: Int, modifiers: Modifiers, elementType: SyntaxElementType?, frameName: String?): SyntaxTreeBuilder.Marker {
  reportFrameError(errorState)
  val marker: SyntaxTreeBuilder.Marker = syntaxBuilder.mark()
  enter_section_impl_(level, modifiers, elementType, frameName)
  return marker
}

private fun SyntaxGeneratedParserRuntime.enter_section_impl_(level: Int, modifiers: Modifiers, elementType: SyntaxElementType?, frameName: String?) {
  errorState.level++
  val frame: SyntaxGeneratedParserRuntime.Frame = errorState.FRAMES.alloc().init(syntaxBuilder, errorState, level, modifiers, elementType, frameName)
  if (((frame.modifiers and _LEFT_) or (frame.modifiers and _LEFT_INNER_)) != _NONE_) {
    val left: SyntaxTreeBuilder.Marker? = syntaxBuilder.lastDoneMarker
    if (invalid_left_marker_guard_(left, frameName)) {
      frame.leftMarker = left
    }
  }
  errorState.currentFrame = frame
  if ((modifiers and _AND_) != _NONE_) {
    if (errorState.predicateCount == 0 && !errorState.predicateSign) {
      throw AssertionError("Incorrect false predicate sign")
    }
    errorState.predicateCount++
  }
  else if ((modifiers and _NOT_) != _NONE_) {
    errorState.predicateSign = errorState.predicateCount != 0 && !errorState.predicateSign
    errorState.predicateCount++
  }
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.exit_section_(
  level: Int,
  marker: SyntaxTreeBuilder.Marker?,
  result: Boolean,
  pinned: Boolean,
  eatMore: Parser?,
) {
  exit_section_(level, marker, null, result, pinned, eatMore)
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.exit_section_(
  level: Int,
  marker: SyntaxTreeBuilder.Marker?,
  elementType: SyntaxElementType?,
  result: Boolean,
  pinned: Boolean,
  eatMore: Parser?,
) {
  val frame: SyntaxGeneratedParserRuntime.Frame? = errorState.currentFrame
  errorState.currentFrame = frame?.parentFrame
  val elementTypeToExit: SyntaxElementType? = frame?.elementType ?: elementType
  if (frame == null || level != frame.level) {
    LOG.error("Unbalanced error section: got $frame, expected level $level")
    if (frame != null) errorState.FRAMES.recycle(frame)
    close_marker_impl_(frame, marker, elementTypeToExit, result)
    return
  }

  close_frame_impl_(errorState, frame, marker, elementTypeToExit, result, pinned)
  exit_section_impl_(errorState, frame, elementTypeToExit, result, pinned, eatMore)
  run_hooks_impl_(errorState, if (pinned || result) elementTypeToExit else null)
  errorState.FRAMES.recycle(frame)
  errorState.level--
}

@ApiStatus.Experimental
fun <T> SyntaxGeneratedParserRuntime.register_hook_(hook: Hook<T>, param: T) {
  errorState.hooks.addLast(SyntaxGeneratedParserRuntime.HookBatch(hook, param, errorState.level))
}

private fun SyntaxGeneratedParserRuntime.run_hooks_impl_(state: SyntaxGeneratedParserRuntime.ErrorState, elementType: SyntaxElementType?) {
  if (state.hooks.isEmpty()) return
  val batch = state.hooks.last()
  var marker: SyntaxTreeBuilder.Marker? = if (elementType == null) null else syntaxBuilder.lastDoneMarker
  if (elementType != null && marker == null) {
    syntaxBuilder.mark().error(SyntaxRuntimeBundle.message("parsing.error.no.expected.done.marker.at.offset", syntaxBuilder.currentOffset))
  }
  while (batch.level >= state.level) {
    if (batch.level == state.level) {
      marker = batch.process(this, marker)
    }
    state.hooks.removeLast()
  }
}

private fun SyntaxGeneratedParserRuntime.exit_section_impl_(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  frame: SyntaxGeneratedParserRuntime.Frame,
  elementType: SyntaxElementType?,
  result: Boolean,
  pinned: Boolean,
  eatMoreParser: Parser?,
) {
  val initialPos: Int = syntaxBuilder.rawTokenIndex()
  replace_variants_with_name_(state, frame, elementType, result, pinned)
  val lastErrorPos = if (frame.lastVariantAt < 0) initialPos else frame.lastVariantAt
  if (!state.suppressErrors && eatMoreParser != null) {
    state.suppressErrors = true
    val eatMoreFlagOnce = !syntaxBuilder.eof() && eatMoreParser.parse(this, frame.level + 1)
    var eatMoreFlag = eatMoreFlagOnce || !result && frame.position == initialPos && lastErrorPos > frame.position

    val latestDoneMarker: SyntaxTreeBuilder.Marker? =
      if ((pinned || result) && (state.altMode || elementType != null) &&
          eatMoreFlagOnce) getLatestExtensibleDoneMarker(syntaxBuilder)
      else null
    // advance to the last error pos
    // skip tokens until lastErrorPos. parseAsTree might look better here...
    var parenCount = 0
    while ((eatMoreFlag || parenCount > 0) && syntaxBuilder.rawTokenIndex() < lastErrorPos) {
      val tokenType: SyntaxElementType? = syntaxBuilder.tokenType
      if (state.braces.isNotEmpty()) {
        if (tokenType === state.braces[0].myLeftBrace) parenCount++
        else if (tokenType === state.braces[0].myRightBrace) parenCount--
      }
      if (syntaxBuilder.rawTokenIndex() >= lastErrorPos) break
      TOKEN_ADVANCER.parse(this, frame.level + 1)
      eatMoreFlag = eatMoreParser.parse(this, frame.level + 1)
    }
    var errorReported = frame.errorReportedAt == initialPos || !result && frame.errorReportedAt >= frame.position
    if (errorReported || eatMoreFlag) {
      if (!errorReported) {
        errorReported = reportError(state, frame, false, true, true)
      }
      else if (eatMoreFlag) {
        TOKEN_ADVANCER.parse(this, frame.level + 1)
      }
      if (eatMoreParser.parse(this, frame.level + 1)) {
        parseAsTree(state, frame.level + 1, DUMMY_BLOCK, true, TOKEN_ADVANCER, eatMoreParser)
      }
    }
    else if (eatMoreFlagOnce || !result && frame.position != syntaxBuilder.rawTokenIndex() || frame.errorReportedAt > initialPos) {
      errorReported = reportError(state, frame, false, true, false)
    }
    else if (!result && pinned && frame.errorReportedAt < 0) {
      errorReported = reportError(state, frame, elementType != null, false, false)
    }
    // whitespace prefix makes the very first frame offset bigger than the marker start offset which is always 0
    if (latestDoneMarker != null && frame.position >= latestDoneMarker.getStartTokenIndex() && frame.position <= latestDoneMarker.getEndTokenIndex()) {
      extend_marker_impl(latestDoneMarker)
    }
    state.suppressErrors = false
    if (errorReported || result) {
      state.clearVariants(true, 0)
      state.clearVariants(false, 0)
      frame.lastVariantAt = -1
      var f: SyntaxGeneratedParserRuntime.Frame? = frame
      while (f != null && f.variantCount > 0) {
        f.variantCount = 0
        f = f.parentFrame
      }
    }
  }
  else if (!result && pinned && frame.errorReportedAt < 0) {
    // do not report if there are errors beyond the current position
    if (lastErrorPos == initialPos) {
      // do not force, inner recoverRoot might have skipped some tokens
      reportError(state, frame, elementType != null && (frame.modifiers and _UPPER_) == _NONE_, false, false)
    }
    else if (lastErrorPos > initialPos) {
      // set the error pos here as if it is reported for future reference
      frame.errorReportedAt = lastErrorPos
    }
  }
  // propagate errorReportedAt up the stack to avoid duplicate reporting
  state.currentFrame?.let { currentFrame ->
    if (currentFrame.errorReportedAt < frame.errorReportedAt) {
      currentFrame.errorReportedAt = frame.errorReportedAt
    }
    if (currentFrame.lastVariantAt < frame.lastVariantAt) {
      currentFrame.lastVariantAt = frame.lastVariantAt
    }
  }
}

private fun SyntaxGeneratedParserRuntime.close_frame_impl_(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  frame: SyntaxGeneratedParserRuntime.Frame,
  marker: SyntaxTreeBuilder.Marker?,
  elementType: SyntaxElementType?,
  result: Boolean,
  pinned: Boolean,
) {
  var marker: SyntaxTreeBuilder.Marker? = marker
  var elementType: SyntaxElementType? = elementType
  if (((frame.modifiers and _AND_) or (frame.modifiers and _NOT_)) != _NONE_) {
    val resetLastPos = !state.suppressErrors && frame.lastVariantAt < 0 && frame.position < syntaxBuilder.rawTokenIndex()
    close_marker_impl_(frame, marker, null, false)
    state.predicateCount--
    if ((frame.modifiers and _NOT_) != _NONE_) state.predicateSign = !state.predicateSign
    marker = if (elementType != null && marker != null && (result || pinned)) syntaxBuilder.mark() else null
    if (resetLastPos) frame.lastVariantAt = syntaxBuilder.rawTokenIndex()
  }
  if (elementType != null && marker != null) {
    if (result || pinned) {
      if ((frame.modifiers and _COLLAPSE_) != _NONE_) {
        val last: SyntaxTreeBuilder.Marker? = syntaxBuilder.lastDoneMarker
        if (last != null && last.getStartTokenIndex() == frame.position &&
            state.typeExtends(last.getNodeType(), elementType) &&
            wasAutoSkipped(syntaxBuilder.rawTokenIndex() - last.getEndTokenIndex())) {
          elementType = last.getNodeType()
          last.drop()
        }
      }
      if ((frame.modifiers and _UPPER_) != _NONE_) {
        marker.drop()
        var f: SyntaxGeneratedParserRuntime.Frame? = frame.parentFrame
        while (f != null) {
          if (f.elementType == null) {
            f = f.parentFrame
          }
          else {
            f.elementType = elementType
            break
          }
        }
      }
      else if ((frame.modifiers and _LEFT_INNER_) != _NONE_ && frame.leftMarker != null) {
        marker.done(elementType)
        frame.leftMarker?.let { extend_marker_impl(it) }
      }
      else if ((frame.modifiers and _LEFT_) != _NONE_ && frame.leftMarker != null) {
        marker.drop()
        frame.leftMarker?.precede()?.done(elementType)
      }
      else {
        if (frame.level == 0) syntaxBuilder.eof() // skip whitespaces

        marker.done(elementType)
      }
    }
    else {
      close_marker_impl_(frame, marker, null, false)
    }
  }
  else if (result || pinned) {
    marker?.drop()
    if ((frame.modifiers and _LEFT_INNER_) != _NONE_ && frame.leftMarker != null) {
      extend_marker_impl(frame.leftMarker!!)
    }
  }
  else {
    close_marker_impl_(frame, marker, null, false)
  }
}

private fun extend_marker_impl(marker: SyntaxTreeBuilder.Marker) {
  val precede: SyntaxTreeBuilder.Marker = marker.precede()
  val elementType: SyntaxElementType = marker.getNodeType()
  if (elementType === SyntaxTokenTypes.ERROR_ELEMENT) {
    precede.error(marker.getErrorMessage() ?: "")
  }
  else {
    precede.done(elementType)
  }
  marker.drop()
}

private fun close_marker_impl_(
  frame: SyntaxGeneratedParserRuntime.Frame?,
  marker: SyntaxTreeBuilder.Marker?,
  elementType: SyntaxElementType?,
  result: Boolean,
) {
  if (marker == null) return
  if (result) {
    if (elementType != null) {
      marker.done(elementType)
    }
    else {
      marker.drop()
    }
  }
  else {
    frame?.let {
      val position: Int = marker.getStartTokenIndex()
      if (frame.errorReportedAt > position) {
        frame.errorReportedAt = frame.parentFrame?.errorReportedAt ?: -1
      }
    }
    marker.rollbackTo()
  }
}

private fun SyntaxGeneratedParserRuntime.replace_variants_with_name_(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  frame: SyntaxGeneratedParserRuntime.Frame,
  elementType: SyntaxElementType?,
  result: Boolean,
  pinned: Boolean,
) {
  val initialPos: Int = syntaxBuilder.rawTokenIndex()
  val willFail = !result && !pinned
  if (willFail && initialPos == frame.position && frame.lastVariantAt == frame.position && frame.name != null && state.variants.size >= frame.variantCount + (if (elementType == null) 0 else 2)) {
    state.clearVariants(true, frame.variantCount)
    addVariantInner(state, frame, initialPos, frame.name)
  }
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.report_error_(result: Boolean): Boolean {
  if (!result) report_error_(errorState, false)
  return result
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.report_error_(state: SyntaxGeneratedParserRuntime.ErrorState, advance: Boolean) {
  val frame: SyntaxGeneratedParserRuntime.Frame? = state.currentFrame
  if (frame == null) {
    LOG.error("unbalanced enter/exit section call: got null")
    return
  }
  val position: Int = syntaxBuilder.rawTokenIndex()
  if (frame.errorReportedAt < position && frame.lastVariantAt > -1 && frame.lastVariantAt <= position) {
    reportError(state, frame, false, true, advance)
  }
}

private fun getLatestExtensibleDoneMarker(builder: SyntaxTreeBuilder): SyntaxTreeBuilder.Marker? {
  val marker: SyntaxTreeBuilder.Production = builder.productions.last()
  if (marker.isCollapsed()) return null
  return marker as? SyntaxTreeBuilder.Marker
}

private fun SyntaxGeneratedParserRuntime.reportError(
  state: SyntaxGeneratedParserRuntime.ErrorState,
  frame: SyntaxGeneratedParserRuntime.Frame,
  inner: Boolean,
  force: Boolean,
  advance: Boolean,
): Boolean {
  val position: Int = syntaxBuilder.rawTokenIndex()
  val expected: String = state.getExpected(position, true)
  if (!force && expected.isEmpty() && !advance) return false

  val actual: CharSequence? = syntaxBuilder.tokenText?.trim()
  val message: String
  if (expected.isEmpty()) {
    if (actual.isNullOrEmpty()) {
      message = SyntaxRuntimeBundle.message("parsing.error.unmatched.input")
    }
    else {
      message = SyntaxRuntimeBundle.message("parsing.error.unexpected", actual.crop(MAX_ERROR_TOKEN_TEXT, true))
    }
  }
  else {
    if (actual.isNullOrEmpty()) {
      message = SyntaxRuntimeBundle.message("parsing.error.expected", expected)
    }
    else {
      message = SyntaxRuntimeBundle.message("parsing.error.expected.got", expected, actual.crop(MAX_ERROR_TOKEN_TEXT, true))
    }
  }
  if (advance) {
    val mark: SyntaxTreeBuilder.Marker = syntaxBuilder.mark()
    TOKEN_ADVANCER.parse(this, frame.level + 1)
    mark.error(message)
  }
  else if (inner) {
    val latestDoneMarker: SyntaxTreeBuilder.Marker? = getLatestExtensibleDoneMarker(syntaxBuilder)
    syntaxBuilder.error(message)
    if (latestDoneMarker != null && frame.position >= latestDoneMarker.getStartTokenIndex() && frame.position <= latestDoneMarker.getEndTokenIndex()) {
      extend_marker_impl(latestDoneMarker)
    }
  }
  else {
    syntaxBuilder.error(message)
  }
  syntaxBuilder.eof() // skip whitespaces
  frame.errorReportedAt = syntaxBuilder.rawTokenIndex()
  return true
}

private fun SyntaxGeneratedParserRuntime.reportFrameError(state: SyntaxGeneratedParserRuntime.ErrorState) {
  if (state.currentFrame == null || state.suppressErrors) return
  val frame: SyntaxGeneratedParserRuntime.Frame? = state.currentFrame
  val pos: Int = syntaxBuilder.rawTokenIndex()
  if (frame != null && frame.errorReportedAt > pos) {
    // report error for previous unsuccessful frame
    val marker: SyntaxTreeBuilder.Marker? = syntaxBuilder.lastDoneMarker
    var endOffset = marker?.getEndTokenIndex() ?: (pos + 1)
    while (endOffset <= pos && isWhitespaceOrComment(syntaxBuilder.rawLookup(endOffset - pos))) endOffset++
    val inner = endOffset == pos
    syntaxBuilder.eof()
    reportError(state, frame, inner, true, false)
  }
}

private fun checkSiblings(
  chunkType: SyntaxElementType,
  parens: ArrayDeque<Pair<SyntaxTreeBuilder.Marker, SyntaxTreeBuilder.Marker?>>,
  siblings: ArrayDeque<Pair<SyntaxTreeBuilder.Marker, Int>>,
) {
  main@ while (!siblings.isEmpty()) {
    val parenPair = parens.firstOrNull()
    val rating: Int = siblings.first().second
    var count = 0
    for (pair in siblings) {
      if (pair.second != rating || parenPair != null && pair.first === parenPair.second) break@main
      if (++count >= MAX_CHILDREN_IN_TREE) {
        val parentMarker: SyntaxTreeBuilder.Marker = pair.first.precede()
        parentMarker.setCustomEdgeTokenBinders(WhitespacesBinders.greedyLeftBinder(), null)
        while (count-- > 0) {
          siblings.removeFirst()
        }
        parentMarker.done(chunkType)
        siblings.addFirst(Pair(parentMarker, rating + 1))
        continue@main
      }
    }
    break
  }
}

//This is a construction used in SQL and Rust parser utils. We extract it here to restrict access to lastVariantAt field in Frame class
@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.parseWithProtectedLastPos(level: Int, parser: Parser): Boolean {
  val prev = errorState.currentFrame?.lastVariantAt ?: -1
  val result = parser.parse(this, level)
  errorState.currentFrame?.lastVariantAt = prev
  return result
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.parseAsTree(
  state: SyntaxGeneratedParserRuntime.ErrorState, level: Int,
  chunkType: SyntaxElementType, checkBraces: Boolean,
  parser: Parser, eatMoreConditionParser: Parser,
): Boolean {
  val parens: ArrayDeque<Pair<SyntaxTreeBuilder.Marker, SyntaxTreeBuilder.Marker?>> = ArrayDeque(4)
  val siblings: ArrayDeque<Pair<SyntaxTreeBuilder.Marker, Int>> = ArrayDeque()
  var marker: SyntaxTreeBuilder.Marker? = null

  val lBrace: SyntaxElementType? = if (checkBraces && state.braces.isNotEmpty()) state.braces[0].myLeftBrace else null
  val rBrace: SyntaxElementType? = if (lBrace != null) state.braces[0].myRightBrace else null
  var totalCount = 0
  var tokenCount = 0
  if (lBrace != null) {
    var tokenIdx = -1
    while (syntaxBuilder.rawLookup(tokenIdx) === SyntaxTokenTypes.WHITE_SPACE) tokenIdx--
    val doneMarker = if (syntaxBuilder.rawLookup(tokenIdx) === lBrace) syntaxBuilder.lastDoneMarker else null
    if (doneMarker != null && doneMarker.getStartOffset() == syntaxBuilder.rawTokenTypeStart(tokenIdx) && doneMarker.getNodeType() === SyntaxTokenTypes.ERROR_ELEMENT) {
      parens.add(Pair(doneMarker.precede(), null))
    }
  }
  var c: Int = current_position_()
  while (true) {
    val tokenType: SyntaxElementType? = syntaxBuilder.tokenType
    if (lBrace != null && (tokenType === lBrace || tokenType === rBrace && !parens.isEmpty())) {
      if (marker != null) {
        marker.done(chunkType)
        siblings.addFirst(Pair(marker, 1))
        marker = null
        tokenCount = 0
      }
      if (tokenType === lBrace) {
        val prev = siblings.firstOrNull()
        parens.addFirst(Pair(syntaxBuilder.mark(), prev?.first))
      }
      checkSiblings(chunkType, parens, siblings)
      TOKEN_ADVANCER.parse(this, level)
      if (tokenType === rBrace) {
        val pair: Pair<SyntaxTreeBuilder.Marker, SyntaxTreeBuilder.Marker?> = parens.removeFirst()
        pair.first.done(chunkType)
        while (!siblings.isEmpty() && siblings.first().first != pair.second) {
          siblings.removeFirst()
        }
        siblings.addFirst(Pair(pair.first, 1))

        // drop all markers inside parens
        checkSiblings(chunkType, parens, siblings)
      }
    }
    else {
      if (marker == null) {
        marker = syntaxBuilder.mark()
        marker.setCustomEdgeTokenBinders(WhitespacesBinders.greedyLeftBinder(), null)
      }
      val result = (!parens.isEmpty() || eatMoreConditionParser.parse(this, level + 1)) &&
                   parser.parse(this, level + 1)
      if (result) {
        tokenCount++
        totalCount++
      }
      else {
        break
      }
    }

    if (tokenCount >= MAX_CHILDREN_IN_TREE) {
      marker?.let {
        marker.done(chunkType)
        siblings.addFirst(Pair(marker, 1))
      }
      checkSiblings(chunkType, parens, siblings)
      marker = null
      tokenCount = 0
    }
    if (!empty_element_parsed_guard_("parseAsTree", c)) break
    c = current_position_()
  }
  marker?.drop()
  for (pair in parens) {
    pair.first.drop()
  }
  return totalCount != 0
}

@ApiStatus.Experimental
fun SyntaxGeneratedParserRuntime.isWhitespaceOrComment(type: SyntaxElementType?): Boolean {
  return type != null && syntaxBuilder.isWhitespaceOrComment(type)
}

private fun CharSequence.crop(length: Int, addEllipses: Boolean): CharSequence {
  if (length >= this.length) return this
  return if (addEllipses) this.substring(0, length) + "..." else this.substring(0, length)
}
