// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.ide.todo

import com.intellij.codeWithMe.ClientId
import com.intellij.codeWithMe.asContextElement
import com.intellij.openapi.Disposable
import com.intellij.openapi.application.*
import com.intellij.openapi.progress.ProgressManager
import com.intellij.openapi.progress.blockingContextToIndicator
import com.intellij.openapi.util.Disposer
import com.intellij.openapi.util.registry.RegistryManager
import com.intellij.openapi.vfs.VirtualFile
import com.intellij.util.concurrency.annotations.RequiresBackgroundThread
import com.intellij.util.concurrency.annotations.RequiresReadLock
import com.intellij.util.ui.tree.TreeUtil
import kotlinx.coroutines.*
import kotlinx.coroutines.future.asCompletableFuture
import java.util.concurrent.CompletableFuture

private val ASYNC_BATCH_SIZE by lazy { RegistryManager.getInstance().get("ide.tree.ui.async.batch.size") }

internal class TodoTreeBuilderCoroutineHelper(private val treeBuilder: TodoTreeBuilder) : Disposable {
  private val scope = CoroutineScope(SupervisorJob())

  init {
    Disposer.register(treeBuilder, this)
  }

  override fun dispose() {
    scope.cancel()
  }

  fun scheduleCacheAndTreeUpdate(vararg constraints: ReadConstraint): CompletableFuture<*> {
    return scope.launch(Dispatchers.EDT + ClientId.current.asContextElement()) {
      treeBuilder.onUpdateStarted()
      constrainedReadAction(*constraints) {
        blockingContextToIndicator {
          treeBuilder.collectFiles()
        }
      }
      treeBuilder.onUpdateFinished()
    }.asCompletableFuture()
  }

  fun scheduleCacheValidationAndTreeUpdate() {
    scope.launch(Dispatchers.EDT + ClientId.current.asContextElement()) {
      val pathsToSelect = TreeUtil.collectSelectedUserObjects(treeBuilder.tree).stream()
      treeBuilder.tree.clearSelection()

      readAction {
        treeBuilder.validateCacheAndUpdateTree()
      }

      TreeUtil.promiseSelect(
        treeBuilder.tree,
        pathsToSelect.map { TodoTreeBuilder.getVisitorFor(it) },
      )
    }
  }

  fun scheduleUpdateTree(): CompletableFuture<*> {
    return scope.launch(Dispatchers.Default + ClientId.current.asContextElement()) {
      readActionBlocking {
        treeBuilder.updateVisibleTree()
      }
    }.asCompletableFuture()
  }

  fun scheduleMarkFilesAsDirtyAndUpdateTree(files: List<VirtualFile>) {
    scope.launch(Dispatchers.Default + ClientId.current.asContextElement()) {
      files.asSequence()
        .filter { it.isValid }
        .forEach { treeBuilder.markFileAsDirty(it) }

      readActionBlocking {
        treeBuilder.updateVisibleTree()
      }
    }
  }
}

@RequiresBackgroundThread
@RequiresReadLock
private fun TodoTreeBuilder.collectFiles() {
  ProgressManager.checkCanceled()
  clearCache()

  collectFiles {
    myFileTree.add(it.virtualFile)

    if (myFileTree.size() % ASYNC_BATCH_SIZE.asInteger() == 0) {
      validateCacheAndUpdateTree()
    }
  }

  validateCacheAndUpdateTree()
}

@RequiresBackgroundThread
@RequiresReadLock
private fun TodoTreeBuilder.validateCacheAndUpdateTree() {
  ProgressManager.checkCanceled()

  todoTreeStructure.validateCache()
  updateVisibleTree()
}

@RequiresBackgroundThread
@RequiresReadLock
private fun TodoTreeBuilder.updateVisibleTree() {
  if (isUpdatable) {
    if (hasDirtyFiles()) { // suppress redundant cache validations
      todoTreeStructure.validateCache()
    }
    model.invalidateAsync()
  }
}
