// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.diff.comparison

import com.intellij.util.diff.binarySearch

/** Returns the character (Unicode code point) at the specified index. */
internal fun CharSequence.codePointAt(index: Int): Int {
  val high = this[index]
  if (high.isHighSurrogate() && index + 1 < this.length) {
    val low = this[index + 1]
    if (low.isLowSurrogate()) {
      return Char.toCodePoint(high, low)
    }
  }
  return high.code
}

internal fun Int.charCount(): Int = if (this >= MIN_SUPPLEMENTARY_CODE_POINT) 2 else 1

/**
 * Note: this implementation is written only for bmp codepoints, not sure if it will work for all due to toChar() downcast
 */
internal fun Int.isAlphabetic(): Boolean {
  return when (toChar().category) {
    CharCategory.UPPERCASE_LETTER,
    CharCategory.LOWERCASE_LETTER,
    CharCategory.TITLECASE_LETTER,
    CharCategory.MODIFIER_LETTER,
    CharCategory.OTHER_LETTER,
    CharCategory.LETTER_NUMBER,
      -> true
    else -> isOtherAlphabetic()
  }
}

internal fun Int.isIdeographic(): Boolean =
  binarySearchInRanges(this, ideographicRangeStarts, ideographicRangeEnds)

internal fun Int.isBmpCodePoint(): Boolean = this ushr 16 == 0

internal fun Int.isHiraganaScript(): Boolean =
  binarySearchInRanges(this, hiraganaScriptRangeStarts, hiraganaScriptRangeEnds)

internal fun Int.isKatakanaScript(): Boolean =
  binarySearchInRanges(this, katakanaScriptRangeStarts, katakanaScriptRangeEnds)

internal fun Int.isThaiScript(): Boolean =
  binarySearchInRanges(this, thaiScriptRangeStarts, thaiScriptRangeEnds)

internal fun Int.isJavaneseScript(): Boolean =
  binarySearchInRanges(this, javaneseScriptRangeStarts, javaneseScriptRangeEnds)

internal fun Int.isDecimalDigit(): Boolean =
  binarySearchInRanges(this, decimalDigitRangeStarts, decimalDigitRangeEnds)

/** Converts a surrogate pair to a unicode code point. */
private fun Char.Companion.toCodePoint(high: Char, low: Char): Int =
  (((high - MIN_HIGH_SURROGATE) shl 10) or (low - MIN_LOW_SURROGATE)) + 0x10000

/** The minimum value of a supplementary code point, `\u0x10000`. */
private const val MIN_SUPPLEMENTARY_CODE_POINT: Int = 0x10000

private fun Int.isOtherAlphabetic(): Boolean =
  binarySearchInRanges(this, otherAlphabeticRangeStarts, otherAlphabeticRangeEnds)

private fun binarySearchInRanges(codepoint: Int, rangeStarts: IntArray, rangeEnds: IntArray): Boolean {
  val insertionPoint = rangeStarts.binarySearch(codepoint)
  return when {
    insertionPoint >= 0 -> true
    insertionPoint == -1 -> false
    else -> {
      codepoint <= rangeEnds[-(insertionPoint + 2)]
    }
  }
}

// Ranges extracted from https://www.unicode.org/Public/UCD/latest/ucd/PropList.txt (PropList-16.0.0.txt, 2024-05-31, 18:09:48 GMT)

private val otherAlphabeticRangeStarts = intArrayOf(
  0x0345, 0x0363, 0x05b0, 0x05bf, 0x05c1, 0x05c4, 0x05c7, 0x0610, 0x064b, 0x0659, 0x0670, 0x06d6, 0x06e1, 0x06e7, 0x06ed, 0x0711, 0x0730,
  0x07a6, 0x0816, 0x081b, 0x0825, 0x0829, 0x0897, 0x08d4, 0x08e3, 0x08f0, 0x0903, 0x093a, 0x093b, 0x093e, 0x0941, 0x0949, 0x094e, 0x0955,
  0x0962, 0x0981, 0x0982, 0x09be, 0x09c1, 0x09c7, 0x09cb, 0x09d7, 0x09e2, 0x0a01, 0x0a03, 0x0a3e, 0x0a41, 0x0a47, 0x0a4b, 0x0a51, 0x0a70,
  0x0a75, 0x0a81, 0x0a83, 0x0abe, 0x0ac1, 0x0ac7, 0x0ac9, 0x0acb, 0x0ae2, 0x0afa, 0x0b01, 0x0b02, 0x0b3e, 0x0b3f, 0x0b40, 0x0b41, 0x0b47,
  0x0b4b, 0x0b56, 0x0b57, 0x0b62, 0x0b82, 0x0bbe, 0x0bc0, 0x0bc1, 0x0bc6, 0x0bca, 0x0bd7, 0x0c00, 0x0c01, 0x0c04, 0x0c3e, 0x0c41, 0x0c46,
  0x0c4a, 0x0c55, 0x0c62, 0x0c81, 0x0c82, 0x0cbe, 0x0cbf, 0x0cc0, 0x0cc6, 0x0cc7, 0x0cca, 0x0ccc, 0x0cd5, 0x0ce2, 0x0cf3, 0x0d00, 0x0d02,
  0x0d3e, 0x0d41, 0x0d46, 0x0d4a, 0x0d57, 0x0d62, 0x0d81, 0x0d82, 0x0dcf, 0x0dd2, 0x0dd6, 0x0dd8, 0x0df2, 0x0e31, 0x0e34, 0x0e4d, 0x0eb1,
  0x0eb4, 0x0ebb, 0x0ecd, 0x0f71, 0x0f7f, 0x0f80, 0x0f8d, 0x0f99, 0x102b, 0x102d, 0x1031, 0x1032, 0x1038, 0x103b, 0x103d, 0x1056, 0x1058,
  0x105e, 0x1062, 0x1067, 0x1071, 0x1082, 0x1083, 0x1085, 0x1087, 0x108d, 0x108f, 0x109a, 0x109d, 0x1712, 0x1732, 0x1752, 0x1772, 0x17b6,
  0x17b7, 0x17be, 0x17c6, 0x17c7, 0x1885, 0x18a9, 0x1920, 0x1923, 0x1927, 0x1929, 0x1930, 0x1932, 0x1933, 0x1a17, 0x1a19, 0x1a1b, 0x1a55,
  0x1a56, 0x1a57, 0x1a58, 0x1a61, 0x1a62, 0x1a63, 0x1a65, 0x1a6d, 0x1a73, 0x1abf, 0x1acc, 0x1b00, 0x1b04, 0x1b35, 0x1b36, 0x1b3b, 0x1b3c,
  0x1b3d, 0x1b42, 0x1b43, 0x1b80, 0x1b82, 0x1ba1, 0x1ba2, 0x1ba6, 0x1ba8, 0x1bac, 0x1be7, 0x1be8, 0x1bea, 0x1bed, 0x1bee, 0x1bef, 0x1c24,
  0x1c2c, 0x1c34, 0x1c36, 0x1dd3, 0x24b6, 0x2de0, 0xa674, 0xa69e, 0xa802, 0xa80b, 0xa823, 0xa825, 0xa827, 0xa880, 0xa8b4, 0xa8c5, 0xa8ff,
  0xa926, 0xa947, 0xa952, 0xa980, 0xa983, 0xa9b4, 0xa9b6, 0xa9ba, 0xa9bc, 0xa9be, 0xa9e5, 0xaa29, 0xaa2f, 0xaa31, 0xaa33, 0xaa35, 0xaa43,
  0xaa4c, 0xaa4d, 0xaa7b, 0xaa7c, 0xaa7d, 0xaab0, 0xaab2, 0xaab7, 0xaabe, 0xaaeb, 0xaaec, 0xaaee, 0xaaf5, 0xabe3, 0xabe5, 0xabe6, 0xabe8,
  0xabe9, 0xfb1e, 0x10376, 0x10a01, 0x10a05, 0x10a0c, 0x10d24, 0x10d69, 0x10eab, 0x10efc, 0x11000, 0x11001, 0x11002, 0x11038, 0x11073,
  0x11080, 0x11082, 0x110b0, 0x110b3, 0x110b7, 0x110c2, 0x11100, 0x11127, 0x1112c, 0x1112d, 0x11145, 0x11180, 0x11182, 0x111b3, 0x111b6,
  0x111bf, 0x111ce, 0x111cf, 0x1122c, 0x1122f, 0x11232, 0x11234, 0x11237, 0x1123e, 0x11241, 0x112df, 0x112e0, 0x112e3, 0x11300, 0x11302,
  0x1133e, 0x11340, 0x11341, 0x11347, 0x1134b, 0x11357, 0x11362, 0x113b8, 0x113bb, 0x113c2, 0x113c5, 0x113c7, 0x113cc, 0x11435, 0x11438,
  0x11440, 0x11443, 0x11445, 0x114b0, 0x114b3, 0x114b9, 0x114ba, 0x114bb, 0x114bf, 0x114c1, 0x115af, 0x115b2, 0x115b8, 0x115bc, 0x115be,
  0x115dc, 0x11630, 0x11633, 0x1163b, 0x1163d, 0x1163e, 0x11640, 0x116ab, 0x116ac, 0x116ad, 0x116ae, 0x116b0, 0x1171d, 0x1171e, 0x1171f,
  0x11720, 0x11722, 0x11726, 0x11727, 0x1182c, 0x1182f, 0x11838, 0x11930, 0x11937, 0x1193b, 0x11940, 0x11942, 0x119d1, 0x119d4, 0x119da,
  0x119dc, 0x119e4, 0x11a01, 0x11a35, 0x11a39, 0x11a3b, 0x11a51, 0x11a57, 0x11a59, 0x11a8a, 0x11a97, 0x11c2f, 0x11c30, 0x11c38, 0x11c3e,
  0x11c92, 0x11ca9, 0x11caa, 0x11cb1, 0x11cb2, 0x11cb4, 0x11cb5, 0x11d31, 0x11d3a, 0x11d3c, 0x11d3f, 0x11d43, 0x11d47, 0x11d8a, 0x11d90,
  0x11d93, 0x11d95, 0x11d96, 0x11ef3, 0x11ef5, 0x11f00, 0x11f03, 0x11f34, 0x11f36, 0x11f3e, 0x11f40, 0x1611e, 0x1612a, 0x1612d, 0x16f4f,
  0x16f51, 0x16f8f, 0x16ff0, 0x1bc9e, 0x1e000, 0x1e008, 0x1e01b, 0x1e023, 0x1e026, 0x1e08f, 0x1e947, 0x1f130, 0x1f150, 0x1f170,
)

private val otherAlphabeticRangeEnds = intArrayOf(
  0x0345, 0x036f, 0x05bd, 0x05bf, 0x05c2, 0x05c5, 0x05c7, 0x061a, 0x0657, 0x065f, 0x0670, 0x06dc, 0x06e4, 0x06e8, 0x06ed, 0x0711, 0x073f,
  0x07b0, 0x0817, 0x0823, 0x0827, 0x082c, 0x0897, 0x08df, 0x08e9, 0x0902, 0x0903, 0x093a, 0x093b, 0x0940, 0x0948, 0x094c, 0x094f, 0x0957,
  0x0963, 0x0981, 0x0983, 0x09c0, 0x09c4, 0x09c8, 0x09cc, 0x09d7, 0x09e3, 0x0a02, 0x0a03, 0x0a40, 0x0a42, 0x0a48, 0x0a4c, 0x0a51, 0x0a71,
  0x0a75, 0x0a82, 0x0a83, 0x0ac0, 0x0ac5, 0x0ac8, 0x0ac9, 0x0acc, 0x0ae3, 0x0afc, 0x0b01, 0x0b03, 0x0b3e, 0x0b3f, 0x0b40, 0x0b44, 0x0b48,
  0x0b4c, 0x0b56, 0x0b57, 0x0b63, 0x0b82, 0x0bbf, 0x0bc0, 0x0bc2, 0x0bc8, 0x0bcc, 0x0bd7, 0x0c00, 0x0c03, 0x0c04, 0x0c40, 0x0c44, 0x0c48,
  0x0c4c, 0x0c56, 0x0c63, 0x0c81, 0x0c83, 0x0cbe, 0x0cbf, 0x0cc4, 0x0cc6, 0x0cc8, 0x0ccb, 0x0ccc, 0x0cd6, 0x0ce3, 0x0cf3, 0x0d01, 0x0d03,
  0x0d40, 0x0d44, 0x0d48, 0x0d4c, 0x0d57, 0x0d63, 0x0d81, 0x0d83, 0x0dd1, 0x0dd4, 0x0dd6, 0x0ddf, 0x0df3, 0x0e31, 0x0e3a, 0x0e4d, 0x0eb1,
  0x0eb9, 0x0ebc, 0x0ecd, 0x0f7e, 0x0f7f, 0x0f83, 0x0f97, 0x0fbc, 0x102c, 0x1030, 0x1031, 0x1036, 0x1038, 0x103c, 0x103e, 0x1057, 0x1059,
  0x1060, 0x1064, 0x106d, 0x1074, 0x1082, 0x1084, 0x1086, 0x108c, 0x108d, 0x108f, 0x109c, 0x109d, 0x1713, 0x1733, 0x1753, 0x1773, 0x17b6,
  0x17bd, 0x17c5, 0x17c6, 0x17c8, 0x1886, 0x18a9, 0x1922, 0x1926, 0x1928, 0x192b, 0x1931, 0x1932, 0x1938, 0x1a18, 0x1a1a, 0x1a1b, 0x1a55,
  0x1a56, 0x1a57, 0x1a5e, 0x1a61, 0x1a62, 0x1a64, 0x1a6c, 0x1a72, 0x1a74, 0x1ac0, 0x1ace, 0x1b03, 0x1b04, 0x1b35, 0x1b3a, 0x1b3b, 0x1b3c,
  0x1b41, 0x1b42, 0x1b43, 0x1b81, 0x1b82, 0x1ba1, 0x1ba5, 0x1ba7, 0x1ba9, 0x1bad, 0x1be7, 0x1be9, 0x1bec, 0x1bed, 0x1bee, 0x1bf1, 0x1c2b,
  0x1c33, 0x1c35, 0x1c36, 0x1df4, 0x24e9, 0x2dff, 0xa67b, 0xa69f, 0xa802, 0xa80b, 0xa824, 0xa826, 0xa827, 0xa881, 0xa8c3, 0xa8c5, 0xa8ff,
  0xa92a, 0xa951, 0xa952, 0xa982, 0xa983, 0xa9b5, 0xa9b9, 0xa9bb, 0xa9bd, 0xa9bf, 0xa9e5, 0xaa2e, 0xaa30, 0xaa32, 0xaa34, 0xaa36, 0xaa43,
  0xaa4c, 0xaa4d, 0xaa7b, 0xaa7c, 0xaa7d, 0xaab0, 0xaab4, 0xaab8, 0xaabe, 0xaaeb, 0xaaed, 0xaaef, 0xaaf5, 0xabe4, 0xabe5, 0xabe7, 0xabe8,
  0xabea, 0xfb1e, 0x1037a, 0x10a03, 0x10a06, 0x10a0f, 0x10d27, 0x10d69, 0x10eac, 0x10efc, 0x11000, 0x11001, 0x11002, 0x11045, 0x11074,
  0x11081, 0x11082, 0x110b2, 0x110b6, 0x110b8, 0x110c2, 0x11102, 0x1112b, 0x1112c, 0x11132, 0x11146, 0x11181, 0x11182, 0x111b5, 0x111be,
  0x111bf, 0x111ce, 0x111cf, 0x1122e, 0x11231, 0x11233, 0x11234, 0x11237, 0x1123e, 0x11241, 0x112df, 0x112e2, 0x112e8, 0x11301, 0x11303,
  0x1133f, 0x11340, 0x11344, 0x11348, 0x1134c, 0x11357, 0x11363, 0x113ba, 0x113c0, 0x113c2, 0x113c5, 0x113ca, 0x113cd, 0x11437, 0x1143f,
  0x11441, 0x11444, 0x11445, 0x114b2, 0x114b8, 0x114b9, 0x114ba, 0x114be, 0x114c0, 0x114c1, 0x115b1, 0x115b5, 0x115bb, 0x115bd, 0x115be,
  0x115dd, 0x11632, 0x1163a, 0x1163c, 0x1163d, 0x1163e, 0x11640, 0x116ab, 0x116ac, 0x116ad, 0x116af, 0x116b5, 0x1171d, 0x1171e, 0x1171f,
  0x11721, 0x11725, 0x11726, 0x1172a, 0x1182e, 0x11837, 0x11838, 0x11935, 0x11938, 0x1193c, 0x11940, 0x11942, 0x119d3, 0x119d7, 0x119db,
  0x119df, 0x119e4, 0x11a0a, 0x11a38, 0x11a39, 0x11a3e, 0x11a56, 0x11a58, 0x11a5b, 0x11a96, 0x11a97, 0x11c2f, 0x11c36, 0x11c3d, 0x11c3e,
  0x11ca7, 0x11ca9, 0x11cb0, 0x11cb1, 0x11cb3, 0x11cb4, 0x11cb6, 0x11d36, 0x11d3a, 0x11d3d, 0x11d41, 0x11d43, 0x11d47, 0x11d8e, 0x11d91,
  0x11d94, 0x11d95, 0x11d96, 0x11ef4, 0x11ef6, 0x11f01, 0x11f03, 0x11f35, 0x11f3a, 0x11f3f, 0x11f40, 0x16129, 0x1612c, 0x1612e, 0x16f4f,
  0x16f87, 0x16f92, 0x16ff1, 0x1bc9e, 0x1e006, 0x1e018, 0x1e021, 0x1e024, 0x1e02a, 0x1e08f, 0x1e947, 0x1f149, 0x1f169, 0x1f189,
)

private val ideographicRangeStarts = intArrayOf(
  0x3006, 0x3007, 0x3021, 0x3038, 0x3400, 0x4e00, 0xf900, 0xfa70, 0x16fe4, 0x17000, 0x18800, 0x18cff, 0x1b170, 0x20000, 0x2a700, 0x2b740,
  0x2b820, 0x2ceb0, 0x2ebf0, 0x2f800, 0x30000, 0x31350,
)

private val ideographicRangeEnds = intArrayOf(
  0x3006, 0x3007, 0x3029, 0x303a, 0x4dbf, 0x9fff, 0xfa6d, 0xfad9, 0x16fe4, 0x187f7, 0x18cd5, 0x18d08, 0x1b2fb, 0x2a6df, 0x2b739, 0x2b81d,
  0x2cea1, 0x2ebe0, 0x2ee5d, 0x2fa1d, 0x3134a, 0x323af,
)

// Ranges extracted from https://www.unicode.org/Public/UCD/latest/ucd/Scripts.txt (Scripts-16.0.0.txt, 2024-04-30, 21:48:40 GMT)

private val hiraganaScriptRangeStarts = intArrayOf(
  0x3041, 0x309d, 0x309f, 0x1b001, 0x1b132, 0x1b150, 0x1f200,
)

private val hiraganaScriptRangeEnds = intArrayOf(
  0x3096, 0x309e, 0x309f, 0x1b11f, 0x1b132, 0x1b152, 0x1f200,
)

private val katakanaScriptRangeStarts = intArrayOf(
  0x30a1, 0x30fd, 0x30ff, 0x31f0, 0x32d0, 0x3300, 0xff66, 0xff71, 0x1aff0, 0x1aff5, 0x1affd, 0x1b000, 0x1b120, 0x1b155, 0x1b164,
)

private val katakanaScriptRangeEnds = intArrayOf(
  0x30fa, 0x30fe, 0x30ff, 0x31ff, 0x32fe, 0x3357, 0xff6f, 0xff9d, 0x1aff3, 0x1affb, 0x1affe, 0x1b000, 0x1b122, 0x1b155, 0x1b167,
)

private val thaiScriptRangeStarts = intArrayOf(
  0x0e01, 0x0e31, 0x0e32, 0x0e34, 0x0e40, 0x0e46, 0x0e47, 0x0e4f, 0x0e50, 0x0e5a,
)

private val thaiScriptRangeEnds = intArrayOf(
  0x0e30, 0x0e31, 0x0e33, 0x0e3a, 0x0e45, 0x0e46, 0x0e4e, 0x0e4f, 0x0e59, 0x0e5b,
)


private val javaneseScriptRangeStarts = intArrayOf(
  0xa980, 0xa983, 0xa984, 0xa9b3, 0xa9b4, 0xa9b6, 0xa9ba, 0xa9bc, 0xa9be, 0xa9c1, 0xa9d0, 0xa9de,
)

private val javaneseScriptRangeEnds = intArrayOf(
  0xa982, 0xa983, 0xa9b2, 0xa9b3, 0xa9b5, 0xa9b9, 0xa9bb, 0xa9bd, 0xa9c0, 0xa9cd, 0xa9d9, 0xa9df,
)

private val decimalDigitRangeStarts = intArrayOf(
  0x0030, 0x0660, 0x06F0, 0x07C0, 0x0966, 0x09E6, 0x0A66, 0x0AE6, 0x0B66, 0x0BE6, 0x0C66, 0x0CE6, 0x0D66, 0x0DE6, 0x0E50, 0x0ED0, 0x0F20,
  0x1040, 0x1090, 0x17E0, 0x1810, 0x1946, 0x19D0, 0x1A80, 0x1A90, 0x1B50, 0x1BB0, 0x1C40, 0x1C50, 0xA620, 0xA8D0, 0xA900, 0xA9D0, 0xA9F0,
  0xAA50, 0xABF0, 0xFF10, 0x104A0, 0x10D30, 0x10D40, 0x11066, 0x110F0, 0x11136, 0x111D0, 0x112F0, 0x11450, 0x114D0, 0x11650, 0x116C0,
  0x116D0, 0x11730, 0x118E0, 0x11950, 0x11BF0, 0x11C50, 0x11D50, 0x11DA0, 0x11F50, 0x16130, 0x16A60, 0x16AC0, 0x16B50, 0x16D70, 0x1CCF0,
  0x1D7CE, 0x1E140, 0x1E2F0, 0x1E4F0, 0x1E5F1, 0x1E950, 0x1FBF0,
)

private val decimalDigitRangeEnds = intArrayOf(
  0x0039, 0x0669, 0x06F9, 0x07C9, 0x096F, 0x09EF, 0x0A6F, 0x0AEF, 0x0B6F, 0x0BEF, 0x0C6F, 0x0CEF, 0x0D6F, 0x0DEF, 0x0E59, 0x0ED9, 0x0F29,
  0x1049, 0x1099, 0x17E9, 0x1819, 0x194F, 0x19D9, 0x1A89, 0x1A99, 0x1B59, 0x1BB9, 0x1C49, 0x1C59, 0xA629, 0xA8D9, 0xA909, 0xA9D9, 0xA9F9,
  0xAA59, 0xABF9, 0xFF19, 0x104A9, 0x10D39, 0x10D49, 0x1106F, 0x110F9, 0x1113F, 0x111D9, 0x112F9, 0x11459, 0x114D9, 0x11659, 0x116C9,
  0x116E3, 0x11739, 0x118E9, 0x11959, 0x11BF9, 0x11C59, 0x11D59, 0x11DA9, 0x11F59, 0x16139, 0x16A69, 0x16AC9, 0x16B59, 0x16D79, 0x1CCF9,
  0x1D7FF, 0x1E149, 0x1E2F9, 0x1E4F9, 0x1E5FA, 0x1E959, 0x1FBF9,
)