// Copyright 2000-2024 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.openapi.vcs.actions;

import com.intellij.icons.AllIcons;
import com.intellij.openapi.LineNumberConstants;
import com.intellij.openapi.actionSystem.*;
import com.intellij.openapi.editor.Editor;
import com.intellij.openapi.editor.ex.EditorGutterComponentEx;
import com.intellij.openapi.project.DumbAware;
import com.intellij.openapi.project.DumbAwareAction;
import com.intellij.openapi.ui.popup.JBPopupFactory;
import com.intellij.openapi.ui.popup.ListPopup;
import com.intellij.openapi.vcs.FilePath;
import com.intellij.openapi.vcs.VcsBundle;
import com.intellij.openapi.vcs.annotate.FileAnnotation;
import com.intellij.openapi.vcs.annotate.TextAnnotationPresentation;
import com.intellij.openapi.vcs.history.VcsRevisionNumber;
import com.intellij.openapi.vcs.impl.AbstractVcsHelperImpl;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.vcsUtil.VcsUtil;
import org.jetbrains.annotations.*;

import java.util.List;

public final class ShowAnnotateOperationsPopup extends DumbAwareAction {
  @Override
  public void update(@NotNull AnActionEvent e) {
    List<AnAction> actions = getActions(e.getDataContext());
    e.getPresentation().setEnabled(actions != null);
  }

  @Override
  public @NotNull ActionUpdateThread getActionUpdateThread() {
    return ActionUpdateThread.EDT;
  }

  @Override
  public void actionPerformed(@NotNull AnActionEvent e) {
    List<AnAction> actions = getActions(e.getDataContext());
    if (actions == null) return;

    String title = getTemplatePresentation().getText();
    DefaultActionGroup group = new DefaultActionGroup(actions);
    ListPopup popup = JBPopupFactory.getInstance().
      createActionGroupPopup(title, group, e.getDataContext(), JBPopupFactory.ActionSelectionAid.NUMBERING, true);
    popup.showInBestPositionFor(e.getDataContext());
  }

  private static @Nullable List<AnAction> getActions(@NotNull DataContext context) {
    Editor editor = context.getData(CommonDataKeys.EDITOR);
    if (editor == null) return null;

    TextAnnotationPresentation presentation = AnnotateToggleAction.getAnnotationPresentation(editor);
    if (presentation == null) return null;

    int line = editor.getCaretModel().getLogicalPosition().line;
    List<AnAction> actions = presentation.getActions(line);

    FileAnnotation fileAnnotation = AnnotateToggleAction.getFileAnnotation(editor);
    if (fileAnnotation != null) {
      int annotationLine = presentation.getAnnotationLine(line);
      actions = ContainerUtil.prepend(actions, new ShowAffectedFilesAction(fileAnnotation, annotationLine));
    }

    return ContainerUtil.nullize(actions);
  }

  @ApiStatus.Internal
  public static class Group extends ActionGroup implements DumbAware {
    {
      getTemplatePresentation().setHideGroupIfEmpty(true);
    }

    @Override
    public AnAction @NotNull [] getChildren(@Nullable AnActionEvent e) {
      if (e == null) return AnAction.EMPTY_ARRAY;

      List<AnAction> actions = getActions(e.getDataContext());
      return actions != null ? actions.toArray(AnAction.EMPTY_ARRAY) : AnAction.EMPTY_ARRAY;
    }
  }

  private static final class ShowAffectedFilesAction extends DumbAwareAction {
    private final FileAnnotation myFileAnnotation;
    private final int myLine;

    private final FileAnnotation.RevisionChangesProvider myChangesProvider;
    private final VcsRevisionNumber myRevisionNumber;
    private final VirtualFile myFile;

    private ShowAffectedFilesAction(@NotNull FileAnnotation fileAnnotation, int line) {
      super(VcsBundle.messagePointer("action.ShowAffectedFilesAction.show.affected.files.text"), AllIcons.Actions.ListChanges);
      myFileAnnotation = fileAnnotation;
      myLine = line;

      myChangesProvider = fileAnnotation.getRevisionsChangesProvider();
      myRevisionNumber = fileAnnotation.getLineRevisionNumber(myLine);
      myFile = fileAnnotation.getFile();
    }

    @Override
    public @NotNull ActionUpdateThread getActionUpdateThread() {
      return ActionUpdateThread.BGT;
    }

    @Override
    public void update(@NotNull AnActionEvent e) {
      boolean visible = myChangesProvider != null && myFile != null;
      e.getPresentation().setVisible(visible);
      e.getPresentation().setEnabled(visible && myRevisionNumber != null);
    }

    @Override
    public void actionPerformed(@NotNull AnActionEvent e) {
      FilePath filePath = VcsUtil.getFilePath(myFile);
      AbstractVcsHelperImpl.loadAndShowCommittedChangesDetails(myFileAnnotation.getProject(), myRevisionNumber, filePath,
                                                               () -> myChangesProvider.getChangesIn(myLine));
    }
  }

  /**
   * @return Annotation line number (ex: to be passed into {@link FileAnnotation#getLineRevisionNumber(int)}).
   * <p>
   * Not to be confused with 'editor line'.
   */
  @CalledInAny
  public static int getAnnotationLineNumber(@NonNls DataContext dataContext) {
    return getAnnotationLineNumber(dataContext, false);
  }

  /**
   * @param approximate true if approximate location is acceptable (ex: when line was modified locally),
   *                    false if {@link LineNumberConstants#ABSENT_LINE_NUMBER} should be returned.
   * @return Annotation line number (ex: to be passed into {@link FileAnnotation#getLineRevisionNumber(int)}).
   * <p>
   * Not to be confused with 'editor line'.
   */
  @CalledInAny
  public static int getAnnotationLineNumber(@NonNls DataContext dataContext, boolean approximate) {
    Editor editor = dataContext.getData(CommonDataKeys.EDITOR);
    if (editor == null) return LineNumberConstants.ABSENT_LINE_NUMBER;

    TextAnnotationPresentation annotationPresentation = AnnotateToggleAction.getAnnotationPresentation(editor);
    if (annotationPresentation == null) return LineNumberConstants.ABSENT_LINE_NUMBER;

    Integer gutterLine = dataContext.getData(EditorGutterComponentEx.LOGICAL_LINE_AT_CURSOR);
    int editorLine = gutterLine != null ? gutterLine : editor.getCaretModel().getLogicalPosition().line;

    return annotationPresentation.getAnnotationLine(editorLine, approximate);
  }
}
