// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.vcs.commit

import com.intellij.openapi.Disposable
import com.intellij.openapi.project.Project
import com.intellij.openapi.vcs.VcsBundle.message
import com.intellij.openapi.vcs.changes.ChangesViewManager
import com.intellij.openapi.vcs.changes.ui.ChangesListView
import com.intellij.openapi.vcs.changes.ui.ChangesViewContentManager
import com.intellij.openapi.vcs.changes.ui.ChangesViewContentManager.Companion.LOCAL_CHANGES
import com.intellij.openapi.vcs.changes.ui.ChangesViewContentManager.Companion.getToolWindowFor
import com.intellij.openapi.vcs.changes.ui.ChangesViewContentManagerListener
import com.intellij.openapi.vcs.changes.ui.isCommitToolWindowShown
import com.intellij.openapi.wm.ToolWindow
import kotlinx.coroutines.CompletableDeferred
import org.jetbrains.annotations.ApiStatus

class ChangesViewCommitPanel @ApiStatus.Internal constructor(
  project: Project,
  private val changesView: ChangesListView,
) : ChangeListViewCommitPanel(project, changesView) {
  private var isHideToolWindowOnCommit = false

  init {
    ChangesViewCommitTabTitleUpdater(project).initSubscription(this)
  }

  override val isActive: Boolean get() = component.isVisible

  override fun activate(): Boolean {
    val toolWindow = getVcsToolWindow() ?: return false
    val contentManager = ChangesViewContentManager.getInstance(project)

    saveToolWindowState()
    changesView.isShowCheckboxes = true
    component.isVisible = true
    commitActionsPanel.isActive = true

    toolbar.updateActionsImmediately()

    contentManager.selectContent(LOCAL_CHANGES)
    toolWindow.activate({ commitMessage.requestFocusInMessage() }, false)
    return true
  }

  override fun deactivate(isOnCommit: Boolean) {
    if (isOnCommit && isHideToolWindowOnCommit) {
      getVcsToolWindow()?.hide(null)
    }

    clearToolWindowState()
    changesView.isShowCheckboxes = false
    component.isVisible = false
    commitActionsPanel.isActive = false

    toolbar.updateActionsImmediately()
  }

  private fun saveToolWindowState() {
    if (!isActive) {
      isHideToolWindowOnCommit = getVcsToolWindow()?.isVisible != true
    }
  }

  private fun clearToolWindowState() {
    isHideToolWindowOnCommit = false
  }

  private fun getVcsToolWindow(): ToolWindow? = getToolWindowFor(project, LOCAL_CHANGES)

  override suspend fun refreshChangesViewBeforeCommit() {
    val deferred = CompletableDeferred<Unit>()
    ChangesViewManager.getInstanceEx(project).scheduleRefresh { deferred.complete(Unit) }
    deferred.await()
  }
}

private class ChangesViewCommitTabTitleUpdater(private val project: Project) : ChangesViewContentManagerListener {
  fun initSubscription(disposable: Disposable) {
    project.messageBus.connect(disposable).subscribe(ChangesViewContentManagerListener.TOPIC, this)
  }

  override fun toolWindowMappingChanged() {
    val tabContent = ChangesViewContentManager.getInstance(project).findContent(LOCAL_CHANGES)
    if (tabContent != null) {
        tabContent.displayName = if (project.isCommitToolWindowShown) message("tab.title.commit") else message("local.changes.tab")
      }
  }
}
