using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

using JetBrains.Platform.DebuggerAgent.Model;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Platform.DebuggerAgent.Model
{
  
  
  public class DebuggerProcessAgentModel : RdExtBase {
    //fields
    //public fields
    [NotNull] public RdEndpoint<bool, CalculateProcessesResult> CalculateProcesses { get { return _CalculateProcesses; }}
    
    //private fields
    [NotNull] private readonly RdEndpoint<bool, CalculateProcessesResult> _CalculateProcesses;
    
    //primary constructor
    private DebuggerProcessAgentModel(
      [NotNull] RdEndpoint<bool, CalculateProcessesResult> calculateProcesses
    )
    {
      if (calculateProcesses == null) throw new ArgumentNullException("calculateProcesses");
      
      _CalculateProcesses = calculateProcesses;
      BindableChildren.Add(new KeyValuePair<string, object>("calculateProcesses", _CalculateProcesses));
    }
    //secondary constructor
    private DebuggerProcessAgentModel (
    ) : this (
      new RdEndpoint<bool, CalculateProcessesResult>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool, CalculateProcessesResult.Read, CalculateProcessesResult.Write)
    ) {}
    //statics
    
    
    
    protected override long SerializationHash => -3003142435538875350L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(InaccessibleProcessInfo.Read, InaccessibleProcessInfo.Write);
      serializers.Register(DesktopClrProcessInfo.Read, DesktopClrProcessInfo.Write);
      serializers.Register(CalculateProcessesResult.Read, CalculateProcessesResult.Write);
      serializers.RegisterEnum<PlatformKind>();
      
      serializers.RegisterToplevelOnce(typeof(DebuggerProcessAgentModel), DebuggerProcessAgentModel.RegisterDeclaredTypesSerializers);
    }
    
    public DebuggerProcessAgentModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("DebuggerProcessAgentModel (");
      using (printer.IndentCookie()) {
        printer.Print("calculateProcesses = "); _CalculateProcesses.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CalculateProcessesResult : IPrintable, IEquatable<CalculateProcessesResult> {
    //fields
    //public fields
    [NotNull] public List<RdProcessInfoBase> Processes {get; private set;}
    
    //private fields
    //primary constructor
    public CalculateProcessesResult(
      [NotNull] List<RdProcessInfoBase> processes
    )
    {
      if (processes == null) throw new ArgumentNullException("processes");
      
      Processes = processes;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CalculateProcessesResult> Read = (ctx, reader) => 
    {
      var processes = ReadRdProcessInfoBaseList(ctx, reader);
      return new CalculateProcessesResult(processes);
    };
    public static CtxReadDelegate<List<RdProcessInfoBase>> ReadRdProcessInfoBaseList = RdProcessInfoBase.Read.List();
    
    public static CtxWriteDelegate<CalculateProcessesResult> Write = (ctx, writer, value) => 
    {
      WriteRdProcessInfoBaseList(ctx, writer, value.Processes);
    };
    public static CtxWriteDelegate<List<RdProcessInfoBase>> WriteRdProcessInfoBaseList = RdProcessInfoBase.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CalculateProcessesResult) obj);
    }
    public bool Equals(CalculateProcessesResult other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Processes.SequenceEqual(other.Processes);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Collections.GetHashCode(Processes);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CalculateProcessesResult (");
      using (printer.IndentCookie()) {
        printer.Print("processes = "); Processes.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class DesktopClrProcessInfo : RdProcessInfoBase
   {
    //fields
    //public fields
    [NotNull] public string MainWindowTitle {get; private set;}
    [NotNull] public string User {get; private set;}
    public PlatformKind Platform {get; private set;}
    [NotNull] public List<string> LoadedRuntimes {get; private set;}
    
    //private fields
    //primary constructor
    public DesktopClrProcessInfo(
      [NotNull] string mainWindowTitle,
      [NotNull] string user,
      PlatformKind platform,
      [NotNull] List<string> loadedRuntimes,
      int id,
      [NotNull] string name
    ) : base (
      id,
      name
     ) 
    {
      if (mainWindowTitle == null) throw new ArgumentNullException("mainWindowTitle");
      if (user == null) throw new ArgumentNullException("user");
      if (loadedRuntimes == null) throw new ArgumentNullException("loadedRuntimes");
      
      MainWindowTitle = mainWindowTitle;
      User = user;
      Platform = platform;
      LoadedRuntimes = loadedRuntimes;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<DesktopClrProcessInfo> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var name = reader.ReadString();
      var mainWindowTitle = reader.ReadString();
      var user = reader.ReadString();
      var platform = (PlatformKind)reader.ReadInt();
      var loadedRuntimes = ReadStringList(ctx, reader);
      return new DesktopClrProcessInfo(mainWindowTitle, user, platform, loadedRuntimes, id, name);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static new CtxWriteDelegate<DesktopClrProcessInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      writer.Write(value.Name);
      writer.Write(value.MainWindowTitle);
      writer.Write(value.User);
      writer.Write((int)value.Platform);
      WriteStringList(ctx, writer, value.LoadedRuntimes);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((DesktopClrProcessInfo) obj);
    }
    public bool Equals(DesktopClrProcessInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return MainWindowTitle == other.MainWindowTitle && User == other.User && Platform == other.Platform && LoadedRuntimes.SequenceEqual(other.LoadedRuntimes) && Id == other.Id && Name == other.Name;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + MainWindowTitle.GetHashCode();
        hash = hash * 31 + User.GetHashCode();
        hash = hash * 31 + (int) Platform;
        hash = hash * 31 + Collections.GetHashCode(LoadedRuntimes);
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("DesktopClrProcessInfo (");
      using (printer.IndentCookie()) {
        printer.Print("mainWindowTitle = "); MainWindowTitle.PrintEx(printer); printer.Println();
        printer.Print("user = "); User.PrintEx(printer); printer.Println();
        printer.Print("platform = "); Platform.PrintEx(printer); printer.Println();
        printer.Print("loadedRuntimes = "); LoadedRuntimes.PrintEx(printer); printer.Println();
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class InaccessibleProcessInfo : RdProcessInfoBase
   {
    //fields
    //public fields
    //private fields
    //primary constructor
    public InaccessibleProcessInfo(
      int id,
      [NotNull] string name
    ) : base (
      id,
      name
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<InaccessibleProcessInfo> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var name = reader.ReadString();
      return new InaccessibleProcessInfo(id, name);
    };
    
    public static new CtxWriteDelegate<InaccessibleProcessInfo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      writer.Write(value.Name);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((InaccessibleProcessInfo) obj);
    }
    public bool Equals(InaccessibleProcessInfo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && Name == other.Name;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("InaccessibleProcessInfo (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum PlatformKind {
    X86,
    X64,
    Unknown
  }
  
  
  public abstract class RdProcessInfoBase {
    //fields
    //public fields
    public int Id {get; private set;}
    [NotNull] public string Name {get; private set;}
    
    //private fields
    //primary constructor
    protected RdProcessInfoBase(
      int id,
      [NotNull] string name
    )
    {
      if (name == null) throw new ArgumentNullException("name");
      
      Id = id;
      Name = name;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdProcessInfoBase> Read = Polymorphic<RdProcessInfoBase>.ReadAbstract(RdProcessInfoBase_Unknown.Read);
    
    public static CtxWriteDelegate<RdProcessInfoBase> Write = Polymorphic<RdProcessInfoBase>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class RdProcessInfoBase_Unknown : RdProcessInfoBase
   {
    //fields
    //public fields
    //private fields
    //primary constructor
    public RdProcessInfoBase_Unknown(
      int id,
      [NotNull] string name
    ) : base (
      id,
      name
     ) 
    {
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<RdProcessInfoBase_Unknown> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var name = reader.ReadString();
      return new RdProcessInfoBase_Unknown(id, name);
    };
    
    public static new CtxWriteDelegate<RdProcessInfoBase_Unknown> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      writer.Write(value.Name);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdProcessInfoBase_Unknown) obj);
    }
    public bool Equals(RdProcessInfoBase_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && Name == other.Name;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdProcessInfoBase_Unknown (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
