using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.Loggers
{
  
  
  [JetBrains.Application.ShellComponent]
  public class LoggerModel : RdExtBase {
    //fields
    //public fields
    [NotNull] public ISource<LogEntry> Message { get { return _Message; }}
    [NotNull] public IRdProperty<string> TestLogDir { get { return _TestLogDir; }}
    [NotNull] public IRdProperty<List<string>> TraceCategories { get { return _TraceCategories; }}
    
    //private fields
    [NotNull] private readonly RdSignal<LogEntry> _Message;
    [NotNull] private readonly RdProperty<string> _TestLogDir;
    [NotNull] private readonly RdProperty<List<string>> _TraceCategories;
    
    //primary constructor
    private LoggerModel(
      [NotNull] RdSignal<LogEntry> message,
      [NotNull] RdProperty<string> testLogDir,
      [NotNull] RdProperty<List<string>> traceCategories
    )
    {
      if (message == null) throw new ArgumentNullException("message");
      if (testLogDir == null) throw new ArgumentNullException("testLogDir");
      if (traceCategories == null) throw new ArgumentNullException("traceCategories");
      
      _Message = message;
      _TestLogDir = testLogDir;
      _TraceCategories = traceCategories;
      _TestLogDir.OptimizeNested = true;
      _TraceCategories.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("message", _Message));
      BindableChildren.Add(new KeyValuePair<string, object>("testLogDir", _TestLogDir));
      BindableChildren.Add(new KeyValuePair<string, object>("traceCategories", _TraceCategories));
    }
    //secondary constructor
    private LoggerModel (
    ) : this (
      new RdSignal<LogEntry>(LogEntry.Read, LogEntry.Write),
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<List<string>>(ReadStringList, WriteStringList)
    ) {}
    //statics
    
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    
    protected override long SerializationHash => 8141100790410599711L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(LogEntry.Read, LogEntry.Write);
      serializers.RegisterEnum<LogEntryType>();
      
      serializers.RegisterToplevelOnce(typeof(JetBrains.Rider.Model.IdeRoot), JetBrains.Rider.Model.IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    public LoggerModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("LoggerModel (");
      using (printer.IndentCookie()) {
        printer.Print("message = "); _Message.PrintEx(printer); printer.Println();
        printer.Print("testLogDir = "); _TestLogDir.PrintEx(printer); printer.Println();
        printer.Print("traceCategories = "); _TraceCategories.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class LogEntry : IPrintable, IEquatable<LogEntry> {
    //fields
    //public fields
    [NotNull] public string LogMessage {get; private set;}
    [NotNull] public string LogMessageWithSensitiveData {get; private set;}
    public LogEntryType Severity {get; private set;}
    
    //private fields
    //primary constructor
    public LogEntry(
      [NotNull] string logMessage,
      [NotNull] string logMessageWithSensitiveData,
      LogEntryType severity
    )
    {
      if (logMessage == null) throw new ArgumentNullException("logMessage");
      if (logMessageWithSensitiveData == null) throw new ArgumentNullException("logMessageWithSensitiveData");
      
      LogMessage = logMessage;
      LogMessageWithSensitiveData = logMessageWithSensitiveData;
      Severity = severity;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<LogEntry> Read = (ctx, reader) => 
    {
      var logMessage = reader.ReadString();
      var logMessageWithSensitiveData = reader.ReadString();
      var severity = (LogEntryType)reader.ReadInt();
      return new LogEntry(logMessage, logMessageWithSensitiveData, severity);
    };
    
    public static CtxWriteDelegate<LogEntry> Write = (ctx, writer, value) => 
    {
      writer.Write(value.LogMessage);
      writer.Write(value.LogMessageWithSensitiveData);
      writer.Write((int)value.Severity);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((LogEntry) obj);
    }
    public bool Equals(LogEntry other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return LogMessage == other.LogMessage && LogMessageWithSensitiveData == other.LogMessageWithSensitiveData && Severity == other.Severity;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + LogMessage.GetHashCode();
        hash = hash * 31 + LogMessageWithSensitiveData.GetHashCode();
        hash = hash * 31 + (int) Severity;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("LogEntry (");
      using (printer.IndentCookie()) {
        printer.Print("logMessage = "); LogMessage.PrintEx(printer); printer.Println();
        printer.Print("logMessageWithSensitiveData = "); LogMessageWithSensitiveData.PrintEx(printer); printer.Println();
        printer.Print("severity = "); Severity.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum LogEntryType {
    ERROR,
    WARN,
    INFO,
    VERBOSE,
    TRACE
  }
}
