using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model.Notifications
{
  
  
  [JetBrains.Application.ShellComponent]
  public class RdNotificationsModel : RdExtBase {
    //fields
    //public fields
    [NotNull] public ISource<RdNotificationEntry> Notification { get { return _Notification; }}
    [NotNull] public IRdCall<RdMessageBoxModel, RdMessageBoxResponse> MessageBox { get { return _MessageBox; }}
    [NotNull] public IViewableMap<int, RdBackgroundTask> Tasks { get { return _Tasks; }}
    
    //private fields
    [NotNull] private readonly RdSignal<RdNotificationEntry> _Notification;
    [NotNull] private readonly RdCall<RdMessageBoxModel, RdMessageBoxResponse> _MessageBox;
    [NotNull] private readonly RdMap<int, RdBackgroundTask> _Tasks;
    
    //primary constructor
    private RdNotificationsModel(
      [NotNull] RdSignal<RdNotificationEntry> notification,
      [NotNull] RdCall<RdMessageBoxModel, RdMessageBoxResponse> messageBox,
      [NotNull] RdMap<int, RdBackgroundTask> tasks
    )
    {
      if (notification == null) throw new ArgumentNullException("notification");
      if (messageBox == null) throw new ArgumentNullException("messageBox");
      if (tasks == null) throw new ArgumentNullException("tasks");
      
      _Notification = notification;
      _MessageBox = messageBox;
      _Tasks = tasks;
      BindableChildren.Add(new KeyValuePair<string, object>("notification", _Notification));
      BindableChildren.Add(new KeyValuePair<string, object>("messageBox", _MessageBox));
      BindableChildren.Add(new KeyValuePair<string, object>("tasks", _Tasks));
    }
    //secondary constructor
    private RdNotificationsModel (
    ) : this (
      new RdSignal<RdNotificationEntry>(RdNotificationEntry.Read, RdNotificationEntry.Write),
      new RdCall<RdMessageBoxModel, RdMessageBoxResponse>(RdMessageBoxModel.Read, RdMessageBoxModel.Write, ReadRdMessageBoxResponse, WriteRdMessageBoxResponse),
      new RdMap<int, RdBackgroundTask>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt, RdBackgroundTask.Read, RdBackgroundTask.Write)
    ) {}
    //statics
    
    public static CtxReadDelegate<RdMessageBoxResponse> ReadRdMessageBoxResponse = new CtxReadDelegate<RdMessageBoxResponse>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadEnum<RdMessageBoxResponse>);
    
    public static CtxWriteDelegate<RdMessageBoxResponse> WriteRdMessageBoxResponse = new CtxWriteDelegate<RdMessageBoxResponse>(JetBrains.Platform.RdFramework.Impl.Serializers.WriteEnum<RdMessageBoxResponse>);
    
    protected override long SerializationHash => 1161365606241707548L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(RdNotificationEntry.Read, RdNotificationEntry.Write);
      serializers.RegisterEnum<RdMessageBoxResponse>();
      serializers.RegisterEnum<RdMessageBoxType>();
      serializers.Register(RdMessageBoxModel.Read, RdMessageBoxModel.Write);
      serializers.Register(RdBackgroundTask.Read, RdBackgroundTask.Write);
      serializers.RegisterEnum<RdNotificationEntryType>();
      
      serializers.RegisterToplevelOnce(typeof(JetBrains.Rider.Model.IdeRoot), JetBrains.Rider.Model.IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    public RdNotificationsModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdNotificationsModel (");
      using (printer.IndentCookie()) {
        printer.Print("notification = "); _Notification.PrintEx(printer); printer.Println();
        printer.Print("messageBox = "); _MessageBox.PrintEx(printer); printer.Println();
        printer.Print("tasks = "); _Tasks.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdBackgroundTask : RdBindableBase {
    //fields
    //public fields
    [NotNull] public string Title {get; private set;}
    [NotNull] public IRdProperty<string> Header { get { return _Header; }}
    [NotNull] public IRdProperty<string> Description { get { return _Description; }}
    [NotNull] public IRdProperty<double> Progress { get { return _Progress; }}
    public bool IsIndeterminate {get; private set;}
    public bool IsCancelable {get; private set;}
    [NotNull] public ISink<RdVoid> Cancel { get { return _Cancel; }}
    public int UpdateInterval {get; private set;}
    
    //private fields
    [NotNull] private readonly RdProperty<string> _Header;
    [NotNull] private readonly RdProperty<string> _Description;
    [NotNull] private readonly RdProperty<double> _Progress;
    [NotNull] private readonly RdSignal<RdVoid> _Cancel;
    
    //primary constructor
    private RdBackgroundTask(
      [NotNull] string title,
      [NotNull] RdProperty<string> header,
      [NotNull] RdProperty<string> description,
      [NotNull] RdProperty<double> progress,
      bool isIndeterminate,
      bool isCancelable,
      [NotNull] RdSignal<RdVoid> cancel,
      int updateInterval
    )
    {
      if (title == null) throw new ArgumentNullException("title");
      if (header == null) throw new ArgumentNullException("header");
      if (description == null) throw new ArgumentNullException("description");
      if (progress == null) throw new ArgumentNullException("progress");
      if (cancel == null) throw new ArgumentNullException("cancel");
      
      Title = title;
      _Header = header;
      _Description = description;
      _Progress = progress;
      IsIndeterminate = isIndeterminate;
      IsCancelable = isCancelable;
      _Cancel = cancel;
      UpdateInterval = updateInterval;
      _Header.OptimizeNested = true;
      _Description.OptimizeNested = true;
      _Progress.OptimizeNested = true;
      _Header.Async = true;
      _Description.Async = true;
      _Progress.Async = true;
      _Header.ValueCanBeNull = true;
      _Description.ValueCanBeNull = true;
      BindableChildren.Add(new KeyValuePair<string, object>("header", _Header));
      BindableChildren.Add(new KeyValuePair<string, object>("description", _Description));
      BindableChildren.Add(new KeyValuePair<string, object>("progress", _Progress));
      BindableChildren.Add(new KeyValuePair<string, object>("cancel", _Cancel));
    }
    //secondary constructor
    public RdBackgroundTask (
      [NotNull] string title,
      bool isIndeterminate,
      bool isCancelable,
      int updateInterval
    ) : this (
      title,
      new RdProperty<string>(ReadStringNullable, WriteStringNullable),
      new RdProperty<string>(ReadStringNullable, WriteStringNullable),
      new RdProperty<double>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadDouble, JetBrains.Platform.RdFramework.Impl.Serializers.WriteDouble),
      isIndeterminate,
      isCancelable,
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      updateInterval
    ) {}
    //statics
    
    public static CtxReadDelegate<RdBackgroundTask> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var title = reader.ReadString();
      var header = RdProperty<string>.Read(ctx, reader, ReadStringNullable, WriteStringNullable);
      var description = RdProperty<string>.Read(ctx, reader, ReadStringNullable, WriteStringNullable);
      var progress = RdProperty<double>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadDouble, JetBrains.Platform.RdFramework.Impl.Serializers.WriteDouble);
      var isIndeterminate = reader.ReadBool();
      var isCancelable = reader.ReadBool();
      var cancel = RdSignal<RdVoid>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var updateInterval = reader.ReadInt();
      return new RdBackgroundTask(title, header, description, progress, isIndeterminate, isCancelable, cancel, updateInterval).WithId(_id);
    };
    public static CtxReadDelegate<string> ReadStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.NullableClass();
    
    public static CtxWriteDelegate<RdBackgroundTask> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.Title);
      RdProperty<string>.Write(ctx, writer, value._Header);
      RdProperty<string>.Write(ctx, writer, value._Description);
      RdProperty<double>.Write(ctx, writer, value._Progress);
      writer.Write(value.IsIndeterminate);
      writer.Write(value.IsCancelable);
      RdSignal<RdVoid>.Write(ctx, writer, value._Cancel);
      writer.Write(value.UpdateInterval);
    };
    public static CtxWriteDelegate<string> WriteStringNullable = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.NullableClass();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RdBackgroundTask (");
      using (printer.IndentCookie()) {
        printer.Print("title = "); Title.PrintEx(printer); printer.Println();
        printer.Print("header = "); _Header.PrintEx(printer); printer.Println();
        printer.Print("description = "); _Description.PrintEx(printer); printer.Println();
        printer.Print("progress = "); _Progress.PrintEx(printer); printer.Println();
        printer.Print("isIndeterminate = "); IsIndeterminate.PrintEx(printer); printer.Println();
        printer.Print("isCancelable = "); IsCancelable.PrintEx(printer); printer.Println();
        printer.Print("cancel = "); _Cancel.PrintEx(printer); printer.Println();
        printer.Print("updateInterval = "); UpdateInterval.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RdMessageBoxModel : IPrintable, IEquatable<RdMessageBoxModel> {
    //fields
    //public fields
    [NotNull] public string Message {get; private set;}
    [NotNull] public string Caption {get; private set;}
    public RdMessageBoxType Type {get; private set;}
    
    //private fields
    //primary constructor
    public RdMessageBoxModel(
      [NotNull] string message,
      [NotNull] string caption,
      RdMessageBoxType type
    )
    {
      if (message == null) throw new ArgumentNullException("message");
      if (caption == null) throw new ArgumentNullException("caption");
      
      Message = message;
      Caption = caption;
      Type = type;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdMessageBoxModel> Read = (ctx, reader) => 
    {
      var message = reader.ReadString();
      var caption = reader.ReadString();
      var type = (RdMessageBoxType)reader.ReadInt();
      return new RdMessageBoxModel(message, caption, type);
    };
    
    public static CtxWriteDelegate<RdMessageBoxModel> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Message);
      writer.Write(value.Caption);
      writer.Write((int)value.Type);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdMessageBoxModel) obj);
    }
    public bool Equals(RdMessageBoxModel other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Message == other.Message && Caption == other.Caption && Type == other.Type;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Message.GetHashCode();
        hash = hash * 31 + Caption.GetHashCode();
        hash = hash * 31 + (int) Type;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdMessageBoxModel (");
      using (printer.IndentCookie()) {
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
        printer.Print("caption = "); Caption.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdMessageBoxResponse {
    YES,
    NO,
    CANCEL,
    OK
  }
  
  
  public enum RdMessageBoxType {
    OK,
    YESNO,
    OKCANCEL
  }
  
  
  public class RdNotificationEntry : IPrintable, IEquatable<RdNotificationEntry> {
    //fields
    //public fields
    [NotNull] public string Title {get; private set;}
    [NotNull] public string Message {get; private set;}
    public bool ShowPopup {get; private set;}
    public RdNotificationEntryType Type {get; private set;}
    
    //private fields
    //primary constructor
    public RdNotificationEntry(
      [NotNull] string title,
      [NotNull] string message,
      bool showPopup,
      RdNotificationEntryType type
    )
    {
      if (title == null) throw new ArgumentNullException("title");
      if (message == null) throw new ArgumentNullException("message");
      
      Title = title;
      Message = message;
      ShowPopup = showPopup;
      Type = type;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RdNotificationEntry> Read = (ctx, reader) => 
    {
      var title = reader.ReadString();
      var message = reader.ReadString();
      var showPopup = reader.ReadBool();
      var type = (RdNotificationEntryType)reader.ReadInt();
      return new RdNotificationEntry(title, message, showPopup, type);
    };
    
    public static CtxWriteDelegate<RdNotificationEntry> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Title);
      writer.Write(value.Message);
      writer.Write(value.ShowPopup);
      writer.Write((int)value.Type);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RdNotificationEntry) obj);
    }
    public bool Equals(RdNotificationEntry other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Title == other.Title && Message == other.Message && ShowPopup == other.ShowPopup && Type == other.Type;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Title.GetHashCode();
        hash = hash * 31 + Message.GetHashCode();
        hash = hash * 31 + ShowPopup.GetHashCode();
        hash = hash * 31 + (int) Type;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RdNotificationEntry (");
      using (printer.IndentCookie()) {
        printer.Print("title = "); Title.PrintEx(printer); printer.Println();
        printer.Print("message = "); Message.PrintEx(printer); printer.Println();
        printer.Print("showPopup = "); ShowPopup.PrintEx(printer); printer.Println();
        printer.Print("type = "); Type.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RdNotificationEntryType {
    ERROR,
    WARN,
    INFO
  }
}
