using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.ReSharper.Roslyn.Host.Models
{
  
  
  public class RoslynModel : RdExtBase {
    //fields
    //public fields
    [NotNull] public IRdProperty<RoslynProjectSession> Session { get { return _Session; }}
    
    //private fields
    [NotNull] private readonly RdProperty<RoslynProjectSession> _Session;
    
    //primary constructor
    private RoslynModel(
      [NotNull] RdProperty<RoslynProjectSession> session
    )
    {
      if (session == null) throw new ArgumentNullException("session");
      
      _Session = session;
      BindableChildren.Add(new KeyValuePair<string, object>("session", _Session));
    }
    //secondary constructor
    private RoslynModel (
    ) : this (
      new RdProperty<RoslynProjectSession>(RoslynProjectSession.Read, RoslynProjectSession.Write)
    ) {}
    //statics
    
    
    
    protected override long SerializationHash => 2689025486291891543L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.RegisterEnum<RoslynSeverity>();
      serializers.Register(RoslynDiagnosticProperty.Read, RoslynDiagnosticProperty.Write);
      serializers.Register(RoslynDiagnostic.Read, RoslynDiagnostic.Write);
      serializers.Register(RoslynProjectSession.Read, RoslynProjectSession.Write);
      serializers.Register(RoslynProjectToLoad.Read, RoslynProjectToLoad.Write);
      serializers.Register(RoslynAnalyzer.Read, RoslynAnalyzer.Write);
      serializers.Register(RoslynRefactoring.Read, RoslynRefactoring.Write);
      serializers.Register(RoslynOpenDocument.Read, RoslynOpenDocument.Write);
      serializers.Register(RoslynRunningTask.Read, RoslynRunningTask.Write);
      serializers.Register(RoslynGetHighlighterArgs.Read, RoslynGetHighlighterArgs.Write);
      serializers.Register(RoslynHighlighter.Read, RoslynHighlighter.Write);
      serializers.Register(RoslynGetQuickFixesArgs.Read, RoslynGetQuickFixesArgs.Write);
      serializers.Register(RoslynQuickFixDescriptor.Read, RoslynQuickFixDescriptor.Write);
      serializers.Register(RoslynLaunchQuickFixArgs.Read, RoslynLaunchQuickFixArgs.Write);
      serializers.Register(RoslynGetRefactoringsArgs.Read, RoslynGetRefactoringsArgs.Write);
      serializers.Register(RoslynLaunchRefactoringArgs.Read, RoslynLaunchRefactoringArgs.Write);
      
      serializers.RegisterToplevelOnce(typeof(RoslynModel), RoslynModel.RegisterDeclaredTypesSerializers);
    }
    
    public RoslynModel(Lifetime lifetime, IProtocol protocol) : this()
    {
      Identify(protocol.Identities, RdId.Root.Mix(GetType().Name));
      Bind(lifetime, protocol, GetType().Name);
      if (Protocol.InitializationLogger.IsTraceEnabled())
        Protocol.InitializationLogger.Trace ("CREATED toplevel object {0}", this.PrintToString());
    }
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynModel (");
      using (printer.IndentCookie()) {
        printer.Print("session = "); _Session.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynAnalyzer : RdBindableBase {
    //fields
    //public fields
    [NotNull] public string DiagnosticId {get; private set;}
    [NotNull] public string Category {get; private set;}
    [NotNull] public string Title {get; private set;}
    [NotNull] public string Description {get; private set;}
    public RoslynSeverity DefaultSeverity {get; private set;}
    [NotNull] public List<string> Languages {get; private set;}
    [NotNull] public string Assembly {get; private set;}
    [NotNull] public IViewableList<string> Projects { get { return _Projects; }}
    
    //private fields
    [NotNull] private readonly RdList<string> _Projects;
    
    //primary constructor
    private RoslynAnalyzer(
      [NotNull] string diagnosticId,
      [NotNull] string category,
      [NotNull] string title,
      [NotNull] string description,
      RoslynSeverity defaultSeverity,
      [NotNull] List<string> languages,
      [NotNull] string assembly,
      [NotNull] RdList<string> projects
    )
    {
      if (diagnosticId == null) throw new ArgumentNullException("diagnosticId");
      if (category == null) throw new ArgumentNullException("category");
      if (title == null) throw new ArgumentNullException("title");
      if (description == null) throw new ArgumentNullException("description");
      if (languages == null) throw new ArgumentNullException("languages");
      if (assembly == null) throw new ArgumentNullException("assembly");
      if (projects == null) throw new ArgumentNullException("projects");
      
      DiagnosticId = diagnosticId;
      Category = category;
      Title = title;
      Description = description;
      DefaultSeverity = defaultSeverity;
      Languages = languages;
      Assembly = assembly;
      _Projects = projects;
      _Projects.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("projects", _Projects));
    }
    //secondary constructor
    public RoslynAnalyzer (
      [NotNull] string diagnosticId,
      [NotNull] string category,
      [NotNull] string title,
      [NotNull] string description,
      RoslynSeverity defaultSeverity,
      [NotNull] List<string> languages,
      [NotNull] string assembly
    ) : this (
      diagnosticId,
      category,
      title,
      description,
      defaultSeverity,
      languages,
      assembly,
      new RdList<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString)
    ) {}
    //statics
    
    public static CtxReadDelegate<RoslynAnalyzer> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var diagnosticId = reader.ReadString();
      var category = reader.ReadString();
      var title = reader.ReadString();
      var description = reader.ReadString();
      var defaultSeverity = (RoslynSeverity)reader.ReadInt();
      var languages = ReadStringList(ctx, reader);
      var assembly = reader.ReadString();
      var projects = RdList<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      return new RoslynAnalyzer(diagnosticId, category, title, description, defaultSeverity, languages, assembly, projects).WithId(_id);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<RoslynAnalyzer> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.DiagnosticId);
      writer.Write(value.Category);
      writer.Write(value.Title);
      writer.Write(value.Description);
      writer.Write((int)value.DefaultSeverity);
      WriteStringList(ctx, writer, value.Languages);
      writer.Write(value.Assembly);
      RdList<string>.Write(ctx, writer, value._Projects);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynAnalyzer (");
      using (printer.IndentCookie()) {
        printer.Print("diagnosticId = "); DiagnosticId.PrintEx(printer); printer.Println();
        printer.Print("category = "); Category.PrintEx(printer); printer.Println();
        printer.Print("title = "); Title.PrintEx(printer); printer.Println();
        printer.Print("description = "); Description.PrintEx(printer); printer.Println();
        printer.Print("defaultSeverity = "); DefaultSeverity.PrintEx(printer); printer.Println();
        printer.Print("languages = "); Languages.PrintEx(printer); printer.Println();
        printer.Print("assembly = "); Assembly.PrintEx(printer); printer.Println();
        printer.Print("projects = "); _Projects.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynDiagnostic : IPrintable, IEquatable<RoslynDiagnostic> {
    //fields
    //public fields
    [NotNull] public string DiagnosticId {get; private set;}
    public int StartOffset {get; private set;}
    public int EndOffset {get; private set;}
    [NotNull] public List<RoslynDiagnosticProperty> Properties {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynDiagnostic(
      [NotNull] string diagnosticId,
      int startOffset,
      int endOffset,
      [NotNull] List<RoslynDiagnosticProperty> properties
    )
    {
      if (diagnosticId == null) throw new ArgumentNullException("diagnosticId");
      if (properties == null) throw new ArgumentNullException("properties");
      
      DiagnosticId = diagnosticId;
      StartOffset = startOffset;
      EndOffset = endOffset;
      Properties = properties;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynDiagnostic> Read = (ctx, reader) => 
    {
      var diagnosticId = reader.ReadString();
      var startOffset = reader.ReadInt();
      var endOffset = reader.ReadInt();
      var properties = ReadRoslynDiagnosticPropertyList(ctx, reader);
      return new RoslynDiagnostic(diagnosticId, startOffset, endOffset, properties);
    };
    public static CtxReadDelegate<List<RoslynDiagnosticProperty>> ReadRoslynDiagnosticPropertyList = RoslynDiagnosticProperty.Read.List();
    
    public static CtxWriteDelegate<RoslynDiagnostic> Write = (ctx, writer, value) => 
    {
      writer.Write(value.DiagnosticId);
      writer.Write(value.StartOffset);
      writer.Write(value.EndOffset);
      WriteRoslynDiagnosticPropertyList(ctx, writer, value.Properties);
    };
    public static CtxWriteDelegate<List<RoslynDiagnosticProperty>> WriteRoslynDiagnosticPropertyList = RoslynDiagnosticProperty.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynDiagnostic) obj);
    }
    public bool Equals(RoslynDiagnostic other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return DiagnosticId == other.DiagnosticId && StartOffset == other.StartOffset && EndOffset == other.EndOffset && Properties.SequenceEqual(other.Properties);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + DiagnosticId.GetHashCode();
        hash = hash * 31 + StartOffset.GetHashCode();
        hash = hash * 31 + EndOffset.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Properties);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynDiagnostic (");
      using (printer.IndentCookie()) {
        printer.Print("diagnosticId = "); DiagnosticId.PrintEx(printer); printer.Println();
        printer.Print("startOffset = "); StartOffset.PrintEx(printer); printer.Println();
        printer.Print("endOffset = "); EndOffset.PrintEx(printer); printer.Println();
        printer.Print("properties = "); Properties.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynDiagnosticProperty : IPrintable, IEquatable<RoslynDiagnosticProperty> {
    //fields
    //public fields
    [NotNull] public string Key {get; private set;}
    [NotNull] public string Value {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynDiagnosticProperty(
      [NotNull] string key,
      [NotNull] string value
    )
    {
      if (key == null) throw new ArgumentNullException("key");
      if (value == null) throw new ArgumentNullException("value");
      
      Key = key;
      Value = value;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynDiagnosticProperty> Read = (ctx, reader) => 
    {
      var key = reader.ReadString();
      var value = reader.ReadString();
      return new RoslynDiagnosticProperty(key, value);
    };
    
    public static CtxWriteDelegate<RoslynDiagnosticProperty> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Key);
      writer.Write(value.Value);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynDiagnosticProperty) obj);
    }
    public bool Equals(RoslynDiagnosticProperty other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Key == other.Key && Value == other.Value;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Key.GetHashCode();
        hash = hash * 31 + Value.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynDiagnosticProperty (");
      using (printer.IndentCookie()) {
        printer.Print("key = "); Key.PrintEx(printer); printer.Println();
        printer.Print("value = "); Value.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynGetHighlighterArgs : IPrintable, IEquatable<RoslynGetHighlighterArgs> {
    //fields
    //public fields
    public Guid ProjectGuid {get; private set;}
    [NotNull] public string FileName {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynGetHighlighterArgs(
      Guid projectGuid,
      [NotNull] string fileName
    )
    {
      if (fileName == null) throw new ArgumentNullException("fileName");
      
      ProjectGuid = projectGuid;
      FileName = fileName;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynGetHighlighterArgs> Read = (ctx, reader) => 
    {
      var projectGuid = reader.ReadGuid();
      var fileName = reader.ReadString();
      return new RoslynGetHighlighterArgs(projectGuid, fileName);
    };
    
    public static CtxWriteDelegate<RoslynGetHighlighterArgs> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProjectGuid);
      writer.Write(value.FileName);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynGetHighlighterArgs) obj);
    }
    public bool Equals(RoslynGetHighlighterArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProjectGuid == other.ProjectGuid && FileName == other.FileName;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProjectGuid.GetHashCode();
        hash = hash * 31 + FileName.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynGetHighlighterArgs (");
      using (printer.IndentCookie()) {
        printer.Print("projectGuid = "); ProjectGuid.PrintEx(printer); printer.Println();
        printer.Print("fileName = "); FileName.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynGetQuickFixesArgs : IPrintable, IEquatable<RoslynGetQuickFixesArgs> {
    //fields
    //public fields
    public Guid ProjectGuid {get; private set;}
    [NotNull] public string FileName {get; private set;}
    [NotNull] public RoslynDiagnostic Diagnostic {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynGetQuickFixesArgs(
      Guid projectGuid,
      [NotNull] string fileName,
      [NotNull] RoslynDiagnostic diagnostic
    )
    {
      if (fileName == null) throw new ArgumentNullException("fileName");
      if (diagnostic == null) throw new ArgumentNullException("diagnostic");
      
      ProjectGuid = projectGuid;
      FileName = fileName;
      Diagnostic = diagnostic;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynGetQuickFixesArgs> Read = (ctx, reader) => 
    {
      var projectGuid = reader.ReadGuid();
      var fileName = reader.ReadString();
      var diagnostic = RoslynDiagnostic.Read(ctx, reader);
      return new RoslynGetQuickFixesArgs(projectGuid, fileName, diagnostic);
    };
    
    public static CtxWriteDelegate<RoslynGetQuickFixesArgs> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProjectGuid);
      writer.Write(value.FileName);
      RoslynDiagnostic.Write(ctx, writer, value.Diagnostic);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynGetQuickFixesArgs) obj);
    }
    public bool Equals(RoslynGetQuickFixesArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProjectGuid == other.ProjectGuid && FileName == other.FileName && Equals(Diagnostic, other.Diagnostic);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProjectGuid.GetHashCode();
        hash = hash * 31 + FileName.GetHashCode();
        hash = hash * 31 + Diagnostic.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynGetQuickFixesArgs (");
      using (printer.IndentCookie()) {
        printer.Print("projectGuid = "); ProjectGuid.PrintEx(printer); printer.Println();
        printer.Print("fileName = "); FileName.PrintEx(printer); printer.Println();
        printer.Print("diagnostic = "); Diagnostic.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynGetRefactoringsArgs : IPrintable, IEquatable<RoslynGetRefactoringsArgs> {
    //fields
    //public fields
    public Guid ProjectGuid {get; private set;}
    [NotNull] public string FileName {get; private set;}
    public int Offset {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynGetRefactoringsArgs(
      Guid projectGuid,
      [NotNull] string fileName,
      int offset
    )
    {
      if (fileName == null) throw new ArgumentNullException("fileName");
      
      ProjectGuid = projectGuid;
      FileName = fileName;
      Offset = offset;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynGetRefactoringsArgs> Read = (ctx, reader) => 
    {
      var projectGuid = reader.ReadGuid();
      var fileName = reader.ReadString();
      var offset = reader.ReadInt();
      return new RoslynGetRefactoringsArgs(projectGuid, fileName, offset);
    };
    
    public static CtxWriteDelegate<RoslynGetRefactoringsArgs> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProjectGuid);
      writer.Write(value.FileName);
      writer.Write(value.Offset);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynGetRefactoringsArgs) obj);
    }
    public bool Equals(RoslynGetRefactoringsArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProjectGuid == other.ProjectGuid && FileName == other.FileName && Offset == other.Offset;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProjectGuid.GetHashCode();
        hash = hash * 31 + FileName.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynGetRefactoringsArgs (");
      using (printer.IndentCookie()) {
        printer.Print("projectGuid = "); ProjectGuid.PrintEx(printer); printer.Println();
        printer.Print("fileName = "); FileName.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynHighlighter : IPrintable, IEquatable<RoslynHighlighter> {
    //fields
    //public fields
    [NotNull] public string Text {get; private set;}
    public RoslynSeverity Severity {get; private set;}
    [NotNull] public RoslynDiagnostic Diagnostic {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynHighlighter(
      [NotNull] string text,
      RoslynSeverity severity,
      [NotNull] RoslynDiagnostic diagnostic
    )
    {
      if (text == null) throw new ArgumentNullException("text");
      if (diagnostic == null) throw new ArgumentNullException("diagnostic");
      
      Text = text;
      Severity = severity;
      Diagnostic = diagnostic;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynHighlighter> Read = (ctx, reader) => 
    {
      var text = reader.ReadString();
      var severity = (RoslynSeverity)reader.ReadInt();
      var diagnostic = RoslynDiagnostic.Read(ctx, reader);
      return new RoslynHighlighter(text, severity, diagnostic);
    };
    
    public static CtxWriteDelegate<RoslynHighlighter> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Text);
      writer.Write((int)value.Severity);
      RoslynDiagnostic.Write(ctx, writer, value.Diagnostic);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynHighlighter) obj);
    }
    public bool Equals(RoslynHighlighter other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Text == other.Text && Severity == other.Severity && Equals(Diagnostic, other.Diagnostic);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Text.GetHashCode();
        hash = hash * 31 + (int) Severity;
        hash = hash * 31 + Diagnostic.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynHighlighter (");
      using (printer.IndentCookie()) {
        printer.Print("text = "); Text.PrintEx(printer); printer.Println();
        printer.Print("severity = "); Severity.PrintEx(printer); printer.Println();
        printer.Print("diagnostic = "); Diagnostic.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynLaunchQuickFixArgs : IPrintable, IEquatable<RoslynLaunchQuickFixArgs> {
    //fields
    //public fields
    public Guid ProjectGuid {get; private set;}
    [NotNull] public string FileName {get; private set;}
    [NotNull] public RoslynDiagnostic Diagnostic {get; private set;}
    [NotNull] public string QuickFixName {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynLaunchQuickFixArgs(
      Guid projectGuid,
      [NotNull] string fileName,
      [NotNull] RoslynDiagnostic diagnostic,
      [NotNull] string quickFixName
    )
    {
      if (fileName == null) throw new ArgumentNullException("fileName");
      if (diagnostic == null) throw new ArgumentNullException("diagnostic");
      if (quickFixName == null) throw new ArgumentNullException("quickFixName");
      
      ProjectGuid = projectGuid;
      FileName = fileName;
      Diagnostic = diagnostic;
      QuickFixName = quickFixName;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynLaunchQuickFixArgs> Read = (ctx, reader) => 
    {
      var projectGuid = reader.ReadGuid();
      var fileName = reader.ReadString();
      var diagnostic = RoslynDiagnostic.Read(ctx, reader);
      var quickFixName = reader.ReadString();
      return new RoslynLaunchQuickFixArgs(projectGuid, fileName, diagnostic, quickFixName);
    };
    
    public static CtxWriteDelegate<RoslynLaunchQuickFixArgs> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProjectGuid);
      writer.Write(value.FileName);
      RoslynDiagnostic.Write(ctx, writer, value.Diagnostic);
      writer.Write(value.QuickFixName);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynLaunchQuickFixArgs) obj);
    }
    public bool Equals(RoslynLaunchQuickFixArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProjectGuid == other.ProjectGuid && FileName == other.FileName && Equals(Diagnostic, other.Diagnostic) && QuickFixName == other.QuickFixName;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProjectGuid.GetHashCode();
        hash = hash * 31 + FileName.GetHashCode();
        hash = hash * 31 + Diagnostic.GetHashCode();
        hash = hash * 31 + QuickFixName.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynLaunchQuickFixArgs (");
      using (printer.IndentCookie()) {
        printer.Print("projectGuid = "); ProjectGuid.PrintEx(printer); printer.Println();
        printer.Print("fileName = "); FileName.PrintEx(printer); printer.Println();
        printer.Print("diagnostic = "); Diagnostic.PrintEx(printer); printer.Println();
        printer.Print("quickFixName = "); QuickFixName.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynLaunchRefactoringArgs : IPrintable, IEquatable<RoslynLaunchRefactoringArgs> {
    //fields
    //public fields
    public Guid ProjectGuid {get; private set;}
    [NotNull] public string FileName {get; private set;}
    public int Offset {get; private set;}
    [NotNull] public string Text {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynLaunchRefactoringArgs(
      Guid projectGuid,
      [NotNull] string fileName,
      int offset,
      [NotNull] string text
    )
    {
      if (fileName == null) throw new ArgumentNullException("fileName");
      if (text == null) throw new ArgumentNullException("text");
      
      ProjectGuid = projectGuid;
      FileName = fileName;
      Offset = offset;
      Text = text;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynLaunchRefactoringArgs> Read = (ctx, reader) => 
    {
      var projectGuid = reader.ReadGuid();
      var fileName = reader.ReadString();
      var offset = reader.ReadInt();
      var text = reader.ReadString();
      return new RoslynLaunchRefactoringArgs(projectGuid, fileName, offset, text);
    };
    
    public static CtxWriteDelegate<RoslynLaunchRefactoringArgs> Write = (ctx, writer, value) => 
    {
      writer.Write(value.ProjectGuid);
      writer.Write(value.FileName);
      writer.Write(value.Offset);
      writer.Write(value.Text);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynLaunchRefactoringArgs) obj);
    }
    public bool Equals(RoslynLaunchRefactoringArgs other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return ProjectGuid == other.ProjectGuid && FileName == other.FileName && Offset == other.Offset && Text == other.Text;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + ProjectGuid.GetHashCode();
        hash = hash * 31 + FileName.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        hash = hash * 31 + Text.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynLaunchRefactoringArgs (");
      using (printer.IndentCookie()) {
        printer.Print("projectGuid = "); ProjectGuid.PrintEx(printer); printer.Println();
        printer.Print("fileName = "); FileName.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
        printer.Print("text = "); Text.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynOpenDocument : RdBindableBase {
    //fields
    //public fields
    [NotNull] public IRdProperty<string> InitialText { get { return _InitialText; }}
    [NotNull] public ISource<string> NewDocumentText { get { return _NewDocumentText; }}
    [NotNull] public ISink<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange> Change { get { return _Change; }}
    
    //private fields
    [NotNull] private readonly RdProperty<string> _InitialText;
    [NotNull] private readonly RdSignal<string> _NewDocumentText;
    [NotNull] private readonly RdSignal<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange> _Change;
    
    //primary constructor
    private RoslynOpenDocument(
      [NotNull] RdProperty<string> initialText,
      [NotNull] RdSignal<string> newDocumentText,
      [NotNull] RdSignal<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange> change
    )
    {
      if (initialText == null) throw new ArgumentNullException("initialText");
      if (newDocumentText == null) throw new ArgumentNullException("newDocumentText");
      if (change == null) throw new ArgumentNullException("change");
      
      _InitialText = initialText;
      _NewDocumentText = newDocumentText;
      _Change = change;
      _InitialText.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("initialText", _InitialText));
      BindableChildren.Add(new KeyValuePair<string, object>("newDocumentText", _NewDocumentText));
      BindableChildren.Add(new KeyValuePair<string, object>("change", _Change));
    }
    //secondary constructor
    public RoslynOpenDocument (
    ) : this (
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdSignal<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdSignal<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange>(JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange.ReadDelegate, JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange.WriteDelegate)
    ) {}
    //statics
    
    public static CtxReadDelegate<RoslynOpenDocument> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var initialText = RdProperty<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var newDocumentText = RdSignal<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var change = RdSignal<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange>.Read(ctx, reader, JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange.ReadDelegate, JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange.WriteDelegate);
      return new RoslynOpenDocument(initialText, newDocumentText, change).WithId(_id);
    };
    
    public static CtxWriteDelegate<RoslynOpenDocument> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      RdProperty<string>.Write(ctx, writer, value._InitialText);
      RdSignal<string>.Write(ctx, writer, value._NewDocumentText);
      RdSignal<JetBrains.Platform.RdFramework.Text.Impl.Intrinsics.RdTextBufferChange>.Write(ctx, writer, value._Change);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynOpenDocument (");
      using (printer.IndentCookie()) {
        printer.Print("initialText = "); _InitialText.PrintEx(printer); printer.Println();
        printer.Print("newDocumentText = "); _NewDocumentText.PrintEx(printer); printer.Println();
        printer.Print("change = "); _Change.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynProjectSession : RdBindableBase {
    //fields
    //public fields
    [NotNull] public string SolutionPath {get; private set;}
    public Guid SolutionGuid {get; private set;}
    public bool UseDefaultCollection {get; private set;}
    [NotNull] public IViewableMap<Guid, RoslynProjectToLoad> ProjectToLoad { get { return _ProjectToLoad; }}
    [NotNull] public IViewableMap<string, RoslynAnalyzer> AnalyzersInProjects { get { return _AnalyzersInProjects; }}
    [NotNull] public IViewableMap<string, RoslynRefactoring> RefactoringsInProjects { get { return _RefactoringsInProjects; }}
    [NotNull] public IViewableMap<string, RoslynOpenDocument> OpenDocuments { get { return _OpenDocuments; }}
    [NotNull] public IViewableMap<int, RoslynRunningTask> RunningTasks { get { return _RunningTasks; }}
    [NotNull] public ISource<string> FileOnDiskChanged { get { return _FileOnDiskChanged; }}
    [NotNull] public IRdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>> GetFileHighlighters { get { return _GetFileHighlighters; }}
    [NotNull] public IRdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>> GetQuickFixes { get { return _GetQuickFixes; }}
    [NotNull] public IRdCall<RoslynLaunchQuickFixArgs, RdVoid> LaunchQuickFix { get { return _LaunchQuickFix; }}
    [NotNull] public IRdCall<RoslynGetRefactoringsArgs, List<string>> GetRefactorings { get { return _GetRefactorings; }}
    [NotNull] public IRdCall<RoslynLaunchRefactoringArgs, RdVoid> LaunchRefactoring { get { return _LaunchRefactoring; }}
    [NotNull] public ISink<string> RunFilesystemUpdate { get { return _RunFilesystemUpdate; }}
    [NotNull] public IRdProperty<string> LastPerformedAction { get { return _LastPerformedAction; }}
    
    //private fields
    [NotNull] private readonly RdMap<Guid, RoslynProjectToLoad> _ProjectToLoad;
    [NotNull] private readonly RdMap<string, RoslynAnalyzer> _AnalyzersInProjects;
    [NotNull] private readonly RdMap<string, RoslynRefactoring> _RefactoringsInProjects;
    [NotNull] private readonly RdMap<string, RoslynOpenDocument> _OpenDocuments;
    [NotNull] private readonly RdMap<int, RoslynRunningTask> _RunningTasks;
    [NotNull] private readonly RdSignal<string> _FileOnDiskChanged;
    [NotNull] private readonly RdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>> _GetFileHighlighters;
    [NotNull] private readonly RdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>> _GetQuickFixes;
    [NotNull] private readonly RdCall<RoslynLaunchQuickFixArgs, RdVoid> _LaunchQuickFix;
    [NotNull] private readonly RdCall<RoslynGetRefactoringsArgs, List<string>> _GetRefactorings;
    [NotNull] private readonly RdCall<RoslynLaunchRefactoringArgs, RdVoid> _LaunchRefactoring;
    [NotNull] private readonly RdSignal<string> _RunFilesystemUpdate;
    [NotNull] private readonly RdProperty<string> _LastPerformedAction;
    
    //primary constructor
    private RoslynProjectSession(
      [NotNull] string solutionPath,
      Guid solutionGuid,
      bool useDefaultCollection,
      [NotNull] RdMap<Guid, RoslynProjectToLoad> projectToLoad,
      [NotNull] RdMap<string, RoslynAnalyzer> analyzersInProjects,
      [NotNull] RdMap<string, RoslynRefactoring> refactoringsInProjects,
      [NotNull] RdMap<string, RoslynOpenDocument> openDocuments,
      [NotNull] RdMap<int, RoslynRunningTask> runningTasks,
      [NotNull] RdSignal<string> fileOnDiskChanged,
      [NotNull] RdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>> getFileHighlighters,
      [NotNull] RdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>> getQuickFixes,
      [NotNull] RdCall<RoslynLaunchQuickFixArgs, RdVoid> launchQuickFix,
      [NotNull] RdCall<RoslynGetRefactoringsArgs, List<string>> getRefactorings,
      [NotNull] RdCall<RoslynLaunchRefactoringArgs, RdVoid> launchRefactoring,
      [NotNull] RdSignal<string> runFilesystemUpdate,
      [NotNull] RdProperty<string> lastPerformedAction
    )
    {
      if (solutionPath == null) throw new ArgumentNullException("solutionPath");
      if (projectToLoad == null) throw new ArgumentNullException("projectToLoad");
      if (analyzersInProjects == null) throw new ArgumentNullException("analyzersInProjects");
      if (refactoringsInProjects == null) throw new ArgumentNullException("refactoringsInProjects");
      if (openDocuments == null) throw new ArgumentNullException("openDocuments");
      if (runningTasks == null) throw new ArgumentNullException("runningTasks");
      if (fileOnDiskChanged == null) throw new ArgumentNullException("fileOnDiskChanged");
      if (getFileHighlighters == null) throw new ArgumentNullException("getFileHighlighters");
      if (getQuickFixes == null) throw new ArgumentNullException("getQuickFixes");
      if (launchQuickFix == null) throw new ArgumentNullException("launchQuickFix");
      if (getRefactorings == null) throw new ArgumentNullException("getRefactorings");
      if (launchRefactoring == null) throw new ArgumentNullException("launchRefactoring");
      if (runFilesystemUpdate == null) throw new ArgumentNullException("runFilesystemUpdate");
      if (lastPerformedAction == null) throw new ArgumentNullException("lastPerformedAction");
      
      SolutionPath = solutionPath;
      SolutionGuid = solutionGuid;
      UseDefaultCollection = useDefaultCollection;
      _ProjectToLoad = projectToLoad;
      _AnalyzersInProjects = analyzersInProjects;
      _RefactoringsInProjects = refactoringsInProjects;
      _OpenDocuments = openDocuments;
      _RunningTasks = runningTasks;
      _FileOnDiskChanged = fileOnDiskChanged;
      _GetFileHighlighters = getFileHighlighters;
      _GetQuickFixes = getQuickFixes;
      _LaunchQuickFix = launchQuickFix;
      _GetRefactorings = getRefactorings;
      _LaunchRefactoring = launchRefactoring;
      _RunFilesystemUpdate = runFilesystemUpdate;
      _LastPerformedAction = lastPerformedAction;
      _RunningTasks.OptimizeNested = true;
      _LastPerformedAction.OptimizeNested = true;
      _GetFileHighlighters.Async = true;
      _GetQuickFixes.Async = true;
      _LaunchQuickFix.Async = true;
      _GetRefactorings.Async = true;
      _LaunchRefactoring.Async = true;
      BindableChildren.Add(new KeyValuePair<string, object>("projectToLoad", _ProjectToLoad));
      BindableChildren.Add(new KeyValuePair<string, object>("analyzersInProjects", _AnalyzersInProjects));
      BindableChildren.Add(new KeyValuePair<string, object>("refactoringsInProjects", _RefactoringsInProjects));
      BindableChildren.Add(new KeyValuePair<string, object>("openDocuments", _OpenDocuments));
      BindableChildren.Add(new KeyValuePair<string, object>("runningTasks", _RunningTasks));
      BindableChildren.Add(new KeyValuePair<string, object>("fileOnDiskChanged", _FileOnDiskChanged));
      BindableChildren.Add(new KeyValuePair<string, object>("getFileHighlighters", _GetFileHighlighters));
      BindableChildren.Add(new KeyValuePair<string, object>("getQuickFixes", _GetQuickFixes));
      BindableChildren.Add(new KeyValuePair<string, object>("launchQuickFix", _LaunchQuickFix));
      BindableChildren.Add(new KeyValuePair<string, object>("getRefactorings", _GetRefactorings));
      BindableChildren.Add(new KeyValuePair<string, object>("launchRefactoring", _LaunchRefactoring));
      BindableChildren.Add(new KeyValuePair<string, object>("runFilesystemUpdate", _RunFilesystemUpdate));
      BindableChildren.Add(new KeyValuePair<string, object>("lastPerformedAction", _LastPerformedAction));
    }
    //secondary constructor
    public RoslynProjectSession (
      [NotNull] string solutionPath,
      Guid solutionGuid,
      bool useDefaultCollection
    ) : this (
      solutionPath,
      solutionGuid,
      useDefaultCollection,
      new RdMap<Guid, RoslynProjectToLoad>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadGuid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteGuid, RoslynProjectToLoad.Read, RoslynProjectToLoad.Write),
      new RdMap<string, RoslynAnalyzer>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynAnalyzer.Read, RoslynAnalyzer.Write),
      new RdMap<string, RoslynRefactoring>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynRefactoring.Read, RoslynRefactoring.Write),
      new RdMap<string, RoslynOpenDocument>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynOpenDocument.Read, RoslynOpenDocument.Write),
      new RdMap<int, RoslynRunningTask>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt, RoslynRunningTask.Read, RoslynRunningTask.Write),
      new RdSignal<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>>(RoslynGetHighlighterArgs.Read, RoslynGetHighlighterArgs.Write, ReadRoslynHighlighterList, WriteRoslynHighlighterList),
      new RdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>>(RoslynGetQuickFixesArgs.Read, RoslynGetQuickFixesArgs.Write, ReadRoslynQuickFixDescriptorList, WriteRoslynQuickFixDescriptorList),
      new RdCall<RoslynLaunchQuickFixArgs, RdVoid>(RoslynLaunchQuickFixArgs.Read, RoslynLaunchQuickFixArgs.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdCall<RoslynGetRefactoringsArgs, List<string>>(RoslynGetRefactoringsArgs.Read, RoslynGetRefactoringsArgs.Write, ReadStringList, WriteStringList),
      new RdCall<RoslynLaunchRefactoringArgs, RdVoid>(RoslynLaunchRefactoringArgs.Read, RoslynLaunchRefactoringArgs.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid),
      new RdSignal<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString)
    ) {}
    //statics
    
    public static CtxReadDelegate<RoslynProjectSession> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var solutionPath = reader.ReadString();
      var solutionGuid = reader.ReadGuid();
      var useDefaultCollection = reader.ReadBool();
      var projectToLoad = RdMap<Guid, RoslynProjectToLoad>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadGuid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteGuid, RoslynProjectToLoad.Read, RoslynProjectToLoad.Write);
      var analyzersInProjects = RdMap<string, RoslynAnalyzer>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynAnalyzer.Read, RoslynAnalyzer.Write);
      var refactoringsInProjects = RdMap<string, RoslynRefactoring>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynRefactoring.Read, RoslynRefactoring.Write);
      var openDocuments = RdMap<string, RoslynOpenDocument>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, RoslynOpenDocument.Read, RoslynOpenDocument.Write);
      var runningTasks = RdMap<int, RoslynRunningTask>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt, JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt, RoslynRunningTask.Read, RoslynRunningTask.Write);
      var fileOnDiskChanged = RdSignal<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var getFileHighlighters = RdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>>.Read(ctx, reader, RoslynGetHighlighterArgs.Read, RoslynGetHighlighterArgs.Write, ReadRoslynHighlighterList, WriteRoslynHighlighterList);
      var getQuickFixes = RdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>>.Read(ctx, reader, RoslynGetQuickFixesArgs.Read, RoslynGetQuickFixesArgs.Write, ReadRoslynQuickFixDescriptorList, WriteRoslynQuickFixDescriptorList);
      var launchQuickFix = RdCall<RoslynLaunchQuickFixArgs, RdVoid>.Read(ctx, reader, RoslynLaunchQuickFixArgs.Read, RoslynLaunchQuickFixArgs.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var getRefactorings = RdCall<RoslynGetRefactoringsArgs, List<string>>.Read(ctx, reader, RoslynGetRefactoringsArgs.Read, RoslynGetRefactoringsArgs.Write, ReadStringList, WriteStringList);
      var launchRefactoring = RdCall<RoslynLaunchRefactoringArgs, RdVoid>.Read(ctx, reader, RoslynLaunchRefactoringArgs.Read, RoslynLaunchRefactoringArgs.Write, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      var runFilesystemUpdate = RdSignal<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var lastPerformedAction = RdProperty<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      return new RoslynProjectSession(solutionPath, solutionGuid, useDefaultCollection, projectToLoad, analyzersInProjects, refactoringsInProjects, openDocuments, runningTasks, fileOnDiskChanged, getFileHighlighters, getQuickFixes, launchQuickFix, getRefactorings, launchRefactoring, runFilesystemUpdate, lastPerformedAction).WithId(_id);
    };
    public static CtxReadDelegate<List<RoslynHighlighter>> ReadRoslynHighlighterList = RoslynHighlighter.Read.List();
    public static CtxReadDelegate<List<RoslynQuickFixDescriptor>> ReadRoslynQuickFixDescriptorList = RoslynQuickFixDescriptor.Read.List();
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<RoslynProjectSession> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.SolutionPath);
      writer.Write(value.SolutionGuid);
      writer.Write(value.UseDefaultCollection);
      RdMap<Guid, RoslynProjectToLoad>.Write(ctx, writer, value._ProjectToLoad);
      RdMap<string, RoslynAnalyzer>.Write(ctx, writer, value._AnalyzersInProjects);
      RdMap<string, RoslynRefactoring>.Write(ctx, writer, value._RefactoringsInProjects);
      RdMap<string, RoslynOpenDocument>.Write(ctx, writer, value._OpenDocuments);
      RdMap<int, RoslynRunningTask>.Write(ctx, writer, value._RunningTasks);
      RdSignal<string>.Write(ctx, writer, value._FileOnDiskChanged);
      RdCall<RoslynGetHighlighterArgs, List<RoslynHighlighter>>.Write(ctx, writer, value._GetFileHighlighters);
      RdCall<RoslynGetQuickFixesArgs, List<RoslynQuickFixDescriptor>>.Write(ctx, writer, value._GetQuickFixes);
      RdCall<RoslynLaunchQuickFixArgs, RdVoid>.Write(ctx, writer, value._LaunchQuickFix);
      RdCall<RoslynGetRefactoringsArgs, List<string>>.Write(ctx, writer, value._GetRefactorings);
      RdCall<RoslynLaunchRefactoringArgs, RdVoid>.Write(ctx, writer, value._LaunchRefactoring);
      RdSignal<string>.Write(ctx, writer, value._RunFilesystemUpdate);
      RdProperty<string>.Write(ctx, writer, value._LastPerformedAction);
    };
    public static CtxWriteDelegate<List<RoslynHighlighter>> WriteRoslynHighlighterList = RoslynHighlighter.Write.List();
    public static CtxWriteDelegate<List<RoslynQuickFixDescriptor>> WriteRoslynQuickFixDescriptorList = RoslynQuickFixDescriptor.Write.List();
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynProjectSession (");
      using (printer.IndentCookie()) {
        printer.Print("solutionPath = "); SolutionPath.PrintEx(printer); printer.Println();
        printer.Print("solutionGuid = "); SolutionGuid.PrintEx(printer); printer.Println();
        printer.Print("useDefaultCollection = "); UseDefaultCollection.PrintEx(printer); printer.Println();
        printer.Print("projectToLoad = "); _ProjectToLoad.PrintEx(printer); printer.Println();
        printer.Print("analyzersInProjects = "); _AnalyzersInProjects.PrintEx(printer); printer.Println();
        printer.Print("refactoringsInProjects = "); _RefactoringsInProjects.PrintEx(printer); printer.Println();
        printer.Print("openDocuments = "); _OpenDocuments.PrintEx(printer); printer.Println();
        printer.Print("runningTasks = "); _RunningTasks.PrintEx(printer); printer.Println();
        printer.Print("fileOnDiskChanged = "); _FileOnDiskChanged.PrintEx(printer); printer.Println();
        printer.Print("getFileHighlighters = "); _GetFileHighlighters.PrintEx(printer); printer.Println();
        printer.Print("getQuickFixes = "); _GetQuickFixes.PrintEx(printer); printer.Println();
        printer.Print("launchQuickFix = "); _LaunchQuickFix.PrintEx(printer); printer.Println();
        printer.Print("getRefactorings = "); _GetRefactorings.PrintEx(printer); printer.Println();
        printer.Print("launchRefactoring = "); _LaunchRefactoring.PrintEx(printer); printer.Println();
        printer.Print("runFilesystemUpdate = "); _RunFilesystemUpdate.PrintEx(printer); printer.Println();
        printer.Print("lastPerformedAction = "); _LastPerformedAction.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynProjectToLoad : RdBindableBase {
    //fields
    //public fields
    [NotNull] public string Path {get; private set;}
    public Guid Guid {get; private set;}
    [NotNull] public string Name {get; private set;}
    [NotNull] public IRdProperty<string> LoadMessages { get { return _LoadMessages; }}
    [NotNull] public IRdProperty<bool> LoadComplete { get { return _LoadComplete; }}
    [NotNull] public IRdProperty<string> SelectedConfiguration { get { return _SelectedConfiguration; }}
    [NotNull] public IRdProperty<string> SelectedPlatform { get { return _SelectedPlatform; }}
    [NotNull] public ISource<RdVoid> Reload { get { return _Reload; }}
    
    //private fields
    [NotNull] private readonly RdProperty<string> _LoadMessages;
    [NotNull] private readonly RdProperty<bool> _LoadComplete;
    [NotNull] private readonly RdProperty<string> _SelectedConfiguration;
    [NotNull] private readonly RdProperty<string> _SelectedPlatform;
    [NotNull] private readonly RdSignal<RdVoid> _Reload;
    
    //primary constructor
    private RoslynProjectToLoad(
      [NotNull] string path,
      Guid guid,
      [NotNull] string name,
      [NotNull] RdProperty<string> loadMessages,
      [NotNull] RdProperty<bool> loadComplete,
      [NotNull] RdProperty<string> selectedConfiguration,
      [NotNull] RdProperty<string> selectedPlatform,
      [NotNull] RdSignal<RdVoid> reload
    )
    {
      if (path == null) throw new ArgumentNullException("path");
      if (name == null) throw new ArgumentNullException("name");
      if (loadMessages == null) throw new ArgumentNullException("loadMessages");
      if (loadComplete == null) throw new ArgumentNullException("loadComplete");
      if (selectedConfiguration == null) throw new ArgumentNullException("selectedConfiguration");
      if (selectedPlatform == null) throw new ArgumentNullException("selectedPlatform");
      if (reload == null) throw new ArgumentNullException("reload");
      
      Path = path;
      Guid = guid;
      Name = name;
      _LoadMessages = loadMessages;
      _LoadComplete = loadComplete;
      _SelectedConfiguration = selectedConfiguration;
      _SelectedPlatform = selectedPlatform;
      _Reload = reload;
      _LoadMessages.OptimizeNested = true;
      _LoadComplete.OptimizeNested = true;
      _SelectedConfiguration.OptimizeNested = true;
      _SelectedPlatform.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("loadMessages", _LoadMessages));
      BindableChildren.Add(new KeyValuePair<string, object>("loadComplete", _LoadComplete));
      BindableChildren.Add(new KeyValuePair<string, object>("selectedConfiguration", _SelectedConfiguration));
      BindableChildren.Add(new KeyValuePair<string, object>("selectedPlatform", _SelectedPlatform));
      BindableChildren.Add(new KeyValuePair<string, object>("reload", _Reload));
    }
    //secondary constructor
    public RoslynProjectToLoad (
      [NotNull] string path,
      Guid guid,
      [NotNull] string name
    ) : this (
      path,
      guid,
      name,
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdSignal<RdVoid>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid)
    ) {}
    //statics
    
    public static CtxReadDelegate<RoslynProjectToLoad> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var path = reader.ReadString();
      var guid = reader.ReadGuid();
      var name = reader.ReadString();
      var loadMessages = RdProperty<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var loadComplete = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var selectedConfiguration = RdProperty<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var selectedPlatform = RdProperty<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var reload = RdSignal<RdVoid>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadVoid, JetBrains.Platform.RdFramework.Impl.Serializers.WriteVoid);
      return new RoslynProjectToLoad(path, guid, name, loadMessages, loadComplete, selectedConfiguration, selectedPlatform, reload).WithId(_id);
    };
    
    public static CtxWriteDelegate<RoslynProjectToLoad> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.Path);
      writer.Write(value.Guid);
      writer.Write(value.Name);
      RdProperty<string>.Write(ctx, writer, value._LoadMessages);
      RdProperty<bool>.Write(ctx, writer, value._LoadComplete);
      RdProperty<string>.Write(ctx, writer, value._SelectedConfiguration);
      RdProperty<string>.Write(ctx, writer, value._SelectedPlatform);
      RdSignal<RdVoid>.Write(ctx, writer, value._Reload);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynProjectToLoad (");
      using (printer.IndentCookie()) {
        printer.Print("path = "); Path.PrintEx(printer); printer.Println();
        printer.Print("guid = "); Guid.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
        printer.Print("loadMessages = "); _LoadMessages.PrintEx(printer); printer.Println();
        printer.Print("loadComplete = "); _LoadComplete.PrintEx(printer); printer.Println();
        printer.Print("selectedConfiguration = "); _SelectedConfiguration.PrintEx(printer); printer.Println();
        printer.Print("selectedPlatform = "); _SelectedPlatform.PrintEx(printer); printer.Println();
        printer.Print("reload = "); _Reload.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynQuickFixDescriptor : IPrintable, IEquatable<RoslynQuickFixDescriptor> {
    //fields
    //public fields
    [NotNull] public string Text {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynQuickFixDescriptor(
      [NotNull] string text
    )
    {
      if (text == null) throw new ArgumentNullException("text");
      
      Text = text;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynQuickFixDescriptor> Read = (ctx, reader) => 
    {
      var text = reader.ReadString();
      return new RoslynQuickFixDescriptor(text);
    };
    
    public static CtxWriteDelegate<RoslynQuickFixDescriptor> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Text);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynQuickFixDescriptor) obj);
    }
    public bool Equals(RoslynQuickFixDescriptor other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Text == other.Text;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Text.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynQuickFixDescriptor (");
      using (printer.IndentCookie()) {
        printer.Print("text = "); Text.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynRefactoring : RdBindableBase {
    //fields
    //public fields
    [NotNull] public string RefactoringClass {get; private set;}
    [NotNull] public string Text {get; private set;}
    [NotNull] public List<string> Languages {get; private set;}
    [NotNull] public string Assembly {get; private set;}
    [NotNull] public IViewableList<string> Projects { get { return _Projects; }}
    [NotNull] public IRdProperty<bool> Enabled { get { return _Enabled; }}
    
    //private fields
    [NotNull] private readonly RdList<string> _Projects;
    [NotNull] private readonly RdProperty<bool> _Enabled;
    
    //primary constructor
    private RoslynRefactoring(
      [NotNull] string refactoringClass,
      [NotNull] string text,
      [NotNull] List<string> languages,
      [NotNull] string assembly,
      [NotNull] RdList<string> projects,
      [NotNull] RdProperty<bool> enabled
    )
    {
      if (refactoringClass == null) throw new ArgumentNullException("refactoringClass");
      if (text == null) throw new ArgumentNullException("text");
      if (languages == null) throw new ArgumentNullException("languages");
      if (assembly == null) throw new ArgumentNullException("assembly");
      if (projects == null) throw new ArgumentNullException("projects");
      if (enabled == null) throw new ArgumentNullException("enabled");
      
      RefactoringClass = refactoringClass;
      Text = text;
      Languages = languages;
      Assembly = assembly;
      _Projects = projects;
      _Enabled = enabled;
      _Projects.OptimizeNested = true;
      _Enabled.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("projects", _Projects));
      BindableChildren.Add(new KeyValuePair<string, object>("enabled", _Enabled));
    }
    //secondary constructor
    public RoslynRefactoring (
      [NotNull] string refactoringClass,
      [NotNull] string text,
      [NotNull] List<string> languages,
      [NotNull] string assembly
    ) : this (
      refactoringClass,
      text,
      languages,
      assembly,
      new RdList<string>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool)
    ) {}
    //statics
    
    public static CtxReadDelegate<RoslynRefactoring> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var refactoringClass = reader.ReadString();
      var text = reader.ReadString();
      var languages = ReadStringList(ctx, reader);
      var assembly = reader.ReadString();
      var projects = RdList<string>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString);
      var enabled = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      return new RoslynRefactoring(refactoringClass, text, languages, assembly, projects, enabled).WithId(_id);
    };
    public static CtxReadDelegate<List<string>> ReadStringList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadString.List();
    
    public static CtxWriteDelegate<RoslynRefactoring> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.RefactoringClass);
      writer.Write(value.Text);
      WriteStringList(ctx, writer, value.Languages);
      writer.Write(value.Assembly);
      RdList<string>.Write(ctx, writer, value._Projects);
      RdProperty<bool>.Write(ctx, writer, value._Enabled);
    };
    public static CtxWriteDelegate<List<string>> WriteStringList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteString.List();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynRefactoring (");
      using (printer.IndentCookie()) {
        printer.Print("refactoringClass = "); RefactoringClass.PrintEx(printer); printer.Println();
        printer.Print("text = "); Text.PrintEx(printer); printer.Println();
        printer.Print("languages = "); Languages.PrintEx(printer); printer.Println();
        printer.Print("assembly = "); Assembly.PrintEx(printer); printer.Println();
        printer.Print("projects = "); _Projects.PrintEx(printer); printer.Println();
        printer.Print("enabled = "); _Enabled.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class RoslynRunningTask : IPrintable, IEquatable<RoslynRunningTask> {
    //fields
    //public fields
    [NotNull] public string TaskName {get; private set;}
    
    //private fields
    //primary constructor
    public RoslynRunningTask(
      [NotNull] string taskName
    )
    {
      if (taskName == null) throw new ArgumentNullException("taskName");
      
      TaskName = taskName;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<RoslynRunningTask> Read = (ctx, reader) => 
    {
      var taskName = reader.ReadString();
      return new RoslynRunningTask(taskName);
    };
    
    public static CtxWriteDelegate<RoslynRunningTask> Write = (ctx, writer, value) => 
    {
      writer.Write(value.TaskName);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((RoslynRunningTask) obj);
    }
    public bool Equals(RoslynRunningTask other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return TaskName == other.TaskName;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + TaskName.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("RoslynRunningTask (");
      using (printer.IndentCookie()) {
        printer.Print("taskName = "); TaskName.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public enum RoslynSeverity {
    Error,
    Warning,
    Info,
    Hidden
  }
}
