using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model
{
  
  
  public class CodeLensSettingsModel : RdExtBase
  {
    //fields
    //public fields
    
    /// <summary>
    /// Code Lens providers available on this backend. Key is the unique ID of the provider - settings are tied to it
    /// </summary>
    [NotNull] public IViewableMap<string, CodeLensProviderModel> Providers { get { return _Providers; }}
    
    /// <summary>
    /// Show SWEA-dependent lenses if SWEA if disabled
    /// </summary>
    [NotNull] public IRdProperty<bool> ShowUnavailableLenses { get { return _ShowUnavailableLenses; }}
    
    //private fields
    [NotNull] private readonly RdMap<string, CodeLensProviderModel> _Providers;
    [NotNull] private readonly RdProperty<bool> _ShowUnavailableLenses;
    
    //primary constructor
    private CodeLensSettingsModel(
      [NotNull] RdMap<string, CodeLensProviderModel> providers,
      [NotNull] RdProperty<bool> showUnavailableLenses
    )
    {
      if (providers == null) throw new ArgumentNullException("providers");
      if (showUnavailableLenses == null) throw new ArgumentNullException("showUnavailableLenses");
      
      _Providers = providers;
      _ShowUnavailableLenses = showUnavailableLenses;
      _ShowUnavailableLenses.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("providers", _Providers));
      BindableChildren.Add(new KeyValuePair<string, object>("showUnavailableLenses", _ShowUnavailableLenses));
    }
    //secondary constructor
    internal CodeLensSettingsModel (
    ) : this (
      new RdMap<string, CodeLensProviderModel>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadString, JetBrains.Platform.RdFramework.Impl.Serializers.WriteString, CodeLensProviderModel.Read, CodeLensProviderModel.Write),
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool)
    ) {}
    //statics
    
    
    
    protected override long SerializationHash => 3241054920507451127L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.RegisterEnum<CodeLensAnchorKind>();
      serializers.Register(CodeLensRelativeOrderingFirst.Read, CodeLensRelativeOrderingFirst.Write);
      serializers.Register(CodeLensRelativeOrderingLast.Read, CodeLensRelativeOrderingLast.Write);
      serializers.Register(CodeLensRelativeOrderingBefore.Read, CodeLensRelativeOrderingBefore.Write);
      serializers.Register(CodeLensRelativeOrderingAfter.Read, CodeLensRelativeOrderingAfter.Write);
      serializers.Register(CodeLensProviderModel.Read, CodeLensProviderModel.Write);
      serializers.Register(CodeLensProviderClickedArguments.Read, CodeLensProviderClickedArguments.Write);
      serializers.Register(CodeLensExtraActionClickedArguments.Read, CodeLensExtraActionClickedArguments.Write);
      serializers.Register(CodeLensPopupModel.Read, CodeLensPopupModel.Write);
      serializers.Register(CodeLensRelativeOrdering_Unknown.Read, CodeLensRelativeOrdering_Unknown.Write);
      
      serializers.RegisterToplevelOnce(typeof(IdeRoot), IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensSettingsModel (");
      using (printer.IndentCookie()) {
        printer.Print("providers = "); _Providers.PrintEx(printer); printer.Println();
        printer.Print("showUnavailableLenses = "); _ShowUnavailableLenses.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  public static class SolutionModelCodeLensSettingsModelEx
   {
    public static CodeLensSettingsModel GetCodeLensSettingsModel(this SolutionModel solutionModel)
    {
      return solutionModel.GetOrCreateExtension("codeLensSettingsModel", () => new CodeLensSettingsModel());
    }
  }
  
  
  public enum CodeLensAnchorKind {
    
    /// <summary>
    /// Above the target range
    /// </summary>
    Top,
    
    /// <summary>
    /// After end of line with target range
    /// </summary>
    Right,
    
    /// <summary>
    /// On the same line as target range, near the scrollbar
    /// </summary>
    NearScroll,
    
    /// <summary>
    /// In any empty space near the target range
    /// </summary>
    EmptySpace,
    
    /// <summary>
    /// Use the global default value from settings
    /// </summary>
    Default
  }
  
  
  public class CodeLensExtraActionClickedArguments : IPrintable, IEquatable<CodeLensExtraActionClickedArguments>
  {
    //fields
    //public fields
    
    /// <summary>
    /// The document in which an extra action was clicked
    /// </summary>
    [NotNull] public EditableEntityId EditableEntity {get; private set;}
    
    /// <summary>
    /// The range on which the clicked lens resides
    /// </summary>
    [NotNull] public RdTextRange Range {get; private set;}
    
    /// <summary>
    /// The action that was clicked
    /// </summary>
    [NotNull] public string ActionId {get; private set;}
    
    //private fields
    //primary constructor
    public CodeLensExtraActionClickedArguments(
      [NotNull] EditableEntityId editableEntity,
      [NotNull] RdTextRange range,
      [NotNull] string actionId
    )
    {
      if (editableEntity == null) throw new ArgumentNullException("editableEntity");
      if (range == null) throw new ArgumentNullException("range");
      if (actionId == null) throw new ArgumentNullException("actionId");
      
      EditableEntity = editableEntity;
      Range = range;
      ActionId = actionId;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeLensExtraActionClickedArguments> Read = (ctx, reader) => 
    {
      var editableEntity = EditableEntityId.Read(ctx, reader);
      var range = RdTextRange.Read(ctx, reader);
      var actionId = reader.ReadString();
      return new CodeLensExtraActionClickedArguments(editableEntity, range, actionId);
    };
    
    public static CtxWriteDelegate<CodeLensExtraActionClickedArguments> Write = (ctx, writer, value) => 
    {
      EditableEntityId.Write(ctx, writer, value.EditableEntity);
      RdTextRange.Write(ctx, writer, value.Range);
      writer.Write(value.ActionId);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensExtraActionClickedArguments) obj);
    }
    public bool Equals(CodeLensExtraActionClickedArguments other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(EditableEntity, other.EditableEntity) && Equals(Range, other.Range) && ActionId == other.ActionId;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + EditableEntity.GetHashCode();
        hash = hash * 31 + Range.GetHashCode();
        hash = hash * 31 + ActionId.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensExtraActionClickedArguments (");
      using (printer.IndentCookie()) {
        printer.Print("editableEntity = "); EditableEntity.PrintEx(printer); printer.Println();
        printer.Print("range = "); Range.PrintEx(printer); printer.Println();
        printer.Print("actionId = "); ActionId.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensPopupModel : RdBindableBase
  {
    //fields
    //public fields
    
    /// <summary>
    /// The control to show in a popup
    /// </summary>
    [NotNull] public JetBrains.Rider.Model.UIAutomation.BeControl Control {get; private set;}
    
    /// <summary>
    /// The document in which to show the popup
    /// </summary>
    [NotNull] public EditableEntityId EditableEntity {get; private set;}
    
    /// <summary>
    /// The range of the popup
    /// </summary>
    [NotNull] public RdTextRange Range {get; private set;}
    
    //private fields
    //primary constructor
    public CodeLensPopupModel(
      [NotNull] JetBrains.Rider.Model.UIAutomation.BeControl control,
      [NotNull] EditableEntityId editableEntity,
      [NotNull] RdTextRange range
    )
    {
      if (control == null) throw new ArgumentNullException("control");
      if (editableEntity == null) throw new ArgumentNullException("editableEntity");
      if (range == null) throw new ArgumentNullException("range");
      
      Control = control;
      EditableEntity = editableEntity;
      Range = range;
      BindableChildren.Add(new KeyValuePair<string, object>("control", Control));
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeLensPopupModel> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var control = JetBrains.Rider.Model.UIAutomation.BeControl.Read(ctx, reader);
      var editableEntity = EditableEntityId.Read(ctx, reader);
      var range = RdTextRange.Read(ctx, reader);
      return new CodeLensPopupModel(control, editableEntity, range).WithId(_id);
    };
    
    public static CtxWriteDelegate<CodeLensPopupModel> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      JetBrains.Rider.Model.UIAutomation.BeControl.Write(ctx, writer, value.Control);
      EditableEntityId.Write(ctx, writer, value.EditableEntity);
      RdTextRange.Write(ctx, writer, value.Range);
    };
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensPopupModel (");
      using (printer.IndentCookie()) {
        printer.Print("control = "); Control.PrintEx(printer); printer.Println();
        printer.Print("editableEntity = "); EditableEntity.PrintEx(printer); printer.Println();
        printer.Print("range = "); Range.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensProviderClickedArguments : IPrintable, IEquatable<CodeLensProviderClickedArguments>
  {
    //fields
    //public fields
    
    /// <summary>
    /// The document in which a lens is clicked
    /// </summary>
    [NotNull] public EditableEntityId EditableEntity {get; private set;}
    
    /// <summary>
    /// The range on which the clicked lens resides
    /// </summary>
    [NotNull] public RdTextRange Range {get; private set;}
    
    //private fields
    //primary constructor
    public CodeLensProviderClickedArguments(
      [NotNull] EditableEntityId editableEntity,
      [NotNull] RdTextRange range
    )
    {
      if (editableEntity == null) throw new ArgumentNullException("editableEntity");
      if (range == null) throw new ArgumentNullException("range");
      
      EditableEntity = editableEntity;
      Range = range;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeLensProviderClickedArguments> Read = (ctx, reader) => 
    {
      var editableEntity = EditableEntityId.Read(ctx, reader);
      var range = RdTextRange.Read(ctx, reader);
      return new CodeLensProviderClickedArguments(editableEntity, range);
    };
    
    public static CtxWriteDelegate<CodeLensProviderClickedArguments> Write = (ctx, writer, value) => 
    {
      EditableEntityId.Write(ctx, writer, value.EditableEntity);
      RdTextRange.Write(ctx, writer, value.Range);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensProviderClickedArguments) obj);
    }
    public bool Equals(CodeLensProviderClickedArguments other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(EditableEntity, other.EditableEntity) && Equals(Range, other.Range);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + EditableEntity.GetHashCode();
        hash = hash * 31 + Range.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensProviderClickedArguments (");
      using (printer.IndentCookie()) {
        printer.Print("editableEntity = "); EditableEntity.PrintEx(printer); printer.Println();
        printer.Print("range = "); Range.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensProviderModel : RdBindableBase
  {
    //fields
    //public fields
    
    /// <summary>
    /// User-visible provider name
    /// </summary>
    [NotNull] public string DisplayName {get; private set;}
    
    /// <summary>
    /// Ordering constraints; these will be used when initially composing lens list of the user
    /// </summary>
    [NotNull] public List<CodeLensRelativeOrdering> OrderingConstraints {get; private set;}
    
    /// <summary>
    /// Default anchor
    /// </summary>
    public CodeLensAnchorKind Anchor {get; private set;}
    
    /// <summary>
    /// Whether or not this provider is enabled by the user. Will be false if Code Lens is disabled
    /// </summary>
    [NotNull] public IRdProperty<bool> IsEnabled { get { return _IsEnabled; }}
    
    /// <summary>
    /// Fired when this entry is clicked in editor or 'More' menu
    /// </summary>
    [NotNull] public IRdSignal<CodeLensProviderClickedArguments> Clicked { get { return _Clicked; }}
    
    /// <summary>
    /// Fired when an extra action is clicked on a lens
    /// </summary>
    [NotNull] public IRdSignal<CodeLensExtraActionClickedArguments> ExtraActionClicked { get { return _ExtraActionClicked; }}
    
    /// <summary>
    /// Shows a popup when set
    /// </summary>
    [NotNull] public IRdProperty<CodeLensPopupModel> Popup { get { return _Popup; }}
    
    //private fields
    [NotNull] private readonly RdProperty<bool> _IsEnabled;
    [NotNull] private readonly RdSignal<CodeLensProviderClickedArguments> _Clicked;
    [NotNull] private readonly RdSignal<CodeLensExtraActionClickedArguments> _ExtraActionClicked;
    [NotNull] private readonly RdProperty<CodeLensPopupModel> _Popup;
    
    //primary constructor
    private CodeLensProviderModel(
      [NotNull] string displayName,
      [NotNull] List<CodeLensRelativeOrdering> orderingConstraints,
      CodeLensAnchorKind anchor,
      [NotNull] RdProperty<bool> isEnabled,
      [NotNull] RdSignal<CodeLensProviderClickedArguments> clicked,
      [NotNull] RdSignal<CodeLensExtraActionClickedArguments> extraActionClicked,
      [NotNull] RdProperty<CodeLensPopupModel> popup
    )
    {
      if (displayName == null) throw new ArgumentNullException("displayName");
      if (orderingConstraints == null) throw new ArgumentNullException("orderingConstraints");
      if (isEnabled == null) throw new ArgumentNullException("isEnabled");
      if (clicked == null) throw new ArgumentNullException("clicked");
      if (extraActionClicked == null) throw new ArgumentNullException("extraActionClicked");
      if (popup == null) throw new ArgumentNullException("popup");
      
      DisplayName = displayName;
      OrderingConstraints = orderingConstraints;
      Anchor = anchor;
      _IsEnabled = isEnabled;
      _Clicked = clicked;
      _ExtraActionClicked = extraActionClicked;
      _Popup = popup;
      _IsEnabled.OptimizeNested = true;
      BindableChildren.Add(new KeyValuePair<string, object>("isEnabled", _IsEnabled));
      BindableChildren.Add(new KeyValuePair<string, object>("clicked", _Clicked));
      BindableChildren.Add(new KeyValuePair<string, object>("extraActionClicked", _ExtraActionClicked));
      BindableChildren.Add(new KeyValuePair<string, object>("popup", _Popup));
    }
    //secondary constructor
    public CodeLensProviderModel (
      [NotNull] string displayName,
      [NotNull] List<CodeLensRelativeOrdering> orderingConstraints,
      CodeLensAnchorKind anchor
    ) : this (
      displayName,
      orderingConstraints,
      anchor,
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdSignal<CodeLensProviderClickedArguments>(CodeLensProviderClickedArguments.Read, CodeLensProviderClickedArguments.Write),
      new RdSignal<CodeLensExtraActionClickedArguments>(CodeLensExtraActionClickedArguments.Read, CodeLensExtraActionClickedArguments.Write),
      new RdProperty<CodeLensPopupModel>(CodeLensPopupModel.Read, CodeLensPopupModel.Write)
    ) {}
    //statics
    
    public static CtxReadDelegate<CodeLensProviderModel> Read = (ctx, reader) => 
    {
      var _id = RdId.Read(reader);
      var displayName = reader.ReadString();
      var orderingConstraints = ReadCodeLensRelativeOrderingList(ctx, reader);
      var anchor = (CodeLensAnchorKind)reader.ReadInt();
      var isEnabled = RdProperty<bool>.Read(ctx, reader, JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool);
      var clicked = RdSignal<CodeLensProviderClickedArguments>.Read(ctx, reader, CodeLensProviderClickedArguments.Read, CodeLensProviderClickedArguments.Write);
      var extraActionClicked = RdSignal<CodeLensExtraActionClickedArguments>.Read(ctx, reader, CodeLensExtraActionClickedArguments.Read, CodeLensExtraActionClickedArguments.Write);
      var popup = RdProperty<CodeLensPopupModel>.Read(ctx, reader, CodeLensPopupModel.Read, CodeLensPopupModel.Write);
      return new CodeLensProviderModel(displayName, orderingConstraints, anchor, isEnabled, clicked, extraActionClicked, popup).WithId(_id);
    };
    public static CtxReadDelegate<List<CodeLensRelativeOrdering>> ReadCodeLensRelativeOrderingList = CodeLensRelativeOrdering.Read.List();
    
    public static CtxWriteDelegate<CodeLensProviderModel> Write = (ctx, writer, value) => 
    {
      value.RdId.Write(writer);
      writer.Write(value.DisplayName);
      WriteCodeLensRelativeOrderingList(ctx, writer, value.OrderingConstraints);
      writer.Write((int)value.Anchor);
      RdProperty<bool>.Write(ctx, writer, value._IsEnabled);
      RdSignal<CodeLensProviderClickedArguments>.Write(ctx, writer, value._Clicked);
      RdSignal<CodeLensExtraActionClickedArguments>.Write(ctx, writer, value._ExtraActionClicked);
      RdProperty<CodeLensPopupModel>.Write(ctx, writer, value._Popup);
    };
    public static CtxWriteDelegate<List<CodeLensRelativeOrdering>> WriteCodeLensRelativeOrderingList = CodeLensRelativeOrdering.Write.List();
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensProviderModel (");
      using (printer.IndentCookie()) {
        printer.Print("displayName = "); DisplayName.PrintEx(printer); printer.Println();
        printer.Print("orderingConstraints = "); OrderingConstraints.PrintEx(printer); printer.Println();
        printer.Print("anchor = "); Anchor.PrintEx(printer); printer.Println();
        printer.Print("isEnabled = "); _IsEnabled.PrintEx(printer); printer.Println();
        printer.Print("clicked = "); _Clicked.PrintEx(printer); printer.Println();
        printer.Print("extraActionClicked = "); _ExtraActionClicked.PrintEx(printer); printer.Println();
        printer.Print("popup = "); _Popup.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public abstract class CodeLensRelativeOrdering{
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeLensRelativeOrdering> Read = Polymorphic<CodeLensRelativeOrdering>.ReadAbstract(CodeLensRelativeOrdering_Unknown.Read);
    
    public static CtxWriteDelegate<CodeLensRelativeOrdering> Write = Polymorphic<CodeLensRelativeOrdering>.Write;
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    //toString
  }
  
  
  public class CodeLensRelativeOrderingAfter : CodeLensRelativeOrdering
  {
    //fields
    //public fields
    [NotNull] public string Id {get; private set;}
    
    //private fields
    //primary constructor
    public CodeLensRelativeOrderingAfter(
      [NotNull] string id
    )
    {
      if (id == null) throw new ArgumentNullException("id");
      
      Id = id;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<CodeLensRelativeOrderingAfter> Read = (ctx, reader) => 
    {
      var id = reader.ReadString();
      return new CodeLensRelativeOrderingAfter(id);
    };
    
    public static new CtxWriteDelegate<CodeLensRelativeOrderingAfter> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensRelativeOrderingAfter) obj);
    }
    public bool Equals(CodeLensRelativeOrderingAfter other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensRelativeOrderingAfter (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensRelativeOrderingBefore : CodeLensRelativeOrdering
  {
    //fields
    //public fields
    [NotNull] public string Id {get; private set;}
    
    //private fields
    //primary constructor
    public CodeLensRelativeOrderingBefore(
      [NotNull] string id
    )
    {
      if (id == null) throw new ArgumentNullException("id");
      
      Id = id;
    }
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<CodeLensRelativeOrderingBefore> Read = (ctx, reader) => 
    {
      var id = reader.ReadString();
      return new CodeLensRelativeOrderingBefore(id);
    };
    
    public static new CtxWriteDelegate<CodeLensRelativeOrderingBefore> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensRelativeOrderingBefore) obj);
    }
    public bool Equals(CodeLensRelativeOrderingBefore other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensRelativeOrderingBefore (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensRelativeOrderingFirst : CodeLensRelativeOrdering
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<CodeLensRelativeOrderingFirst> Read = (ctx, reader) => 
    {
      return new CodeLensRelativeOrderingFirst();
    };
    
    public static new CtxWriteDelegate<CodeLensRelativeOrderingFirst> Write = (ctx, writer, value) => 
    {
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensRelativeOrderingFirst) obj);
    }
    public bool Equals(CodeLensRelativeOrderingFirst other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return true;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensRelativeOrderingFirst (");
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensRelativeOrderingLast : CodeLensRelativeOrdering
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<CodeLensRelativeOrderingLast> Read = (ctx, reader) => 
    {
      return new CodeLensRelativeOrderingLast();
    };
    
    public static new CtxWriteDelegate<CodeLensRelativeOrderingLast> Write = (ctx, writer, value) => 
    {
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensRelativeOrderingLast) obj);
    }
    public bool Equals(CodeLensRelativeOrderingLast other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return true;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensRelativeOrderingLast (");
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeLensRelativeOrdering_Unknown : CodeLensRelativeOrdering
  {
    //fields
    //public fields
    
    //private fields
    //primary constructor
    //secondary constructor
    //statics
    
    public static new CtxReadDelegate<CodeLensRelativeOrdering_Unknown> Read = (ctx, reader) => 
    {
      return new CodeLensRelativeOrdering_Unknown();
    };
    
    public static new CtxWriteDelegate<CodeLensRelativeOrdering_Unknown> Write = (ctx, writer, value) => 
    {
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeLensRelativeOrdering_Unknown) obj);
    }
    public bool Equals(CodeLensRelativeOrdering_Unknown other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return true;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeLensRelativeOrdering_Unknown (");
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
