using System;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using JetBrains.Annotations;

using JetBrains.Platform.RdFramework;
using JetBrains.Platform.RdFramework.Base;
using JetBrains.Platform.RdFramework.Impl;
using JetBrains.Platform.RdFramework.Tasks;
using JetBrains.Platform.RdFramework.Util;
using JetBrains.Platform.RdFramework.Text;

using JetBrains.Util.Collections;
using JetBrains.Util.Logging;
using JetBrains.Util.PersistentMap;
using Lifetime = JetBrains.DataFlow.Lifetime;

// ReSharper disable RedundantEmptyObjectCreationArgumentList
// ReSharper disable InconsistentNaming
// ReSharper disable RedundantOverflowCheckingContext


namespace JetBrains.Rider.Model
{
  
  
  public class CodeStructureTree : RdExtBase
  {
    //fields
    //public fields
    [NotNull] public IRdProperty<bool> ToolWindowVisible { get { return _ToolWindowVisible; }}
    [NotNull] public IRdProperty<CodeStructureTreeNodesList> TreeNodes { get { return _TreeNodes; }}
    
    /// <summary>
    /// Receives the ID of a node and navigates to it in the editor, transferring focus to it if necessary
    /// </summary>
    [NotNull] public ISink<CodeStructureNavigateTo> NavigateTo { get { return _NavigateTo; }}
    
    //private fields
    [NotNull] private readonly RdProperty<bool> _ToolWindowVisible;
    [NotNull] private readonly RdProperty<CodeStructureTreeNodesList> _TreeNodes;
    [NotNull] private readonly RdSignal<CodeStructureNavigateTo> _NavigateTo;
    
    //primary constructor
    private CodeStructureTree(
      [NotNull] RdProperty<bool> toolWindowVisible,
      [NotNull] RdProperty<CodeStructureTreeNodesList> treeNodes,
      [NotNull] RdSignal<CodeStructureNavigateTo> navigateTo
    )
    {
      if (toolWindowVisible == null) throw new ArgumentNullException("toolWindowVisible");
      if (treeNodes == null) throw new ArgumentNullException("treeNodes");
      if (navigateTo == null) throw new ArgumentNullException("navigateTo");
      
      _ToolWindowVisible = toolWindowVisible;
      _TreeNodes = treeNodes;
      _NavigateTo = navigateTo;
      _ToolWindowVisible.OptimizeNested = true;
      _TreeNodes.OptimizeNested = true;
      _TreeNodes.Async = true;
      BindableChildren.Add(new KeyValuePair<string, object>("toolWindowVisible", _ToolWindowVisible));
      BindableChildren.Add(new KeyValuePair<string, object>("treeNodes", _TreeNodes));
      BindableChildren.Add(new KeyValuePair<string, object>("navigateTo", _NavigateTo));
    }
    //secondary constructor
    internal CodeStructureTree (
    ) : this (
      new RdProperty<bool>(JetBrains.Platform.RdFramework.Impl.Serializers.ReadBool, JetBrains.Platform.RdFramework.Impl.Serializers.WriteBool),
      new RdProperty<CodeStructureTreeNodesList>(CodeStructureTreeNodesList.Read, CodeStructureTreeNodesList.Write),
      new RdSignal<CodeStructureNavigateTo>(CodeStructureNavigateTo.Read, CodeStructureNavigateTo.Write)
    ) {}
    //statics
    
    
    
    protected override long SerializationHash => -2750105384626411263L;
    
    protected override Action<ISerializers> Register => RegisterDeclaredTypesSerializers;
    public static void RegisterDeclaredTypesSerializers(ISerializers serializers)
    {
      serializers.Register(CodeStructureTreeNodesList.Read, CodeStructureTreeNodesList.Write);
      serializers.Register(CodeStructureNavigateTo.Read, CodeStructureNavigateTo.Write);
      serializers.Register(CodeStructureTreeNode.Read, CodeStructureTreeNode.Write);
      
      serializers.RegisterToplevelOnce(typeof(IdeRoot), IdeRoot.RegisterDeclaredTypesSerializers);
    }
    
    //custom body
    //equals trait
    //hash code trait
    //pretty print
    public override void Print(PrettyPrinter printer)
    {
      printer.Println("CodeStructureTree (");
      using (printer.IndentCookie()) {
        printer.Print("toolWindowVisible = "); _ToolWindowVisible.PrintEx(printer); printer.Println();
        printer.Print("treeNodes = "); _TreeNodes.PrintEx(printer); printer.Println();
        printer.Print("navigateTo = "); _NavigateTo.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  public static class SolutionCodeStructureTreeEx
   {
    public static CodeStructureTree GetCodeStructureTree(this Solution solution)
    {
      return solution.GetOrCreateExtension("codeStructureTree", () => new CodeStructureTree());
    }
  }
  
  
  public class CodeStructureNavigateTo : IPrintable, IEquatable<CodeStructureNavigateTo>
  {
    //fields
    //public fields
    
    /// <summary>
    /// The ID (CodeStructureTreeNode.id) of a node
    /// </summary>
    public int Node {get; private set;}
    
    /// <summary>
    /// If true, transfer focus to the editor
    /// </summary>
    public bool RequestFocus {get; private set;}
    
    //private fields
    //primary constructor
    public CodeStructureNavigateTo(
      int node,
      bool requestFocus
    )
    {
      Node = node;
      RequestFocus = requestFocus;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeStructureNavigateTo> Read = (ctx, reader) => 
    {
      var node = reader.ReadInt();
      var requestFocus = reader.ReadBool();
      return new CodeStructureNavigateTo(node, requestFocus);
    };
    
    public static CtxWriteDelegate<CodeStructureNavigateTo> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Node);
      writer.Write(value.RequestFocus);
    };
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeStructureNavigateTo) obj);
    }
    public bool Equals(CodeStructureNavigateTo other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Node == other.Node && RequestFocus == other.RequestFocus;
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Node.GetHashCode();
        hash = hash * 31 + RequestFocus.GetHashCode();
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeStructureNavigateTo (");
      using (printer.IndentCookie()) {
        printer.Print("node = "); Node.PrintEx(printer); printer.Println();
        printer.Print("requestFocus = "); RequestFocus.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeStructureTreeNode : IPrintable, IEquatable<CodeStructureTreeNode>
  {
    //fields
    //public fields
    
    /// <summary>
    /// ID of the element. Used in navigateTo() and in RdFileStructureConstant.
    /// </summary>
    public int Id {get; private set;}
    
    /// <summary>
    /// List of indices in treeNodes array of the children of this node.
    /// </summary>
    [NotNull] public List<int> Children {get; private set;}
    
    /// <summary>
    /// Whether the Navigate and Navigate to Source actions are enabled for this node.
    /// </summary>
    public bool Navigable {get; private set;}
    
    /// <summary>
    /// Start offset of the element represented by this node.
    /// </summary>
    public int Offset {get; private set;}
    
    /// <summary>
    /// End offset of the element represented by this node.
    /// </summary>
    public int EndOffset {get; private set;}
    
    /// <summary>
    /// Text to show in the structure view tree for this node.
    /// </summary>
    [NotNull] public string Name {get; private set;}
    
    /// <summary>
    /// Icon to show in the structure view tree for this node.
    /// </summary>
    [CanBeNull] public JetBrains.Rider.Model.Icon Icon {get; private set;}
    
    //private fields
    //primary constructor
    public CodeStructureTreeNode(
      int id,
      [NotNull] List<int> children,
      bool navigable,
      int offset,
      int endOffset,
      [NotNull] string name,
      [CanBeNull] JetBrains.Rider.Model.Icon icon
    )
    {
      if (children == null) throw new ArgumentNullException("children");
      if (name == null) throw new ArgumentNullException("name");
      
      Id = id;
      Children = children;
      Navigable = navigable;
      Offset = offset;
      EndOffset = endOffset;
      Name = name;
      Icon = icon;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeStructureTreeNode> Read = (ctx, reader) => 
    {
      var id = reader.ReadInt();
      var children = ReadIntList(ctx, reader);
      var navigable = reader.ReadBool();
      var offset = reader.ReadInt();
      var endOffset = reader.ReadInt();
      var name = reader.ReadString();
      var icon = ReadIconNullable(ctx, reader);
      return new CodeStructureTreeNode(id, children, navigable, offset, endOffset, name, icon);
    };
    public static CtxReadDelegate<List<int>> ReadIntList = JetBrains.Platform.RdFramework.Impl.Serializers.ReadInt.List();
    public static CtxReadDelegate<JetBrains.Rider.Model.Icon> ReadIconNullable = JetBrains.Rider.Model.Icon.Read.NullableClass();
    
    public static CtxWriteDelegate<CodeStructureTreeNode> Write = (ctx, writer, value) => 
    {
      writer.Write(value.Id);
      WriteIntList(ctx, writer, value.Children);
      writer.Write(value.Navigable);
      writer.Write(value.Offset);
      writer.Write(value.EndOffset);
      writer.Write(value.Name);
      WriteIconNullable(ctx, writer, value.Icon);
    };
    public static CtxWriteDelegate<List<int>> WriteIntList = JetBrains.Platform.RdFramework.Impl.Serializers.WriteInt.List();
    public static CtxWriteDelegate<JetBrains.Rider.Model.Icon> WriteIconNullable = JetBrains.Rider.Model.Icon.Write.NullableClass();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeStructureTreeNode) obj);
    }
    public bool Equals(CodeStructureTreeNode other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Id == other.Id && Children.SequenceEqual(other.Children) && Navigable == other.Navigable && Offset == other.Offset && EndOffset == other.EndOffset && Name == other.Name && Equals(Icon, other.Icon);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + Id.GetHashCode();
        hash = hash * 31 + Collections.GetHashCode(Children);
        hash = hash * 31 + Navigable.GetHashCode();
        hash = hash * 31 + Offset.GetHashCode();
        hash = hash * 31 + EndOffset.GetHashCode();
        hash = hash * 31 + Name.GetHashCode();
        hash = hash * 31 + (Icon != null ?Icon.GetHashCode() : 0);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeStructureTreeNode (");
      using (printer.IndentCookie()) {
        printer.Print("id = "); Id.PrintEx(printer); printer.Println();
        printer.Print("children = "); Children.PrintEx(printer); printer.Println();
        printer.Print("navigable = "); Navigable.PrintEx(printer); printer.Println();
        printer.Print("offset = "); Offset.PrintEx(printer); printer.Println();
        printer.Print("endOffset = "); EndOffset.PrintEx(printer); printer.Println();
        printer.Print("name = "); Name.PrintEx(printer); printer.Println();
        printer.Print("icon = "); Icon.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
  
  
  public class CodeStructureTreeNodesList : IPrintable, IEquatable<CodeStructureTreeNodesList>
  {
    //fields
    //public fields
    [CanBeNull] public EditableEntityId ForFile {get; private set;}
    [NotNull] public List<CodeStructureTreeNode> TreeNodes {get; private set;}
    
    //private fields
    //primary constructor
    public CodeStructureTreeNodesList(
      [CanBeNull] EditableEntityId forFile,
      [NotNull] List<CodeStructureTreeNode> treeNodes
    )
    {
      if (treeNodes == null) throw new ArgumentNullException("treeNodes");
      
      ForFile = forFile;
      TreeNodes = treeNodes;
    }
    //secondary constructor
    //statics
    
    public static CtxReadDelegate<CodeStructureTreeNodesList> Read = (ctx, reader) => 
    {
      var forFile = ReadEditableEntityIdNullable(ctx, reader);
      var treeNodes = ReadCodeStructureTreeNodeList(ctx, reader);
      return new CodeStructureTreeNodesList(forFile, treeNodes);
    };
    public static CtxReadDelegate<EditableEntityId> ReadEditableEntityIdNullable = EditableEntityId.Read.NullableClass();
    public static CtxReadDelegate<List<CodeStructureTreeNode>> ReadCodeStructureTreeNodeList = CodeStructureTreeNode.Read.List();
    
    public static CtxWriteDelegate<CodeStructureTreeNodesList> Write = (ctx, writer, value) => 
    {
      WriteEditableEntityIdNullable(ctx, writer, value.ForFile);
      WriteCodeStructureTreeNodeList(ctx, writer, value.TreeNodes);
    };
    public static CtxWriteDelegate<EditableEntityId> WriteEditableEntityIdNullable = EditableEntityId.Write.NullableClass();
    public static CtxWriteDelegate<List<CodeStructureTreeNode>> WriteCodeStructureTreeNodeList = CodeStructureTreeNode.Write.List();
    //custom body
    //equals trait
    public override bool Equals(object obj)
    {
      if (ReferenceEquals(null, obj)) return false;
      if (ReferenceEquals(this, obj)) return true;
      if (obj.GetType() != GetType()) return false;
      return Equals((CodeStructureTreeNodesList) obj);
    }
    public bool Equals(CodeStructureTreeNodesList other)
    {
      if (ReferenceEquals(null, other)) return false;
      if (ReferenceEquals(this, other)) return true;
      return Equals(ForFile, other.ForFile) && TreeNodes.SequenceEqual(other.TreeNodes);
    }
    //hash code trait
    public override int GetHashCode()
    {
      unchecked {
        var hash = 0;
        hash = hash * 31 + (ForFile != null ?ForFile.GetHashCode() : 0);
        hash = hash * 31 + Collections.GetHashCode(TreeNodes);
        return hash;
      }
    }
    //pretty print
    public void Print(PrettyPrinter printer)
    {
      printer.Println("CodeStructureTreeNodesList (");
      using (printer.IndentCookie()) {
        printer.Print("forFile = "); ForFile.PrintEx(printer); printer.Println();
        printer.Print("treeNodes = "); TreeNodes.PrintEx(printer); printer.Println();
      }
      printer.Print(")");
    }
    //toString
    public override string ToString()
    {
      var printer = new SingleLinePrettyPrinter();
      Print(printer);
      return printer.ToString();
    }
  }
}
